/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer;

/**
 * ノードの情報を表現するインターフェースを定義します。
 */
public interface Node {

    /*
     * ノードの型
     */

    /** 子の要素を持つコンテナ型。 */
    int TYPE_CONTAINER = 1;

    /** テキストノード。 */
    int TYPE_TEXT = 2;

    /** データ型。 */
    int TYPE_VALUE = 3;

    /** 処理を実行する型。 */
    int TYPE_COMMAND = 4;

    /** コメント型。 */
    int TYPE_COMMENT = 5;

    /*
     * ID識別子の可視領域
     */

    /** ID識別子のスコープ：公開。 */
    int SCOPE_PUBLIC = 100;

    /** ID識別子のスコープ：インスタンスの所有者。 */
    int SCOPE_OWNER = 101;

    /** ID識別子のスコープ：ドキュメント内のみ。 */
    int SCOPE_PRIVATE = 102;

    /*
     * ノードに関する情報
     */

    /**
     * 指定されたノードの種別に属する場合Trueを返却します。
     * 
     * @param type
     *            検証する種別
     * @return 指定されたノードの種別に属する場合True
     */
    boolean isType(int type);

    /**
     * ノードのスコープを返却します。
     * 
     * @return ノードのスコープ
     */
    int getScope();

    /*
     * ノードのデータ
     */

    /**
     * ノードの値表現を返却します。 <br>
     * isType(TYPE_VALUE) が True の場合必ず、有効な値を返却する必要がある。 <br>
     * isType(TYPE_VALUE) が False を返却する場合の実装は任意とします。
     * 
     * @return ノードの値表現
     */
    Object getNodeValue();

    /**
     * ノードのコンテキスト情報を返却します。
     * 
     * @return ノードのコンテキスト情報
     */
    NodeContext getNodeContext();

    /**
     * 属性を検索し存在しない場合は拡張属性を検索し値を返却します。 <br>
     * 属性名が存在しない場合 <code>null</code> を返す。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    String getAttribute(String name);

    /**
     * 属性を検索し存在しない場合は拡張属性を検索し値を返却します。 <br>
     * 属性名が存在しない場合 <code>defaultValue</code> を返す。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    String getAttribute(String name, String defaultValue);

    /**
     * テキスト情報を評価して返却します。 <br>
     * テキスト情報を持たないノードは空文字を返却します。 <br>
     * テキスト情報を評価する基準はノードの実装により異なることが認められています。 前後のスペースの評価や子のエレメントの値を含めるか否か、またテキストを変換して返却する場合も考えられます。
     * 
     * @return テキスト情報
     */
    String getText();

    /*
     * イベント通知
     */

    /**
     * ドキュメント解析処理を開始する直前に初期化の通知を受ける。
     * 
     * @param nodeContext
     *            ノードのコンテキスト情報
     * @throws CompositeException
     *             構成情報例外。
     */
    void initialize(NodeContext nodeContext);

    /**
     * 解析処理プロセス開始の通知を受ける。 開始タグの解析に対応します。
     * 
     * @throws CompositeException
     *             構成情報例外。
     */
    void begin();

    /**
     * 子のノードの追加の通知を受ける。 子ノードの解析終了直後に対応します。
     * 
     * @param node
     *            子のノード
     * @throws CompositeException
     *             構成情報例外。
     */
    void child(Node node);

    /**
     * 解析処理プロセス完了通知を受ける。 終了タグの解析に対応します。 <br>
     * この通知が完了した段階で、値を持つノードは値の生成が可能である必要が有ります。
     * 
     * @throws CompositeException
     *             構成情報例外。
     */
    void end();

    /**
     * ドキュメント解析処理プロセスを完了の通知を受ける。
     * 
     * @throws CompositeException
     *             構成情報例外。
     */
    void finish();

}