/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * 解析処理のコンテキスト情報を提供します。
 */
public class CompositeContext {

    /*
     * instance fields
     */

    /* Composer. */
    private final Composer composer;

    /* IDを持つノードを保管します。 */
    private HashMap idsMap = new HashMap();

    /* 属性情報を保管します。 */
    private HashMap attributes = new HashMap();

    CompositeContext(Composer composer) {
        this.composer = composer;
    }

    /*
     * Attribute
     */

    /**
     * 属性の名前のセットを全て返却します。
     * 
     * @return 属性の名前のセット
     */
    public Iterator getAttributeNameIterator() {
        return this.attributes.keySet().iterator();
    }

    /**
     * 引数の名前を持つ属性値を返却します。 <br>
     * 指定された属性が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object getAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.get(name);
    }

    /**
     * 引数の名前を持つ属性が存在する場合その属性値を存在しない場合は第二引数を返却します。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    public Object getAttribute(String name, Object defaultValue) {
        Object value = getAttribute(name);
        if (value == null) {
            return defaultValue;
        }
        return value;
    }

    /**
     * 接頭辞の一致する属性をMapに格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性のMap
     */
    public Map getAttributePrefix(String prefix) {
        if (prefix == null) {
            throw new NullPointerException();
        }
        HashMap m = new HashMap();
        Set entrys = this.attributes.entrySet();
        for (Iterator i = entrys.iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            String name = (String) e.getKey();
            if (name.startsWith(prefix)) {
                m.put(name, e.getValue());
            }
        }
        return m;
    }

    /**
     * 属性を登録する、既存の属性が存在する場合は既存の属性値を返却し上書きします。 既存の属性が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param name
     *            属性名
     * @param value
     *            属性値
     * @return 既存の属性値、存在しない場合は null
     */
    public Object setAttribute(String name, Object value) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.put(name, value);
    }

    /**
     * 属性を削除し削除された属性値を返却します。 属性が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object removeAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.remove(name);
    }

    /**
     * 接頭辞の一致する属性を削除し削除された属性をMapに格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性のMap
     */
    public Map removeAttributePrefix(String prefix) {
        if (prefix == null) {
            throw new NullPointerException();
        }
        HashMap m = new HashMap();
        Object[] keys = this.attributes.keySet().toArray();
        for (int i = 0; i < keys.length; i++) {
            String name = (String) keys[i];
            if (name.startsWith(prefix)) {
                m.put(name, this.attributes.remove(name));
            }
        }
        return m;
    }

    /**
     * 属性を全て削除します。
     */
    public void removeAttributeAll() {
        this.attributes.clear();
    }

    /*
     * ID
     */

    /**
     * 全てのパブリックノードのIDを返す。
     * 
     * @return 全てのパブリックノードのID
     */
    public Iterator getPublicNodeIdIterator() {
        return this.idsMap.keySet().iterator();
    }

    /**
     * IDが示すパブリックノードを返す。
     * 
     * @param publicId
     *            ID
     * @return IDが示すパブリックノード
     * @throws NullPointerException
     *             id が null の場合発生する
     */
    public Node getPublicNodeById(String publicId) {
        if (publicId == null) {
            throw new NullPointerException();
        }
        if (-1 == publicId.lastIndexOf(':')) {
            throw new CompositeException("It is not a public node. id:" + publicId + ".");
        }
        return (Node) idsMap.get(publicId);
    }

    /* IDを持つパブリックノードを追加します。 */
    private void addPublicReferenceNode(Node node) {
        if (node == null) {
            throw new NullPointerException();
        }
        if (Node.SCOPE_PUBLIC != node.getScope()) {
            throw new CompositeException("It isn't public node. node:" + node + ".");
        }
        String id = this.composer.getCompositeRule().getNodeId(node);
        if (null != id) {
            String publicId = composer.getDocumentContext().getPublicId();
            if (null != publicId && publicId.length() != 0) {
                id = publicId + ':' + id;
            } else {
                throw new CompositeException("A public identifier is necessary for the public node. node:" + node + ".");
            }
            Object old = idsMap.put(id, node);
            if (old != null) {
                //rollback
                idsMap.put(id, old);
                throw new CompositeException("That ID has already been registered. id:" + id + ".");
            }
        }
    }

    /*
     * Composer.parse
     */

    /**
     * パブリックノードの解析処理プロセスの開始の通知を受けます。
     * 
     * @param node
     *            パブリックノード
     */
    void startElement(Node node) {
        addPublicReferenceNode(node);
    }

    /**
     * パブリックノードの解析処理プロセスの終了の通知を受けます。
     * 
     * @param node
     *            パブリックノード
     */
    void endElement(Node node) {
        //no op
    }

    /**
     * 解析処理プロセスの開始の通知を受けます。
     */
    void begin() {
        // no op
    }

    /**
     * 解析処理プロセスの終了の通知を受けます。
     */
    void finish() {
        //no op
    }

    /*
     * Get And Set
     */

    /**
     * 解析処理に使用する ClassLoader を返します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return (null == getComposer()) ? null : getComposer().getClassLoader();
    }

    /**
     * Composer を返却します。
     * 
     * @return Composer。
     */
    public Composer getComposer() {
        return composer;
    }

}
