/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer;

import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.DTDHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

/**
 * <p>
 * XMLで定義された情報をオブジェクトにマッピングし生成されたオブジェクトを合成する機能を提供します。
 * </p>
 * <p>
 * このライブラリはXMLで定義された情報を <code>shohaku.composer.CompositeRule</code> の指定するルールに基づき <code>shohaku.composer.Node</code>
 * のツリー構造に変換します。 <br>
 * 値型の <code>shohaku.composer.Node</code> は自身と内包する要素に基づきオブジェクトを生成します。 <br>
 * 複数のノードを再起的に合成する事により、単純なデータ型の集合から複雑なデータ構造を生成します。 <br>
 * また一度定義されたノードは一定の依存関係を含むものの相対的に独立するため複数の文脈で再利用することができます。
 * </p>
 * <p>
 * <code>Composer</code> のデフォルト実装には基本型と利用頻度が高いと思われるデータ型、および何でも生成可能な汎用型のノードが提供されています。 <br>
 * また複雑な構造を容易に生成するために反復処理や条件分岐を実現する制御インターフェースが提供されています。 <br>
 * デフォルトノード以外の独自のノードの追加やデフォルトノードとの交換、使用するノードの選択、制約の追加も容易に実現できます。
 * </p>
 * <p>
 * <code>Composer</code> は汎用的なデータ構造を基に複雑なデータ構造を生成したい場合に有用に設計されています。 <br>
 * 解決を求められている問題領域によっては他のAPIの使用を検討してください。
 * </p>
 * <p>
 * 使用例として <code>shohaku.configutils</code> パッケージを参照する事が出来ます。
 * </p>
 *  
 */
public class Composer {

    /*
     * static fields
     */

    /* apache commons logging. */
    private static final Log staticLog = LogFactory.getLog(Composer.class);

    /*
     * instance fields
     */

    /* 標準のログ. */
    private Log log;

    /* org.xml.sax.SAXParser. */
    private SAXParser parser;

    /* org.xml.sax.XMLReader. */
    private XMLReader reader;

    /* javax.xml.parsers.SAXParserFactory. */
    private SAXParserFactory factory;

    /* 解析処理に使用するClassLoader. */
    private ClassLoader classLoader;

    /* CompositeRule. */
    private CompositeRule rule;

    /* CompositeContext. */
    private CompositeContext compositeContext;

    /* DocumentContext. */
    private DocumentContext documentContext;

    /* 解析処理内でエラーが発生した場合に例外を発生させるかを指定する（デフォルトはfalse）。 */
    private boolean errThrowable;

    /**
     * デフォルトコンストラクタ。
     */
    public Composer() {
        initialize();

    }

    /**
     * このオブジェクトを初期化します。
     */
    public void initialize() {
        this.compositeContext = new CompositeContext(this);
        this.log = staticLog;
        this.parser = null;
        this.reader = null;
        this.factory = null;
        this.classLoader = Composer.class.getClassLoader();
        this.rule = null;
        this.errThrowable = false;
    }

    /*
     * Parser
     */

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param inStream
     *            XMLデータ入力ストリーム
     * @throws CompositeException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(CompositeRule rule, InputStream inStream) {
        parse(rule, new InputSource(inStream));
    }

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param input
     *            XMLデータソース
     * @throws CompositeException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(CompositeRule rule, InputSource input) {

        parseBegin(rule);

        try {

            getXMLReader().parse(input);

        } catch (SAXException e) {
            throw new CompositeException("composer parse", e);
        } catch (IOException e) {
            throw new CompositeException("composer parse", e);
        }

        parseFinish();
    }

    /* 解析処理プロセス開始時のフィールド設定を行います。 */
    private void parseBegin(CompositeRule rule) {

        this.rule = rule;
        this.documentContext = new DocumentContext(this.compositeContext, this.rule);
        this.compositeContext.begin();
        this.documentContext.begin();
        parseBeginSAX(rule);

    }

    private void parseBeginSAX(CompositeRule rule) {
        Boolean dtdValidate = rule.getValidating();
        if (dtdValidate != null) {
            this.setValidating(dtdValidate.booleanValue());
        }
        Boolean namespaceAware = rule.getNamespaceAware();
        if (namespaceAware != null) {
            this.setNamespaceAware(namespaceAware.booleanValue());
        }
        EntityResolver entityResolver = rule.getEntityResolver();
        if (entityResolver != null) {
            this.setEntityResolver(entityResolver);
        }
        DTDHandler dtdHandler = rule.getDTDHandler();
        if (dtdHandler != null) {
            this.setDTDHandler(dtdHandler);
        }
        ErrorHandler errorHandler = rule.getErrorHandler();
        if (errorHandler != null) {
            this.setErrorHandler(errorHandler);
        }
        try {
            getXMLReader().setContentHandler(new SAXContentHandler(this));
        } catch (SAXException e) {
            throw new CompositeException("call getXMLReader()", e);
        }
    }

    /* 解析処理プロセス完了時のフィールド設定を行います。 */
    private void parseFinish() {
        this.compositeContext.finish();
        this.documentContext.finish();
    }

    /*
     * Logger
     */

    /**
     * 解析処理プロセス内で使用するログを返却します。
     * 
     * @return ログ
     */
    public Log getLogger() {
        return log;
    }

    /**
     * 解析処理プロセス内で使用するログを格納します。
     * 
     * @param log
     *            ログ
     */
    public void setLogger(Log log) {
        if (log == null) {
            throw new NullPointerException();
        }
        this.log = log;
    }

    /*
     * SAXParser
     */

    /**
     * 解析処理に使用する <code>SAXParserFactory</code> を返します。 <br>
     * 生成に失敗した場合 <code>null</code> が返ります。
     * 
     * @return 解析処理に使用する <code>SAXParserFactory</code>
     */
    public SAXParserFactory getFactory() {
        if (factory == null) {
            try {
                factory = SAXParserFactory.newInstance();
            } catch (FactoryConfigurationError e) {
                log.error("Composer.getFactory: ", e);
                return null;
            }
        }
        return factory;
    }

    /**
     * 解析処理に使用する <code>SAXParser</code> を返します。 <br>
     * 生成に失敗した場合 <code>null</code> が返ります。
     * 
     * @return 解析処理に使用する SAXParser
     */
    public SAXParser getParser() {
        // Return the parser we already created (if any)
        if (parser != null) {
            return parser;
        }
        // Create a new parser
        try {
            parser = getFactory().newSAXParser();
        } catch (Exception e) {
            log.error("Composer.getParser: ", e);
            return null;
        }
        return parser;
    }

    /**
     * 解析処理に使用する <code>XMLReader</code> を返します。 <br>
     * 生成に失敗した場合 <code>null</code> が返ります。
     * 
     * @return 解析処理に使用する <code>XMLReader</code>
     */
    public XMLReader getReader() {
        try {
            return getXMLReader();
        } catch (Exception e) {
            log.error("Composer.getReader:", e);
            return (null);
        }
    }

    /**
     * 解析処理に使用する <code>XMLReader</code> を返します。 <br>
     * 生成に失敗した場合 <code>SAXException</code> が発生します。
     * 
     * @return 解析処理に使用する <code>XMLReader</code>
     * @throws SAXException
     *             リーダの生成に失敗した場合発生します。
     */
    public XMLReader getXMLReader() throws SAXException {
        if (reader == null) {
            reader = getParser().getXMLReader();
            reader.setDTDHandler(new DefaultDTDHandler());
            reader.setEntityResolver(new DefaultEntityResolver());
            reader.setErrorHandler(new DefaultErrorHandler(this));
        }
        return reader;
    }

    /*
     * SAXParser Exceptions for Properties
     */

    /**
     * XMLReader の基本となる実装で要求された特定のプロパティを返します。
     * 
     * @param name
     *            取り出されるプロパティの名前
     * @return 要求されたプロパティの値
     * @throws SAXNotRecognizedException
     *             基本となる XMLReader がプロパティ名を認識しない場合
     * @throws SAXNotSupportedException
     *             基本となる XMLReader はプロパティ名を認識するが、そのプロパティをサポートしない場合
     * @see javax.xml.parsers.SAXParser#getProperty(java.lang.String)
     */
    public Object getProperty(String name) throws SAXNotRecognizedException, SAXNotSupportedException {
        return (getParser().getProperty(name));
    }

    /**
     * XMLReader の基本となる実装に特定のプロパティを設定します。
     * 
     * @param name
     *            設定されるプロパティの名前
     * @param value
     *            設定されるプロパティの値
     * @throws SAXNotRecognizedException
     *             基本となる XMLReader がプロパティ名を認識しない場合
     * @throws SAXNotSupportedException
     *             基本となる XMLReader はプロパティ名を認識するが、そのプロパティをサポートしない場合
     * @see javax.xml.parsers.SAXParser#setProperty(java.lang.String, java.lang.Object)
     */
    public void setProperty(String name, Object value) throws SAXNotRecognizedException, SAXNotSupportedException {
        getParser().setProperty(name, value);
    }

    /*
     * SAXParserFactory Exceptions for Feature
     */

    /**
     * XMLReader の基本となる実装で要求された特定の機能の値を返します。
     * 
     * @param name
     *            取り出される機能の名前
     * @return 要求された機能の値
     * @throws ParserConfigurationException
     *             要求された構成を満たすパーサを生成できない場合
     * @throws SAXNotRecognizedException
     *             基本となる XMLReader が機能名を認識しない場合
     * @throws SAXNotSupportedException
     *             基本となる XMLReader は機能名を認識するが、その機能をサポートしない場合
     * @see javax.xml.parsers.SAXParserFactory#getFeature(java.lang.String)
     */
    public boolean getFeature(String name) throws ParserConfigurationException, SAXNotRecognizedException,
            SAXNotSupportedException {
        return getFactory().getFeature(name);
    }

    /**
     * XMLReader の基本となる実装に特定の機能の値を設定します。
     * 
     * @param name
     *            設定される機能の名前
     * @param value
     *            設定される機能の値
     * @throws ParserConfigurationException
     *             要求された構成を満たすパーサを生成できない場合
     * @throws SAXNotRecognizedException
     *             基本となる XMLReader が機能名を認識しない場合
     * @throws SAXNotSupportedException
     *             基本となる XMLReader は機能名を認識するが、その機能をサポートしない場合
     * @see javax.xml.parsers.SAXParserFactory#setFeature(java.lang.String, boolean)
     */
    public void setFeature(String name, boolean value) throws ParserConfigurationException, SAXNotRecognizedException,
            SAXNotSupportedException {
        getFactory().setFeature(name, value);
    }

    /*
     * EntityResolver
     */

    /**
     * XMLリーダにエンティティリゾルバの登録します。 <br>
     * 登録されない場合はデフォルトのエンティティリゾルバを使用します。
     * 
     * @param resolver
     *            エンティティリゾルバ
     * @see org.xml.sax.XMLReader#setEntityResolver(org.xml.sax.EntityResolver)
     */
    public void setEntityResolver(EntityResolver resolver) {
        getReader().setEntityResolver(resolver);
    }

    /**
     * XMLリーダに登録されるエンティティリゾルバを返却します。 登録されない場合はデフォルトのエンティティリゾルバを返却します。
     * 
     * @return エンティティリゾルバ
     * @see org.xml.sax.XMLReader#getEntityResolver()
     */
    public EntityResolver getEntityResolver() {
        return getReader().getEntityResolver();
    }

    /*
     * DTDHandler
     */

    /**
     * XMLリーダに DTD イベントハンドラの登録します。 <br>
     * 登録されない場合はデフォルトの DTD イベントハンドラを使用します。
     * 
     * @param dtdHandler
     *            DTD イベントハンドラ
     * @see org.xml.sax.XMLReader#setDTDHandler(org.xml.sax.DTDHandler)
     */
    public void setDTDHandler(DTDHandler dtdHandler) {
        getReader().setDTDHandler(dtdHandler);
    }

    /**
     * XMLリーダに登録される DTD イベントハンドラを返却します。 登録されない場合はデフォルトの DTD イベントハンドラを返却します。
     * 
     * @return DTD イベントハンドラ
     * @see org.xml.sax.XMLReader#getDTDHandler()
     */
    public DTDHandler getDTDHandler() {
        return getReader().getDTDHandler();
    }

    /*
     * ErrorHandler
     */

    /**
     * XMLリーダにエラーイベントハンドラの登録します。 <br>
     * 登録されない場合はデフォルトのエラーイベントハンドラを使用します。
     * 
     * @param errorHandler
     *            エラーイベントハンドラ
     * @see org.xml.sax.XMLReader#setErrorHandler(org.xml.sax.ErrorHandler)
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        getReader().setErrorHandler(errorHandler);
    }

    /**
     * XMLリーダに登録されるエラーイベントハンドラを返却します。 登録されない場合はデフォルトのエラーイベントハンドラを返却します。
     * 
     * @return エラーイベントハンドラ
     * @see org.xml.sax.XMLReader#getErrorHandler()
     */
    public ErrorHandler getErrorHandler() {
        return getReader().getErrorHandler();
    }

    /*
     * SAXParserFactory Validating
     */

    /**
     * 構文解析時に XML コンテンツを検証するパーサを作成するようにファクトリが設定されているかどうかを示します。
     * 
     * @return 構文解析時にドキュメントの妥当性を検証する場合は true、そうでない場合は false
     * @see javax.xml.parsers.SAXParserFactory#isValidating()
     */
    public boolean isValidating() {
        return getFactory().isValidating();
    }

    /**
     * 構文解析時にドキュメントの妥当性を検証するように指定します。デフォルトでは、この値は false に設定されます。
     * 
     * @param validating
     *            構文解析時にドキュメントの妥当性を検証する場合は true、そうでない場合は false
     * @see javax.xml.parsers.SAXParserFactory#setValidating(boolean)
     */
    public void setValidating(boolean validating) {
        getFactory().setValidating(validating);
    }

    /*
     * SAXParserFactory NamespaceAware
     */

    /**
     * ファクトリが名前空間を認識するパーサを作成するように設定されているかどうかを示します。
     * 
     * @return 名前空間を認識するパーサを作成するように設定されている場合は true、そうでない場合は false
     * @see javax.xml.parsers.SAXParserFactory#isNamespaceAware()
     */
    public boolean isNamespaceAware() {
        return getFactory().isNamespaceAware();
    }

    /**
     * ファクトリで作成されたパーサが XML 名前空間をサポートするように指定します。デフォルトでは、この値は false に設定されます。
     * 
     * @param namespaceAware
     *            構文解析時にドキュメントの妥当性を検証する場合は true、そうでない場合は false
     * @see javax.xml.parsers.SAXParserFactory#setNamespaceAware(boolean)
     */
    public void setNamespaceAware(boolean namespaceAware) {
        getFactory().setNamespaceAware(namespaceAware);
    }

    /*
     * ClassLoader
     */

    /**
     * 解析処理に使用する ClassLoader を返します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return this.classLoader;
    }

    /**
     * 解析処理に使用する ClassLoader を設定します.
     * 
     * @param classLoader
     *            解析処理に使用する ClassLoader
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;

    }

    /*
     * Error Throwable
     */

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを返却します。
     * 
     * @return <code>true</code> の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public boolean isErrThrowable() {
        return errThrowable;
    }

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを格納します。デフォルトでは、この値は false に設定されます。
     * 
     * @param errThrowable
     *            <code>true</code> の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public void setErrThrowable(boolean errThrowable) {
        this.errThrowable = errThrowable;
    }

    /*
     * Context
     */

    /**
     * 解析処理全体のコンテキスト情報を返します。
     * 
     * @return 解析処理全体のコンテキスト情報
     */
    public CompositeContext getCompositeContext() {
        return compositeContext;
    }

    /**
     * 解析中又は直前に解析したドキュメントのコンテキスト情報を返します。
     * 
     * @return 解析中又は直前に解析したドキュメントのコンテキスト情報
     */
    public DocumentContext getDocumentContext() {
        return documentContext;
    }

    /*
     * CompositeRule
     */

    /**
     * 解析中又は直前に使用した構成ルールを返します。
     * 
     * @return 解析中又は直前に使用した構成ルール
     */
    public CompositeRule getCompositeRule() {
        return rule;
    }

    /*
     * Root Node
     */

    /**
     * 直前の解析処理のルートノードを返します。
     * 
     * @return 直前の解析処理のルートノード
     */
    public Node getRoot() {
        return (null == getDocumentContext()) ? null : getDocumentContext().getRoot();
    }

    /* Default implementation of the EntityResolver interface. */
    private static class DefaultEntityResolver implements EntityResolver {
        public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
            return null;
        }
    }

    /* Default implementation of DTDHandler interface. */
    private static class DefaultDTDHandler implements DTDHandler {

        public void notationDecl(String name, String publicId, String systemId) throws SAXException {
            // no op
        }

        public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName)
                throws SAXException {
            // no op
        }

    }

    /* Default implementation of the ErrorHandler interface. */
    private static class DefaultErrorHandler implements ErrorHandler {

        private final Composer composer;

        DefaultErrorHandler(Composer composer) {
            this.composer = composer;
        }

        public void warning(SAXParseException e) throws SAXException {
            composer.getLogger().warn("ErrorHandler.warning: ", e);
        }

        public void error(SAXParseException e) throws SAXException {
            composer.getLogger().error("ErrorHandler.error: ", e);
        }

        public void fatalError(SAXParseException e) throws SAXException {
            composer.getLogger().fatal("ErrorHandler.fatalError: ", e);
            throw e;
        }

    }

}
