/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General
 * Public License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this library; if not, write to
 * the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.util.cel;

import java.util.LinkedHashSet;

import shohaku.core.collections.Parameters;
import shohaku.core.lang.Seek;

/**
 * セットを生成する式を提供します。
 */
public class SetExpression implements Expression {

    /* この式のリテラル文字を定義します。 */
    private static final Literal[] literals = new Literal[] { new Literal('<', '>') };

    /**
     * リテラル文字 ' <', '>' を返します。
     * 
     * @return リテラル文字
     */
    public Literal[] getLiteral() {
        return literals;
    }

    /**
     * 式を実行して値を返します。
     * 
     * @param binder
     *            処理基のバインダー
     * @param literal
     *            リテラル文字
     * @param expression
     *            式
     * @param begin
     *            開始インデックス
     * @param end
     *            終了インデックス
     * @param values
     *            引数値の一覧
     * @return 式の結果
     */
    public Object execute(CELBinder binder, Literal literal, String expression,
            int begin, int end, Parameters values) {
        String exp = expression.substring(begin, end).trim();
        LinkedHashSet set = new LinkedHashSet();
        int i = 0;
        while (i < exp.length()) {
            int inx = binder.findNextIndex(exp, i);
            if (0 <= inx) {
                set.add(binder.getValue(exp, i, inx, values));
                i = Seek.skipTo(exp, ' ', inx);
                if (i >= exp.length()) {
                    break;
                }
                if (',' != exp.charAt(i)) {
                    throw new IllegalArgumentException();
                }
                i = Seek.skipTo(exp, ' ', i + 1);
            } else {
                break;
            }
        }
        return set;
    }

    /**
     * 式の終端のインデックスを検索して返します。 発見出来ない場合は -1 を返します。
     * 
     * @param binder
     *            処理基のバインダー
     * @param expression
     *            式
     * @param literal
     *            リテラル文字
     * @param begin
     *            開始インデックス
     * @return 式の終端のインデックス
     */
    public int findNextIndex(CELBinder binder, Literal literal, String expression, int begin) {
        int end = (begin + 1);//'<'.length() == 1
        end = Seek.skipTo(expression, ' ', end);
        while (end < expression.length()) {
            end = binder.findNextIndex(expression, end);
            if (-1 == end) {
                return -1;
            }
            end = Seek.skipTo(expression, new char[] { ',', ' ' }, end);
            if (end >= expression.length()) {
                break;
            }
            char c = expression.charAt(end);
            if (c == '>') {
                break;
            }
        }
        return (end + 1);//end to next
    }
}