/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util.cel;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import shohaku.core.collections.Parameters;

/**
 * 簡易式言語を任意に束ね実行する機能を提供します。 <br>
 * <br>
 * <code>CEL</code> は <code>(Composite Expression Language)</code> の略です。
 * <p>
 * この言語はインタープリタであり、完全に実行時にコンパイルされます。 <br>
 * この言語は速度や機能の豊富さ、厳密性等よりも、柔軟性と簡易性を重視して設計されています。 <br>
 * 厳密性や高度な機能を実現する場合、繰り返し実行する場合等の用途には向いていません。 <br>
 * 平易な式を一回または数回処理する用途で使用する事が最適です。
 * </p>
 * <p>
 * 束ねる「式」に応じて機能を増減出来ます。また機能拡張や機能縮小も比較的に簡単です。
 * </p>
 *  
 */
public class CELBinder {

    /* このインスタンスで使用する式のマッピング情報。 */
    private final Map expressionMap;

    /* このインスタンスで使用する式。 */
    private final Expression[] expressions;

    /* このインスタンスで使用するロケール。 */
    private final Locale locale;

    /**
     * デフォルトのロケールと空の式で初期化します。
     */
    public CELBinder() {
        this(Locale.getDefault(), new Expression[0]);
    }

    /**
     * 式とデフォルトのロケールで初期化します。
     * 
     * @param exps
     *            式
     */
    public CELBinder(Expression[] exps) {
        this(Locale.getDefault(), exps, toMappingExpressions(exps));
    }

    /**
     * ロケールと式を指定して初期化します。
     * 
     * @param locale
     *            ロケール
     * @param exps
     *            式
     */
    public CELBinder(Locale locale, Expression[] exps) {
        this(locale, exps, toMappingExpressions(exps));
    }

    /* フォーマッタを初期化します。 */
    private CELBinder(Locale locale, Expression[] exps, Map fsm) {
        this.expressions = exps;
        this.locale = locale;
        this.expressionMap = fsm;
    }

    /**
     * 登録されている式一覧を返却します．
     * 
     * @return 登録されている式一覧
     */
    public Expression[] getExpressions() {
        return (Expression[]) expressions.clone();
    }

    /**
     * ロケールを返却します．
     * 
     * @return ロケール
     */
    public Locale getLocale() {
        return locale;
    }

    /**
     * 式変換を実行します。
     * 
     * @param expression
     *            式構文
     * @return 式の結果
     */
    public Object getValue(String expression) {
        return getValue(expression, null);
    }

    /**
     * 式変換を実行します。
     * 
     * @param expression
     *            式構文
     * @param args
     *            式構文から参照する値
     * @return 式の結果
     */
    public Object getValue(String expression, Parameters args) {
        return getValue(expression, 0, expression.length(), args);
    }

    /**
     * 式変換を実行します。
     * 
     * @param expression
     *            式構文
     * @param begin
     *            式構文中の式の開始インデックス
     * @param end
     *            式構文中の式の終了インデックス
     * @param args
     *            式構文から参照する値
     * @return 式の結果
     */
    public Object getValue(String expression, int begin, int end, Parameters args) {
        return executeValue(expression, begin, end, args);
    }

    /* リテラル文字を持たないリテラルクラス。 */
    private static final Literal NO_SERIAL = new Literal('\u0000', '\u0000');

    /* 個々の要素の式変換を実行します。 */
    private Object executeValue(String expression, int begin, int end, Parameters values) {

        final char beginLiteral = expression.charAt(begin);
        final char endLiteral = expression.charAt(end - 1);
        Literal literal = new Literal(beginLiteral, endLiteral);
        Expression exp = (Expression) expressionMap.get(literal);
        if (exp == null) {
            exp = (Expression) expressionMap.get(NO_SERIAL);
            return exp.execute(this, NO_SERIAL, expression, begin, end, values);
        } else {
            return exp.execute(this, literal, expression, (begin + 1), (end - 1), values);
        }

    }

    /**
     * 次の式の開始位置（式の終端の次）を検出しインデックスを返します。
     * 
     * @param expression
     *            式文字列
     * @param begin
     *            式の相対開始位置
     * @return 次の式の開始位置
     */
    public int findNextIndex(String expression, int begin) {
        final char beginLiteral = expression.charAt(begin);
        for (Iterator i = expressionMap.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            Literal literal = (Literal) e.getKey();
            if (beginLiteral == literal.begin) {
                Expression exp = (Expression) e.getValue();
                int end = exp.findNextIndex(this, literal, expression, begin);
                if (-1 < end) {
                    return end;
                }
            }
        }
        Expression exp = (Expression) expressionMap.get(NO_SERIAL);
        return exp.findNextIndex(this, NO_SERIAL, expression, begin);

    }

    /*
     * static
     */

    /* 式を識別子単位でマッピングして返却します。 */
    private static Map toMappingExpressions(Expression[] exps) {
        Map m = new HashMap((int) (exps.length * 1.5));
        for (int i = 0; i < exps.length; i++) {
            Literal[] literals = exps[i].getLiteral();
            for (int j = 0; j < literals.length; j++) {
                m.put(literals[j], exps[i]);
            }
        }
        return Collections.unmodifiableMap(m);
    }

    /*
     * Default Binder
     */

    /* 基本型を生成する式。 */
    private static final Expression[] baseTypeCreationExpressions;
    static {
        List exps = new ArrayList();
        exps.add(new PrimitiveExpression());
        exps.add(new StringExpression());
        exps.add(new DateTimeExpression());
        exps.add(new RegexPatternExpression());
        baseTypeCreationExpressions = (Expression[]) exps.toArray(new Expression[0]);
    }

    /* 基本型を生成する式。 */
    private static final CELBinder baseTypeCreationBinder = new CELBinder(baseTypeCreationExpressions);

    /**
     * 基本型の生成式バインダーを返却します。 <br>
     * プリミティブ型、文字列型、日付型、正規表現の生成機能をを提供しています。
     * 
     * @return 基本型の生成式バインダー
     */
    public static CELBinder getBaseTypeCreationBinder() {
        return baseTypeCreationBinder;
    }

    /* コレクション型を生成する式。 */
    private static final Expression[] collectionCreationExpressions;
    static {
        List exps = new ArrayList();
        exps.add(new PrimitiveExpression());
        exps.add(new StringExpression());
        exps.add(new DateTimeExpression());
        exps.add(new RegexPatternExpression());
        exps.add(new ListExpression());
        exps.add(new SetExpression());
        exps.add(new MapExpression());
        collectionCreationExpressions = (Expression[]) exps.toArray(new Expression[0]);
    }

    /* コレクション型を生成する式。 */
    private static final CELBinder collectionCreationBinder = new CELBinder(collectionCreationExpressions);

    /**
     * コレクション型が生成可能なバインダーを返却します。 <br>
     * 基本型に加え、コレクション（Map, Set, List）の生成機能をを提供しています。
     * 
     * @return コレクション型が生成可能なバインダー
     */
    public static CELBinder getCollectionCreationBinder() {
        return collectionCreationBinder;
    }

    /* 大半のオブジェクト型が生成可能な式。 */
    private static final Expression[] objectCreationExpressions;
    static {
        List exps = new ArrayList();
        exps.add(new PrimitiveExpression());
        exps.add(new StringExpression());
        exps.add(new DateTimeExpression());
        exps.add(new RegexPatternExpression());
        exps.add(new ListExpression());
        exps.add(new SetExpression());
        exps.add(new MapExpression());
        exps.add(new ObjectCreateExpression());
        objectCreationExpressions = (Expression[]) exps.toArray(new Expression[0]);
    }

    /* 大半のオブジェクト型が生成可能な式。 */
    private static final CELBinder objectCreationBinder = new CELBinder(objectCreationExpressions);

    /**
     * 大半のオブジェクト型が生成可能なバインダーを返却します。 <br>
     * コレクション型に加え、インスタンス生成式を提供しています。
     * 
     * @return 大半のオブジェクト型が生成可能なバインダー
     */
    public static CELBinder getObjectCreationBinder() {
        return objectCreationBinder;
    }

}
