/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * オブジェクトの内部情報を探索するヘルパーメソッド群を提供します。
 */
public class Seek {

    /**
     * 引数に一致する文字をスキップしたインデックスを返します。
     * 
     * @param cs
     *            文字シーケンス
     * @param c
     *            スキップ対象の文字
     * @param begin
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skipTo(CharSequence cs, char c, int begin) {
        int inx = begin;
        while (inx < cs.length()) {
            if (cs.charAt(inx) != c) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数に一致する文字をスキップしたインデックスを返します。
     * 
     * @param cs
     *            文字シーケンス
     * @param chars
     *            スキップ対象の文字の配列
     * @param begin
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skipTo(CharSequence cs, char[] chars, int begin) {
        int inx = begin;
        while (inx < cs.length()) {
            char c = cs.charAt(inx);
            boolean hit = false;
            for (int i = 0; i < chars.length; i++) {
                if (c == chars[i]) {
                    hit = true;
                    break;
                }
            }
            if (!hit) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 指定の値と一致するコレクションの要素のカウント数を返します。
     * 
     * @param c
     *            走査対象のコレクション
     * @param o
     *            検索する値
     * @return 指定の値と一致するコレクションの要素のカウント数
     */
    public static int count(Collection c, Object o) {
        if (c instanceof Set) {
            return (c.contains(o)) ? 1 : 0;
        } else {
            int count = 0;
            for (Iterator i = c.iterator(); i.hasNext();) {
                if (Eval.isEquals(i.next(), o)) {
                    count++;
                }
            }
            return count;
        }
    }

    /**
     * 指定の値と一致するコレクションの全ての要素をコレクションに格納して返します。
     * 
     * @param c
     *            走査対象のコレクション
     * @param o
     *            検索する値
     * @return 指定の値と一致するコレクションの全ての要素
     */
    public static Collection getAll(Collection c, Object o) {
        List a = new ArrayList();
        for (Iterator i = c.iterator(); i.hasNext();) {
            Object e = i.next();
            if (Eval.isEquals(e, o)) {
                a.add(e);
            }
        }
        return a;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。 <br>
     * <code>o</code> が <code>null</code> の場合 <code>(a[i] == null)</code> で検証します。
     * 
     * @param list
     *            検索元のリスト
     * @param o
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置がリストの範囲外の場合
     */
    public static int indexOf(List list, Object o, int begin, int end) {
        checkIndexOfException(list, begin, end);
        if (o == null) {
            for (int i = begin; i <= end; i++) {
                if (list.get(i) == null) {
                    return i;
                }
            }
        } else {
            for (int i = begin; i <= end; i++) {
                if (o.equals(list.get(i))) {
                    return i;
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。 <br>
     * <code>o</code> が <code>null</code> の場合 <code>(a[i] == null)</code> で検証します。
     * 
     * @param list
     *            検索元のリスト
     * @param o
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置がリストの範囲外の場合
     */
    public static int lastIndexOf(List list, Object o, int begin, int end) {
        checkIndexOfException(list, begin, end);
        if (o == null) {
            for (int i = end; i >= begin; i--) {
                if (list.get(i) == null) {
                    return i;
                }
            }
        } else {
            for (int i = end; i >= begin; i--) {
                if (o.equals(list.get(i))) {
                    return i;
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。 <br>
     * <code>o</code> が <code>null</code> の場合 <code>(a[i] == null)</code> で検証します。
     * 
     * @param a
     *            検索元の配列
     * @param o
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(Object[] a, Object o, int begin, int end) {
        checkIndexOfException(a, begin, end);
        if (o == null) {
            for (int i = begin; i <= end; i++) {
                if (a[i] == null) {
                    return i;
                }
            }
        } else {
            for (int i = begin; i <= end; i++) {
                if (o.equals(a[i])) {
                    return i;
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。 <br>
     * <code>o</code> が <code>null</code> の場合 <code>(a[i] == null)</code> で検証します。
     * 
     * @param a
     *            検索元の配列
     * @param o
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(Object[] a, Object o, int begin, int end) {
        checkIndexOfException(a, begin, end);
        if (o == null) {
            for (int i = end; i >= begin; i--) {
                if (a[i] == null) {
                    return i;
                }
            }
        } else {
            for (int i = end; i >= begin; i--) {
                if (o.equals(a[i])) {
                    return i;
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(boolean[] a, boolean e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(boolean[] a, boolean e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(char[] a, char e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(char[] a, char e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(byte[] a, byte e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(byte[] a, byte e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(short[] a, short e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(short[] a, short e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(int[] a, int e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(int[] a, int e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(long[] a, long e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(long[] a, long e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(float[] a, float e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(float[] a, float e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int indexOf(double[] a, double e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = begin; i <= end; i++) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を終了位置から開始位置まで線形検索しそのインデックスを返します。
     * 
     * @param a
     *            検索元の配列
     * @param e
     *            検索する値
     * @param begin
     *            検索の開始インデックス
     * @param end
     *            検索の終了インデックス
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     * @throws NullPointerException
     *             <code>a</code> に <code>null</code> が指定された場合
     * @throws IndexOutOfBoundsException
     *             開始位置または終了位置が配列の範囲外の場合
     */
    public static int lastIndexOf(double[] a, double e, int begin, int end) {
        checkIndexOfException(a, begin, end);
        for (int i = end; i >= begin; i--) {
            if (e == a[i]) {
                return i;
            }
        }
        return -1;
    }

    /* */
    private static void checkIndexOfException(List list, int begin, int end) {
        if (list == null) {
            throw new NullPointerException("list is null.");
        }
        if (!Eval.isRange(list, begin) || !Eval.isRange(list, end)) {
            throw new IndexOutOfBoundsException("begin:" + begin + ", end:" + end + ", size:" + list.size());
        }
    }

    /* */
    private static void checkIndexOfException(Object a, int begin, int end) {
        if (a == null) {
            throw new NullPointerException("a is null.");
        }
        if (!Eval.isArray(a)) {
            throw new IllegalArgumentException("is not Array:" + a);
        }
        if (!Eval.isRange(a, begin) || !Eval.isRange(a, end)) {
            throw new IndexOutOfBoundsException("begin:" + begin + ", end:" + end + ", size:" + Array.getLength(a));
        }
    }

}
