/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.lang;

import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * クラスローダからリソースを読込む機能を提供します。
 */
public class ResourceLoader {

    /** apache commons logging. */
    private static final Log log = LogFactory.getLog(ResourceLoader.class);

    /*
     * class
     */

    /**
     * 指定された名前のクラスをロードして返します。 <br>
     * クラスが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param className
     *            クラス名
     * @return クラス
     * @throws NoSuchResourceException
     *             クラスが見つからなかった場合
     */
    public static Class getClass(String className) throws NoSuchResourceException {
        return getClass(className, null);
    }

    /**
     * 指定された名前のクラスをロードして返します。 <br>
     * クラスが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param className
     *            クラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return クラス
     * @throws NoSuchResourceException
     *             クラスが見つからなかった場合
     */
    public static Class getClass(String className, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = loader;
        //指定クラスの ClassLoader
        try {
            if (classLoader != null) {
                return classLoader.loadClass(className);
            }
        } catch (Exception e) {
            // no op
        }
        //Current ClassLoader から取得
        try {
            return Class.forName(className);
        } catch (Exception e) {
            // no op
        }
        //Current Thread の ClassLoader から取得
        try {
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                return classLoader.loadClass(className);
            }
        } catch (Exception e) {
            // no op
        }
        //システムクラスローダを使って検索します。
        try {
            classLoader = ClassLoader.getSystemClassLoader();
            return classLoader.loadClass(className);
        } catch (Exception e) {
            // no op
        }
        throw new NoSuchResourceException("Not Find Class:" + className);
    }

    /*
     * URL
     */

    /**
     * リソースパスが示すリソースのURLを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static URL getResource(String resource) throws NoSuchResourceException {
        return getResource(resource, ResourceLoader.class);
    }

    /**
     * リソースパスが示すリソースのURLを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param c
     *            リソースの検索に使用するクラス情報
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static URL getResource(String resource, Class c) throws NoSuchResourceException {
        return getResource(resource, c.getClassLoader());
    }

    /**
     * リソースパスが示すリソースのURLを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static URL getResource(String resource, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = null;
        URL url = null;
        try {
            //指定クラスの ClassLoader
            classLoader = loader;
            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null) {
                    return url;
                }
            }
            //Current Thread の ClassLoader
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null) {
                    return url;
                }
            }
            //システムクラスローダを使って検索します。
            url = ClassLoader.getSystemResource(resource);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Not Find Resource '" + resource, e);
            }
        }
        if (url == null) {
            throw new NoSuchResourceException("Not Find Resource '" + resource);
        }
        return url;
    }

    /*
     * InputStream
     */

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static InputStream getResourceAsStream(String resource) throws NoSuchResourceException {
        return getResourceAsStream(resource, ResourceLoader.class);
    }

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param c
     *            リソースの検索に使用するクラス情報
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static InputStream getResourceAsStream(String resource, Class c) throws NoSuchResourceException {
        return getResourceAsStream(resource, c.getClassLoader());
    }

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返します。 <br>
     * リソースが見つからなかった場合 <code>NoSuchResourceException</code> を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static InputStream getResourceAsStream(String resource, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = null;
        InputStream inStream = null;
        try {
            //指定クラスの ClassLoader
            classLoader = loader;
            if (classLoader != null) {
                inStream = classLoader.getResourceAsStream(resource);
                if (inStream != null) {
                    return inStream;
                }
            }
            //Current Thread の ClassLoader
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                inStream = classLoader.getResourceAsStream(resource);
                if (inStream != null) {
                    return inStream;
                }
            }
            //システムクラスローダを使って検索します。
            inStream = ClassLoader.getSystemResourceAsStream(resource);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Not Find Resource '" + resource, e);
            }
        }
        if (inStream == null) {
            throw new NoSuchResourceException("Not Find Resource '" + resource);
        }
        return inStream;
    }

    /*
     * instance
     */

    /**
     * クラス名のインスタンスを生成して返します。 <br>
     * 生成に失敗した場合は <code>ObjectCreationException</code> を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    public static Object getInstance(String className) throws ObjectCreationException {
        return getInstance(className, null);
    }

    /**
     * 指定されたクラス名のインスタンスを生成して返します。 <br>
     * 生成に失敗した場合は <code>ObjectCreationException</code> を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    public static Object getInstance(String className, ClassLoader loader) throws ObjectCreationException {
        try {
            return getClass(className, loader).newInstance();
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("class:" + className, e);
            }
            throw new ObjectCreationException("class:'" + className, e);
        }
    }

    /**
     * 指定されたクラス名のパラメータ型が示すコンストラクタからインスタンスを生成して返します。 <br>
     * 生成に失敗した場合は <code>ObjectCreationException</code> を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @param parameterTypes
     *            コンストラクタに渡すパラメータの型
     * @param parameterValues
     *            コンストラクタに渡すパラメータの値
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    public static Object getInstance(String className, ClassLoader loader, Class[] parameterTypes,
            Object[] parameterValues) throws ObjectCreationException {
        try {
            Class c = getClass(className, loader);
            Constructor constructor = c.getConstructor(parameterTypes);
            return constructor.newInstance(parameterValues);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("class:" + className, e);
            }
            throw new ObjectCreationException("class:'" + className, e);
        }

    }

    /**
     * 指定されたクラスのインスタンスを生成して返します。 <br>
     * 生成に失敗した場合は <code>ObjectCreationException</code> を発生します。
     * 
     * @param c
     *            生成するクラス
     * @return 指定クラスのインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    public static Object getInstance(Class c) throws ObjectCreationException {
        try {
            return c.newInstance();
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("class:" + c, e);
            }
            throw new ObjectCreationException("class:" + c, e);
        }
    }

    /**
     * 指定されたクラスのパラメータ型が示すコンストラクタからインスタンスを生成して返します。 <br>
     * 生成に失敗した場合は <code>ObjectCreationException</code> を発生します。
     * 
     * @param c
     *            生成する
     * @param parameterTypes
     *            コンストラクタに渡すパラメータの型
     * @param parameterValues
     *            コンストラクタに渡すパラメータの値
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    public static Object getInstance(Class c, Class[] parameterTypes, Object[] parameterValues)
            throws ObjectCreationException {
        try {
            Constructor constructor = c.getConstructor(parameterTypes);
            return constructor.newInstance(parameterValues);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("class:" + c, e);
            }
            throw new ObjectCreationException("class:" + c, e);
        }
    }

    /*
     * utils
     */

    /**
     * 現在の <code>Thread</code> のコンテキスト <code>ClassLoader</code> を返します． <br>
     * 取得に失敗した場合Nullを返します。 (JDK 1.2 以降で有効です)
     * 
     * @return 現在の <code>Thread</code> のコンテキスト <code>ClassLoader</code> または <code>null</code>。
     */
    public static ClassLoader getContextClassLoader() {
        ClassLoader classLoader = null;

        try {
            // Are we running on a JDK 1.2 or later system?
            Method method = Thread.class.getMethod("getContextClassLoader", null);

            // Get the thread context class loader (if there is one)
            try {
                classLoader = (ClassLoader) method.invoke(Thread.currentThread(), null);
            } catch (IllegalAccessException e) {
                // no op
            } catch (InvocationTargetException e) {
                // no op
            }
        } catch (NoSuchMethodException e) {
            // Assume we are running on JDK 1.1
            // no op
        }

        // Return thread context class loader or null
        return classLoader;
    }

}
