/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.lang.reflect.Array;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import shohaku.core.collections.Group;

/**
 * 値を評価する式のヘルパーメソッド群を提供します。
 */
public class Eval {

    /*
     * Blank
     */

    /**
     * 引数が <code>null</code> または空の文字シーケンスであるか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が <code>null</code> または空の文字シーケンスの場合のみ <code>true</code>
     */
    public static boolean isBlank(CharSequence cs) {
        return (cs == null || cs.length() == 0);
    }

    /**
     * 引数が全て <code>null</code> または空の文字シーケンスであるか評価します。
     * 
     * @param cs1
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数が全て <code>null</code> または空の文字シーケンスの場合のみ <code>true</code>
     */
    public static boolean isEndBlank(CharSequence cs1, CharSequence cs2) {
        return ((cs1 == null || cs1.length() == 0) && (cs2 == null || cs2.length() == 0));
    }

    /**
     * 全ての配列の要素が <code>null</code> または空の文字シーケンスであるか評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が <code>null</code> または空の文字シーケンスの場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isEndBlank(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            CharSequence cs = css[i];
            if (cs != null && cs.length() != 0) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に <code>null</code> または空の文字シーケンスが含まれるか評価します。
     * 
     * @param cs1
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数に <code>null</code> または空の文字シーケンスが含まれる場合のみ <code>true</code>
     */
    public static boolean isOrBlank(CharSequence cs1, CharSequence cs2) {
        return ((cs1 == null || cs1.length() == 0) || (cs2 == null || cs2.length() == 0));
    }

    /**
     * 全ての配列の要素が <code>null</code> または空の文字シーケンスが含まれるか評価します。
     * 
     * @param css
     *            評価する文字シーケンスの配列
     * @return 全ての配列の要素が <code>null</code> または空の文字シーケンスが含まれる場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isOrBlank(CharSequence[] css) {
        for (int i = 0; i < css.length; i++) {
            CharSequence cs = css[i];
            if (cs == null || cs.length() == 0) {
                return true;
            }
        }
        return false;
    }

    /*
     * Empty
     */

    /**
     * 引数が <code>null</code> または空の状態であるか評価します。
     * 
     * @param os
     *            評価する配列
     * @return 引数が <code>null</code> または空の状態の場合のみ <code>true</code>
     */
    public static boolean isEmpty(Object[] os) {
        return (os == null || os.length == 0);
    }

    /**
     * 引数が <code>null</code> または空の状態であるか評価します。
     * 
     * @param m
     *            評価するマップ
     * @return 引数が <code>null</code> または空の状態の場合のみ <code>true</code>
     */
    public static boolean isEmpty(Map m) {
        return (m == null || m.isEmpty());
    }

    /**
     * 引数が <code>null</code> または空の状態であるか評価します。
     * 
     * @param c
     *            評価するコレクション
     * @return 引数が <code>null</code> または空の状態の場合のみ <code>true</code>
     */
    public static boolean isEmpty(Collection c) {
        return (c == null || c.isEmpty());
    }

    /**
     * 引数が <code>null</code> または空の状態であるか評価します。
     * 
     * @param g
     *            評価するグループ
     * @return 引数が <code>null</code> または空の状態の場合のみ <code>true</code>
     */
    public static boolean isEmpty(Group g) {
        return (g == null || g.isEmpty());
    }

    /*
     * Null
     */

    /**
     * 引数が全て <code>null</code> 値であるか評価します。
     * 
     * @param o1
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数が全て <code>null</code> 値の場合のみ <code>true</code>
     */
    public static boolean isEndNull(Object o1, Object o2) {
        return (o1 == null && o2 == null);
    }

    /**
     * 全ての配列の要素が <code>null</code> 値であるか評価します。
     * 
     * @param os
     *            評価する配列
     * @return 全ての配列の要素が <code>null</code> 値の場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isEndNull(Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (os[i] != null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に <code>null</code> 値が含まれるか評価します。
     * 
     * @param o1
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数に <code>null</code> 値が含まれる場合のみ <code>true</code>
     */
    public static boolean isOrNull(Object o1, Object o2) {
        return (o1 == null || o2 == null);
    }

    /**
     * 引数に <code>null</code> 値が含まれるか評価します。
     * 
     * @param o1
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数に <code>null</code> 値が含まれる場合のみ <code>true</code>
     */
    public static boolean isOrNull(Object o1, Object o2, Object o3) {
        return (o1 == null || o2 == null || o3 == null);
    }
    
    /**
     * 引数に <code>null</code> 値が含まれるか評価します。
     * 
     * @param o1
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @param o4
     *            評価する四番目の値
     * @return 引数に <code>null</code> 値が含まれる場合のみ <code>true</code>
     */
    public static boolean isOrNull(Object o1, Object o2, Object o3, Object o4) {
        return (o1 == null || o2 == null || o3 == null || o4 == null);
    }

    /**
     * 配列の要素に <code>null</code> 値が含まれるか評価します。
     * 
     * @param os
     *            評価する配列
     * @return 配列の要素に <code>null</code> 値が含まれる場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isOrNull(Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (os[i] == null) {
                return true;
            }
        }
        return false;
    }

    /*
     * CharSequence Length
     */

    /**
     * 指定された文字シーケンスの長さが指定の長さと同一か評価します。
     * 
     * @param a
     *            評価する文字シーケンス
     * @param length
     *            文字シーケンスの長さ
     * @return 指定された文字シーケンスの長さが指定の長さと同一の場合のみ <code>true</code>
     */
    public static boolean isLength(CharSequence a, int length) {
        if (a == null) {
            return false;
        }
        return (a.length() == length);
    }

    /**
     * 指定された文字シーケンスの長さが指定の長さの範囲内か評価します。
     * 
     * @param a
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @param max
     *            文字数の最大値
     * @return 指定された文字シーケンスの長さが指定の長さの範囲内の場合のみ <code>true</code>
     */
    public static boolean isLength(CharSequence a, int min, int max) {
        if (a == null) {
            return false;
        }
        return isRange(a.length(), min, max);
    }

    /*
     * Array Length
     */

    /**
     * 指定された配列の長さが指定の配列長と同一か評価します。
     * 
     * @param a
     *            評価する配列
     * @param length
     *            配列長
     * @return 指定された配列の長さが指定の配列長と同一の場合のみ <code>true</code>
     * @exception IllegalArgumentException
     *                引数が配列でない場合
     */
    public static boolean isArrayLength(Object a, int length) {
        if (a == null) {
            return false;
        }
        return (Array.getLength(a) == length);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 指定された配列の範囲内のインデックスの場合のみ <code>true</code>
     * @exception IllegalArgumentException
     *                引数が配列でない場合
     */
    public static boolean isArrayIndex(Object a, int index) {
        if (a == null) {
            return false;
        }
        int length = Array.getLength(a);
        return (0 <= index && index < length);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param fromIndex
     *            開始インデックス
     * @param toIndex
     *            終了インデックス
     * @return 指定された配列の範囲内のインデックスの場合のみ <code>true</code>
     * @exception IllegalArgumentException
     *                引数が配列でない場合
     */
    public static boolean isArrayIndex(Object a, int fromIndex, int toIndex) {
        if (a == null) {
            return false;
        }
        int length = Array.getLength(a);
        return ((0 <= fromIndex && fromIndex < length) && (0 <= toIndex && toIndex < length) && fromIndex <= toIndex);
    }

    /**
     * 指定された全ての配列の長さが同一か評価します。
     * 
     * @param as
     *            評価するの配列を格納とする配列
     * @return 指定された全ての配列の長さが同一の場合のみ <code>true</code>
     * @exception IllegalArgumentException
     *                引数が配列でないまたは引数が空配列の場合
     */
    public static boolean isEqualsLength(Object[] as) {
        if (isEmpty(as)) {
            throw new IllegalArgumentException("array as empty");
        }
        if (isOrNull(as)) {
            return false;
        }
        int l = Array.getLength(as[0]);
        for (int i = 1; i < as.length; i++) {
            if (l != Array.getLength(as[i])) {
                return false;
            }
        }
        return true;
    }

    /*
     * Equals
     */

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(char[] a, char[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(boolean[] a, boolean[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(byte[] a, byte[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(short[] a, short[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(int[] a, int[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(long[] a, long[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(float[] a, float[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(double[] a, double[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列の全要素が一致するか評価します。 <br>
     * 要素は同値か双方 <code>null</code> の場合 <code>true</code> と評価します。
     * 
     * @param a
     *            評価基の配列
     * @param a2
     *            評価先の配列
     * @return 配列の全要素が一致する場合 <code>true</code>
     */
    public static boolean isEquals(Object[] a, Object[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        if (a.length != a2.length) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (!isEquals(a[i], a2[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定の値が同値であるか評価します。 <br>
     * 同値か双方 <code>null</code> の場合 <code>true</code> と評価します。
     * 
     * @param o
     *            評価基の配列
     * @param o2
     *            評価先の配列
     * @return 同値か双方 <code>null</code> の場合 <code>true</code>
     */
    public static boolean isEquals(Object o, Object o2) {
        if (o == o2) {
            return true;
        }
        return (o != null && o.equals(o2));
    }

    /**
     * 指定の値と配列の要素が全て同一値であるか評価します。
     * 
     * @param o
     *            評価基のオブジェクト
     * @param os
     *            評価先のオブジェクト配列
     * @return 指定の値と配列の要素が全て同一値の場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isEndEquals(Object o, Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (!isEquals(o, os[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定の値と配列の要素に同一値が含まれるか評価します。
     * 
     * @param o
     *            評価基のオブジェクト
     * @param os
     *            評価先のオブジェクト配列
     * @return 指定の値と配列の要素に同一値が含まれる場合のみ <code>true</code>
     * @throws NullPointerException
     *             引数の配列が <code>null</code> の場合
     */
    public static boolean isOrEquals(Object o, Object[] os) {
        for (int i = 0; i < os.length; i++) {
            if (isEquals(o, os[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            評価先の接尾辞の配列
     * @return 指定の文字列に接尾辞の内いずれかが含まれる場合のみ <code>true</code>
     */
    public static boolean isOrSuffix(CharSequence cs, char[] suffixs) {
        if (isBlank(cs)) {
            return false;
        }
        char suffix = cs.charAt(cs.length() - 1);
        for (int i = 0; i < suffixs.length; i++) {
            if (suffix == suffixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            評価先の接尾辞の配列
     * @return 指定の文字シーケンスに接尾辞の内いずれかが含まれる場合のみ <code>true</code>
     */
    public static boolean isOrSuffix(CharSequence cs, CharSequence[] suffixs) {
        if (isBlank(cs)) {
            return false;
        }
        for (int i = 0; i < suffixs.length; i++) {
            if (startsWith(cs, suffixs[i], (cs.length() - suffixs[i].length()))) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスの位置から前方一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param offset
     *            評価を開始する相対インデックス
     * @return 指定の文字シーケンスの位置から前方一致する場合のみ <code>true</code>
     */
    public static boolean isStartsWith(CharSequence cs, CharSequence prefix, int offset) {
        if (isBlank(cs)) {
            return false;
        }
        if (startsWith(cs, prefix, offset)) {
            return true;
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @return 指定の文字列に接頭辞の内いずれかが含まれる場合のみ <code>true</code>
     */
    public static boolean isOrPrefix(CharSequence cs, char[] prefixs) {
        if (isBlank(cs)) {
            return false;
        }
        char prefix = cs.charAt(0);
        for (int i = 0; i < prefixs.length; i++) {
            if (prefix == prefixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @return 指定の文字シーケンスに接頭辞の内いずれかが含まれる場合のみ <code>true</code>
     */
    public static boolean isOrPrefix(CharSequence cs, CharSequence[] prefixs) {
        if (isBlank(cs)) {
            return false;
        }
        for (int i = 0; i < prefixs.length; i++) {
            if (startsWith(cs, prefixs[i], 0)) {
                return true;
            }
        }
        return false;
    }

    private static boolean startsWith(CharSequence cs, CharSequence prefix, int offset) {

        int plen = prefix.length();
        if ((offset < 0) || (offset > cs.length() - plen)) {
            return false;
        }
        int to = offset;
        int po = 0;
        while (--plen >= 0) {
            if (cs.charAt(to++) != prefix.charAt(po++)) {
                return false;
            }
        }
        return true;
    }

    /*
     * Range
     */

    /**
     * 指定された文字が指定の範囲内か評価します。
     * 
     * @param c
     *            評価する文字
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(char c, char from, char to) {
        return (from <= c && c <= to);
    }

    /**
     * 指定された文字シーケンスが指定の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(CharSequence cs, char from, char to) {
        int len = cs.length();
        for (int i = 0; i < len; i++) {
            char c = cs.charAt(i);
            if (from > c || c > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された文字シーケンスが指定の文字列数と同一であると同時に文字の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param length
     *            文字列数
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の文字列数と同一であると同時に文字の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(CharSequence cs, int length, char from, char to) {
        int len = cs.length();
        if (len != length) {
            return false;
        }
        for (int i = 0; i < len; i++) {
            char c = cs.charAt(i);
            if (from > c || c > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された文字シーケンスが指定の文字列数と指定の文字の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @param max
     *            文字数の最大値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の文字列数と文字の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(CharSequence cs, int min, int max, char from, char to) {
        int len = cs.length();
        if (!isRange(len, min, max)) {
            return false;
        }
        for (int i = 0; i < len; i++) {
            char c = cs.charAt(i);
            if (from > c || c > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定されたインデックスがリストの範囲内か評価します。
     * 
     * @param list
     *            範囲を示すリスト
     * @param index
     *            評価するインデックス
     * @return インデックスがリストの範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(List list, int index) {
        return (0 <= index && index <= (list.size() - 1));
    }

    /**
     * 指定されたインデックスが配列の範囲内か評価します。
     * 
     * @param array
     *            範囲を示す配列
     * @param index
     *            評価するインデックス
     * @return インデックスが配列の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(Object array, int index) {
        return (0 <= index && index <= (Array.getLength(array) - 1));
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param i
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(int i, int from, int to) {
        return (from <= i && i <= to);
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param i
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(long i, long from, long to) {
        return (from <= i && i <= to);
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param i
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(double i, double from, double to) {
        return (from <= i && i <= to);
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(byte[] a, byte from, byte to) {
        for (int i = 0; i < a.length; i++) {
            byte n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(short[] a, short from, short to) {
        for (int i = 0; i < a.length; i++) {
            short n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(int[] a, int from, int to) {
        for (int i = 0; i < a.length; i++) {
            int n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(long[] a, long from, long to) {
        for (int i = 0; i < a.length; i++) {
            long n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(float[] a, float from, float to) {
        for (int i = 0; i < a.length; i++) {
            float n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param from
     *            範囲の最小値
     * @param to
     *            範囲の最大値
     * @return 指定の範囲内の場合のみ <code>true</code>
     */
    public static boolean isRange(double[] a, double from, double to) {
        for (int i = 0; i < a.length; i++) {
            double n = a[i];
            if (from > n && n > to) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定されたオブジェクトが全てのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param classes
     *            割り当て可能か評価先するクラスの配列
     * @return オブジェクトが全てのクラス型に割り当て可能の場合のみ <code>true</code>
     */
    public static boolean isAndInstanceOf(Object o, Class[] classes) {
        for (int i = 0; i < classes.length; i++) {
            Class c = classes[i];
            if (!c.isInstance(o)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定されたオブジェクトがいずれかのクラス型に割り当て可能か評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param classes
     *            割り当て可能か評価先するクラスの配列
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合のみ <code>true</code>
     */
    public static boolean isOrInstanceOf(Object o, Class[] classes) {
        for (int i = 0; i < classes.length; i++) {
            Class c = classes[i];
            if (c.isInstance(o)) {
                return true;
            }
        }
        return false;
    }

    /**
     * クラスが同一であるか評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param c
     *            比較するクラス
     * @return クラスが同一の場合のみ <code>true</code>
     */
    public static boolean isTypeOf(Object o, Class c) {
        return (o != null && o.getClass().equals(c));
    }

    /**
     * 配列のコンポーネント型が指定のクラスと同一であるか評価します。
     * 
     * @param a
     *            評価する配列
     * @param c
     *            評価先のクラス
     * @return 配列のコンポーネント型が指定のクラスと同一の場合のみ <code>true</code>
     */
    public static boolean isComponentTypeOf(Object a, Class c) {
        if (a == null) {
            return false;
        }
        Class type = a.getClass();
        if (!type.isArray()) {
            throw new IllegalArgumentException("is not Array");
        }
        return (type.getComponentType().equals(c));
    }

    /**
     * 引数 <code>o</code> が配列 <code>a</code> に含まれているか評価します。
     * 
     * @param a
     *            評価基の配列
     * @param o
     *            検索するオブジェクト
     * @return 引数 <code>o</code> が配列 <code>a</code> に含まれている場合のみ <code>true</code>
     */
    public static boolean isContains(Object[] a, Object o) {
        if (a == null) {
            return false;
        }
        return (-1 < Seek.indexOf(a, o, 0, a.length));
    }

    /**
     * 引数 <code>o</code> がコレクション <code>c</code> に含まれているか評価します。
     * 
     * @param c
     *            評価基のコレクション
     * @param o
     *            検索するオブジェクト
     * @return 引数 <code>o</code> がコレクション <code>c</code> に含まれている場合のみ <code>true</code>
     */
    public static boolean isContains(Collection c, Object o) {
        if (c == null) {
            return false;
        }
        return c.contains(o);
    }

    /**
     * 指定の文字シーケンスに指定の文字が含まれているか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param c
     *            検索する文字
     * @return 文字シーケンスに指定の文字が含まれている場合のみ <code>true</code>
     */
    public static boolean isContains(CharSequence cs, char c) {
        if (cs == null) {
            return false;
        }
        if (cs instanceof String) {
            return (0 <= ((String) cs).indexOf(c));
        }
        int len = cs.length();
        for (int i = 0; i < len; i++) {
            if (cs.charAt(i) == c) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに指定の文字列が含まれているか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param str
     *            検索する文字列
     * @return 文字シーケンスに指定の文字列が含まれている場合のみ <code>true</code>
     */
    public static boolean isContains(CharSequence cs, String str) {
        if (cs == null) {
            return false;
        }
        if (cs instanceof String) {
            return (0 <= ((String) cs).indexOf(str));
        }
        if (cs instanceof StringBuffer) {
            return (0 <= ((StringBuffer) cs).indexOf(str));
        }
        return (0 <= indexOf(cs, str.toCharArray()));
    }

    private static int indexOf(CharSequence source, char[] target) {

        int i = 0;
        int max = source.length();
        int targetCount = target.length;
        while (true) {
            if (i > max) {
                return -1;
            }
            int j = i - 1;
            int end = j + targetCount;
            int k = -1;
            while (j < end) {
                if (source.charAt(j++) != target[k++]) {
                    i++;
                    break;
                }
            }
            return i;
        }
    }

    /**
     * 配列型であるか評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @return 配列型の場合のみ <code>true</code>
     */
    public static boolean isArray(Object o) {
        return (o != null && o.getClass().isArray());
    }

    /*
     * Date
     */

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param pattern
     *            文字列の書式パターン
     * @return 日付文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDate(CharSequence date, String pattern) {
        return isDate(date, pattern, Locale.getDefault());
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param pattern
     *            文字列の書式パターン
     * @param locale
     *            ロケール
     * @return 日付文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDate(CharSequence date, String pattern, Locale locale) {
        if (date == null) {
            return false;
        }
        String sdate = date.toString();
        if (pattern.length() != sdate.length()) {
            return false;
        }
        DateFormat format = new SimpleDateFormat(pattern, new DateFormatSymbols(locale));
        format.setLenient(false);
        try {
            format.parse(sdate);
        } catch (ParseException e) {
            return false;
        }
        return true;
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param patterns
     *            文字列の書式パターンの配列
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDate(CharSequence date, String[] patterns) {
        return isDate(date, patterns, Locale.getDefault());
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param patterns
     *            文字列の書式パターンの配列
     * @param locale
     *            ロケール
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDate(CharSequence date, String[] patterns, Locale locale) {
        if (date == null) {
            return false;
        }
        String sdate = date.toString();
        SimpleDateFormat format = new SimpleDateFormat();
        format.setDateFormatSymbols(new DateFormatSymbols(locale));
        format.setLenient(false);
        for (int i = 0; i < patterns.length; i++) {
            String ptrn = patterns[i];
            if (ptrn.length() != sdate.length()) {
                continue;//false
            }
            format.applyPattern(ptrn);
            try {
                format.parse(sdate);
                return true;
            } catch (ParseException e) {
                //false
            }
        }
        return false;
    }

    /*
     * Decimal
     */

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param pattern
     *            文字列の書式パターン
     * @return 数値文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDecimal(CharSequence num, String pattern) {
        return isDecimal(num, pattern, Locale.getDefault());
    }

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param pattern
     *            文字列の書式パターン
     * @param locale
     *            ロケール
     * @return 数値文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDecimal(CharSequence num, String pattern, Locale locale) {
        if (num == null) {
            return false;
        }
        try {
            NumberFormat format = new DecimalFormat(pattern, new DecimalFormatSymbols(locale));
            format.parse(num.toString());
        } catch (ParseException e) {
            return false;
        }
        return true;
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param patterns
     *            文字列の書式パターンの配列
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDecimal(CharSequence num, String[] patterns) {
        return isDecimal(num, patterns, Locale.getDefault());
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param patterns
     *            文字列の書式パターンの配列
     * @param locale
     *            ロケール
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合のみ <code>true</code>
     */
    public static boolean isDecimal(CharSequence num, String[] patterns, Locale locale) {
        if (num == null) {
            return false;
        }
        String snum = num.toString();
        DecimalFormat format = new DecimalFormat();
        format.setDecimalFormatSymbols(new DecimalFormatSymbols(locale));
        for (int i = 0; i < patterns.length; i++) {
            try {
                format.applyPattern(patterns[i]);
                format.parse(snum);
                return true;
            } catch (ParseException e) {
                //false;
            }
        }
        return false;
    }

    /*
     * Encode
     */

    /**
     * 指定の文字セットで指定された文字をエンコードできるかどうかを評価します。
     * 
     * @param c
     *            エンコードできるかどうかを評価する文字
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり <code>true</code>
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, String charsetName) throws UnsupportedCharsetException,
            IllegalCharsetNameException, UnsupportedOperationException {

        return isEncode(c, Charset.forName(charsetName));

    }

    /**
     * 指定の文字セットで指定された文字をエンコードできるかどうかを評価します。
     * 
     * @param c
     *            エンコードできるかどうかを評価する文字
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり <code>true</code>
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, Charset charset) throws UnsupportedOperationException {
        CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(c);
    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかどうかを評価します。
     * 
     * @param cs
     *            エンコードできるかどうかを評価する文字シーケンス
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり <code>true</code>
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, String charsetName) throws UnsupportedCharsetException,
            IllegalCharsetNameException, UnsupportedOperationException {

        return isEncode(cs, Charset.forName(charsetName));

    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかどうかを評価します。
     * 
     * @param cs
     *            エンコードできるかどうかを評価する文字シーケンス
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり <code>true</code>
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, Charset charset) throws UnsupportedOperationException {
        CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(cs);
    }

}
