/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.io;

import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import shohaku.core.beans.BeanUtilities;
import shohaku.core.util.Format;

/**
 * オブジェクトの内部情報を出力する機能を持つ拡張プリントストリームを提供します。 <br>
 * <br>
 * 配列・多次元配列の要素の出力や日付のフォーマット出力、 <code>javaBean</code> のプロパティ出力等の機能が提供されています。
 */
public class IntrospectPrintStream extends PrintStream {

    /**
     * 標準の出力ストリームを出力先として初期化します。
     */
    public IntrospectPrintStream() {
        this(System.out);
    }

    /**
     * 標準の出力ストリームを出力先として初期化します。
     * 
     * @param autoFlush
     *            <code>true</code> の場合 <code>println()</code> メソッドでは出力バッファをフラッシュする
     */
    public IntrospectPrintStream(boolean autoFlush) {
        this(System.out, autoFlush);
    }

    /**
     * 指定された出力ストリームを出力先として初期化します。
     * 
     * @param out
     *            出力ストリーム
     */
    public IntrospectPrintStream(OutputStream out) {
        this(out, false);
    }

    /**
     * 指定された出力ストリームを出力先として初期化します。
     * 
     * @param out
     *            出力ストリーム
     * @param autoFlush
     *            <code>true</code> の場合 <code>println()</code> メソッドでは出力バッファをフラッシュする
     */
    public IntrospectPrintStream(OutputStream out, boolean autoFlush) {
        super(out, autoFlush);
    }

    /*
     * print
     */

    /**
     * 登録済みの出力プリンタを使用して出力ストリームへ出力します。
     * 
     * @param o
     *            出力値
     */
    public void print(Object o) {
        //Null
        if (o == null) {
            this.printNull();
            return;
        }
        //配列, Map, Collection
        if (o.getClass().isArray()) {
            this.printArray(o);
        } else if (o instanceof Map) {
            this.printMap((Map) o);
        } else if (o instanceof Collection) {
            this.printColl((Collection) o);
        } else {
            super.print(o);
        }
    }

    /*
     * 空間出力
     */

    /**
     * 引数分の改行を出力します。
     * 
     * @param len
     *            改行数
     */
    public void printlf(int len) {
        synchronized (this) {
            for (int i = 0; i < len; i++) {
                this.println();
            }
        }
    }

    /**
     * 引数分のスペースを出力します。
     * 
     * @param len
     *            スペース数
     */
    public void printsp(int len) {
        synchronized (this) {
            for (int i = 0; i < len; i++) {
                this.print(' ');
            }
        }
    }

    /*
     * コレクション
     */

    /**
     * 多次元配列を出力します。
     * 
     * <pre>
     *  &lt;ArrayClassType(length)&gt;[item1, item2, &lt;ArrayClassType(length)&gt;[item3-1, item3-2, …], item4]
     * </pre>
     * 
     * 配列以外は print(Object) を呼ぶ
     * 
     * @param a
     *            出力値
     */
    public void printArray(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
            } else if (a.getClass().isArray()) {
                this.printClass(a);
                printDeepArrays(a);
            } else {
                super.print(a);
            }
        }
    }

    /**
     * タイトルと多次元配列を出力します。
     * 
     * <pre>
     *  タイトル : &lt;ArrayClassType(length)&gt;[item1, item2, &lt;ArrayClassType(length)&gt;[item3-1, item3-2, …], item4]
     * </pre>
     * 
     * @param title
     *            タイトル
     * @param a
     *            出力値
     */
    public void printArray(Object title, Object a) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.println();
            this.printArray(a);
            this.println();
        }
    }

    /**
     * コレクションを出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;[ item1, item2, item3, … ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printColl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.print("[]");
            } else {
                this.printClass(c);
                super.print('[');
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                while (hasNext) {
                    Object o = i.next();
                    this.print((o == c) ? "(this Collection)" : o);
                    hasNext = i.hasNext();
                    if (hasNext) {
                        super.print(", ");
                    }
                }
                super.print(']');
            }
        }
    }

    /**
     * マップを出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ key1:value1, key2:value2, key3:value3, … }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printMap(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.print("{}");
            } else {
                this.printClass(m);
                this.print('{');
                boolean st = true;
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.print((key == m) ? "(this Map)" : key);
                    this.print(':');
                    this.print((value == m) ? "(this Map)" : value);
                }
                this.print('}');
            }
        }
    }

    /*
     * JavaBean
     */

    /**
     * <code>JavaBean</code> の保有するプロパティを全て出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ property1=value1, property2=value2, … }
     * </pre>
     * 
     * @param bean
     *            出力する <code>JavaBean</code>
     */
    public void printBean(Object bean) {
        synchronized (this) {

            //プロパティをMapに格納して全取得
            Map props;
            try {
                props = BeanUtilities.getProperties(bean);
            } catch (Exception e) {
                this.println();
                return;
            }
            //出力開始
            //クラス名を出力する
            this.printClass(bean);
            //全プロパティを出力する
            this.print('{');
            boolean st = true;
            for (Iterator i = props.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                if (!"class".equals(e.getKey())) {
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.print((String) e.getKey());
                    this.print('=');
                    this.print(e.getValue());
                }
            }
            this.print('}');
        }
    }

    /**
     * 全ての <code>JavaBean</code> の保有するプロパティを全て出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[length]
     *       [
     *       [0]&lt;ClassName&gt;{ property1=value1, property2=value2, … }
     *       [1]&lt;ClassName&gt;{ property1=value1, property2=value2, … }
     *       …
     *     ]
     * </pre>
     * 
     * @param beans
     *            出力する <code>JavaBean</code> の配列
     */
    public void printBeans(Object[] beans) {
        synchronized (this) {
            //全Beanを出力
            this.printClass(beans);
            this.println();
            this.println('[');
            for (int i = 0; i < beans.length; i++) {
                this.printsp(2);
                this.print('[');
                this.print(i);
                this.print(']');
                this.printBean(beans[i]);
                this.println();
            }
            this.println(']');
        }
    }

    /**
     * public, static, final であるフィールドを全て出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       public static final [field name1] = [field value1]
     *       public static final transient [field name2] = [field value2]
     *       ･･･
     *     }
     * </pre>
     * 
     * @param c
     *            出力するクラス
     */
    public void printConstants(Class c) {
        synchronized (this) {
            try {
                //クラス名を出力する
                this.printClass(c);
                //全フィールドを出力する
                Field[] fields = c.getFields();
                this.println('{');
                for (int i = 0; i < fields.length; i++) {
                    int modifiers = fields[i].getModifiers();
                    // public and static and final
                    if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
                        this.printsp(2);
                        this.print("public static final ");
                        if (Modifier.isTransient(modifiers)) {
                            this.print("transient ");
                        }
                        this.print(fields[i].getName());
                        this.print(" = ");
                        this.println(fields[i].get(null));
                    }
                }
                this.println('}');
            } catch (Exception e) {
                //no op
            }
        }
    }

    /**
     * public であるフィールドを全て出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       public static final [field name1] = [field value1]
     *       public static [field name2] = [field value2]
     *       public final [field name3] = [field value3]
     *       public [field name4] = [field value4]
     *       public transient [field name5] = [field value5]
     *       public volatile [field name6] = [field value6]
     *       ･･･
     *     }  
     * </pre>
     * 
     * @param o
     *            出力するオブジェクト
     */
    public void printFields(Object o) {
        synchronized (this) {
            try {
                //クラス名を出力する
                this.printClass(o);
                //全フィールドを出力する
                Field[] fields = o.getClass().getFields();
                this.println('{');
                for (int i = 0; i < fields.length; i++) {
                    int modifiers = fields[i].getModifiers();
                    // public
                    if (Modifier.isPublic(modifiers)) {
                        this.printsp(2);
                        this.print("public ");
                        if (Modifier.isStatic(modifiers)) {
                            this.print("static ");
                        }
                        if (Modifier.isFinal(modifiers)) {
                            this.print("final ");
                        }
                        if (Modifier.isTransient(modifiers)) {
                            this.print("transient ");
                        }
                        if (Modifier.isVolatile(modifiers)) {
                            this.print("volatile ");
                        }
                        this.print(fields[i].getName());
                        this.print(" = ");
                        this.println(fields[i].get(o));
                    }
                }
                this.println('}');
            } catch (Exception e) {
                //no op
            }
        }
    }

    /**
     * クラス名を出力します。
     * 
     * @param o
     *            出力するオブジェクト
     */
    public void printClass(Object o) {
        synchronized (this) {
            if (o == null) {
                return;
            } else {
                super.print('<');
                super.print((o instanceof Class) ? ((Class) o).getName() : o.getClass().getName());
                super.print('>');
            }
        }
    }

    /**
     * クラス名を出力します。
     * 
     * @param o
     *            出力するオブジェクト
     */
    public void printClass(Object[] o) {
        synchronized (this) {
            if (o == null) {
                return;
            } else {
                super.print('<');
                this.print(o.getClass().getComponentType().getName());
                this.print('[');
                this.print(o.length);
                this.print(']');
                super.print('>');
            }
        }
    }

    /**
     * null 値の文字列表現を出力します。
     */
    public void printNull() {
        this.print("null");
    }

    /*
     * Helper
     */

    /**
     * 指定された配列の「深層内容」の文字列表現を返します。 <br>
     * このメソッドは <code>Message.toString(Object)</code> の同等の変換仕様で設計されています。
     * 
     * @param a
     *            出力する配列
     */
    protected void printDeepArrays(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                return;
            }
            Class aClass = a.getClass();
            if (aClass.isArray()) {
                Class type = aClass.getComponentType();
                if (type.isPrimitive()) {
                    super.print(Format.toString(a));
                } else {
                    // element is an array of object references
                    printDeepArrays((Object[]) a, new HashSet(), 0);
                }
            } else {
                this.print(a);
            }
        }
    }

    /* deepToString(Object[] a) の再起メソッドです。 */
    private void printDeepArrays(Object[] a, Set dejaVu, int row) {
        if (a == null) {
            this.printNull();
            return;
        }
        dejaVu.add(a);
        super.print('[');
        for (int i = 0; i < a.length; i++) {
            if (i != 0) {
                this.print(", ");
            }

            Object e = a[i];
            if (e == null) {
                this.printNull();
            } else if (dejaVu.contains(e)) {
                super.print("[...]");
            } else {
                Class eClass = e.getClass();

                if (eClass.isArray()) {
                    Class type = eClass.getComponentType();
                    if (type.isPrimitive()) {
                        super.print(Format.toString(e));
                    } else {
                        // element is an array of object references
                        printDeepArrays((Object[]) e, dejaVu, (row + 1));
                    }
                } else {
                    this.print(e);
                }
            }
        }
        super.print(']');
        dejaVu.remove(a);
    }

}