/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * 指定されたマップをラッピングするビューを定義します。 <br>
 * <br>
 * このクラスはマップの実装を隠蔽する目的で使用されます。またマップをラッピングするクラスの基底実装の役割を持ちます。
 */
public class WrappedMap implements Map, Serializable {

    /** 基となるマップ。 */
    protected final Map map;

    /**
     * ラップするマップを指定して初期化します。
     * 
     * @param m
     *            ラップするマップ
     */
    public WrappedMap(final Map m) {
        if (m == null) {
            throw new NullPointerException();
        }
        this.map = m;
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#size()
     */
    public int size() {
        return map.size();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#clear()
     */
    public void clear() {
        map.clear();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#isEmpty()
     */
    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key) {
        return map.containsKey(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    public boolean containsValue(Object value) {
        return map.containsValue(value);
    }

    /**
     * 基となるマップの同メソッドの戻り値を <code>WrappedCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see java.util.Map#values()
     */
    public Collection values() {
        return new WrappedCollection(map.values());
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#putAll(java.util.Map)
     */
    public void putAll(Map m) {
        map.putAll(m);
    }

    /**
     * 基となるマップの同メソッドの戻り値を <code>WrappedMap.WrappedMapEntrySet</code> でラッピングしたセットを返します。
     * 
     * @see java.util.Map#entrySet()
     */
    public Set entrySet() {
        return new WrappedMapEntrySet(map.entrySet());
    }

    /**
     * 基となるマップの同メソッドの戻り値を <code>WrappedSet</code> でラッピングしたセットを返します。
     * 
     * @see java.util.Map#keySet()
     */
    public Set keySet() {
        return new WrappedSet(map.keySet());
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#get(java.lang.Object)
     */
    public Object get(Object key) {
        return map.get(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#remove(java.lang.Object)
     */
    public Object remove(Object key) {
        return map.remove(key);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    public Object put(Object key, Object value) {
        return map.put(key, value);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        return map.equals(o);
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return map.hashCode();
    }

    /**
     * 基となるマップの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return map.toString();
    }

    /**
     * <code>Map.Entry</code> をラッピングするセットビューを定義します。
     */
    protected class WrappedMapEntrySet extends WrappedSet {

        /**
         * ラップするセットを指定して初期化します。
         * 
         * @param s
         *            ラップするセット
         */
        protected WrappedMapEntrySet(Set s) {
            super(s);
        }

        /**
         * <code>Map.Entry</code> をラップする反復子を返します。
         * 
         * @see java.util.Collection#iterator()
         */
        public Iterator iterator() {
            return new WrappedIterator(set.iterator()) {
                public Object next() {
                    return new WrappedMapEntry((Map.Entry) i.next());
                }
            };
        }

        /**
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray()
         */
        public Object[] toArray() {
            Object[] a = set.toArray();
            for (int i = 0; i < a.length; i++) {
                a[i] = new WrappedMapEntry((Map.Entry) a[i]);
            }
            return a;
        }

        /**
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray(java.lang.Object[])
         */
        public Object[] toArray(Object a[]) {
            Object[] arr = coll.toArray(a.length == 0 ? a : (Object[]) java.lang.reflect.Array.newInstance(a.getClass()
                    .getComponentType(), 0));
            for (int i = 0; i < arr.length; i++) {
                arr[i] = new WrappedMapEntry((Map.Entry) arr[i]);
            }
            if (arr.length > a.length) {
                return arr;
            }

            System.arraycopy(arr, 0, a, 0, arr.length);
            if (a.length > arr.length) {
                a[arr.length] = null;
            }
            return a;
        }

        /**
         * 
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> に変換して検索します。
         * 
         * @see java.util.Collection#contains(java.lang.Object)
         */
        public boolean contains(Object o) {
            if (!(o instanceof Map.Entry)) {
                return false;
            }
            return set.contains(new WrappedMapEntry((Map.Entry) o));
        }

        /**
         * <code>WrappedMap.WrappedMapEntry</code> で検索するため <code>Set#contains(Object o)</code> を呼び出して実装します。
         * 
         * @see java.util.Collection#containsAll(java.util.Collection)
         */
        public boolean containsAll(Collection coll) {
            Iterator e = coll.iterator();
            while (e.hasNext()) {
                if (!contains(e.next())) { // Invokes safe contains() above
                    return false;
                }
            }
            return true;
        }

    }

    /**
     * <code>Map.Entry</code> をラッピングするビューを定義します。
     */
    protected class WrappedMapEntry implements Map.Entry {

        /** 基となるマップエントリ。 */
        protected Map.Entry e;

        /**
         * ラップするマップエントリを指定して初期化します。
         * 
         * @param e
         *            ラップするマップエントリ
         */
        protected WrappedMapEntry(Map.Entry e) {
            this.e = e;
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.util.Map.Entry#getKey()
         */
        public Object getKey() {
            return e.getKey();
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.util.Map.Entry#getValue()
         */
        public Object getValue() {
            return e.getValue();
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.util.Map.Entry#setValue(java.lang.Object)
         */
        public Object setValue(Object value) {
            return e.setValue(value);
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#hashCode()
         */
        public int hashCode() {
            return e.hashCode();
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        public boolean equals(Object o) {
            return e.equals(o);
        }

        /**
         * 基となるマップエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#toString()
         */
        public String toString() {
            return e.toString();
        }
    }
}