/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import shohaku.core.collections.Group;

/**
 * 指定されたグループの変更不可能なビューを定義します。 <br>
 * ラッピングされたグループが直列化可能の場合は直列化可能です。
 */
public class UnmodifiableGroup extends WrappedGroup {

    /**
     * ラップするグループを指定して初期化します。
     * 
     * @param g
     *            ラップするグループ
     */
    public UnmodifiableGroup(Group g) {
        super(g);
    }

    /*
     * Unmodifiable
     */

    /**
     * 基となるグループの同メソッドの戻り値を <code>UnmodifiableCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see shohaku.core.collections.Group#get(java.lang.Object)
     */
    public Collection get(Object key) {
        Collection c = group.get(key);
        return (c != null) ? new UnmodifiableCollection(c) : null;
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>UnmodifiableSet</code> でラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#keySet()
     */
    public Set keySet() {
        if (keySet == null) {
            keySet = new UnmodifiableSet(group.keySet());
        }
        return keySet;
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>UnmodifiableGroup.UnmodifiableEntrySet</code> でラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#entrySet()
     */
    public Set entrySet() {
        if (entrySet == null) {
            entrySet = new UnmodifiableEntrySet(group.entrySet());
        }
        return entrySet;
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>UnmodifiableIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator()
     */
    public Iterator iterator() {
        return new UnmodifiableIterator(group.iterator());
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>UnmodifiableIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator(java.lang.Object)
     */
    public Iterator iterator(Object key) {
        return new UnmodifiableIterator(group.iterator(key));
    }

    /*
     * UnsupportedOperation
     */

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object)
     */
    public boolean add(Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object, java.lang.Object)
     */
    public boolean add(Object key, Object value) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#addAll(shohaku.core.collections.Group)
     */
    public void addAll(Group g) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#addAll(java.lang.Object, java.util.Collection)
     */
    public boolean addAll(Object key, Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object)
     */
    public Collection remove(Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object, java.lang.Object)
     */
    public boolean remove(Object key, Object value) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#clear()
     */
    public void clear() {
        throw new UnsupportedOperationException();
    }

    /*
     * classes
     */

    /**
     * <code>Group.Entry</code> をラッピングする変更不可能なセットビューを定義します。
     */
    protected class UnmodifiableEntrySet extends UnmodifiableSet {

        /**
         * ラップするセットを指定して初期化します。
         * 
         * @param s
         *            ラップするセット
         */
        protected UnmodifiableEntrySet(Set s) {
            super(s);
        }

        /**
         * <code>Group.Entry</code> をラップする反復子を返します。
         * 
         * @see java.util.Collection#iterator()
         */
        public Iterator iterator() {
            return wrappedIterator(coll.iterator());
        }

        /**
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray()
         */
        public Object[] toArray() {
            Object[] a = coll.toArray();
            for (int i = 0; i < a.length; i++) {
                a[i] = wrappedEntry((Group.Entry) a[i]);
            }
            return a;
        }

        /**
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray(java.lang.Object[])
         */
        public Object[] toArray(Object a[]) {
            Object[] arr = coll.toArray(a.length == 0 ? a : (Object[]) java.lang.reflect.Array.newInstance(a.getClass()
                    .getComponentType(), 0));
            for (int i = 0; i < arr.length; i++) {
                arr[i] = wrappedEntry((Group.Entry) arr[i]);
            }
            if (arr.length > a.length) {
                return arr;
            }

            System.arraycopy(arr, 0, a, 0, arr.length);
            if (a.length > arr.length) {
                a[arr.length] = null;
            }
            return a;
        }

        /**
         * 
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> に変換して検索します。
         * 
         * @see java.util.Collection#contains(java.lang.Object)
         */
        public boolean contains(Object o) {
            if (!(o instanceof Group.Entry)) {
                return false;
            }
            return coll.contains(wrappedEntry((Group.Entry) o));
        }

        /**
         * <code>WrappedGroup.WrappedGroupEntry</code> で検索するため <code>Set#contains(Object o)</code> を呼び出して実装します。
         * 
         * @see java.util.Collection#containsAll(java.util.Collection)
         */
        public boolean containsAll(Collection coll) {
            Iterator e = coll.iterator();
            while (e.hasNext()) {
                if (!contains(e.next())) { // Invokes safe contains() above
                    return false;
                }
            }
            return true;
        }

        /**
         * <code>WrappedGroup.WrappedGroupEntry</code> で検索するため <code>Set#containsAll(Set s)</code> を呼び出して実装します。
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }

            if (!(o instanceof Set)) {
                return false;
            }
            Set s = (Set) o;
            if (s.size() != coll.size()) {
                return false;
            }
            return containsAll(s); // Invokes safe containsAll() above
        }

        /**
         * 反復子を変更不可能なビューでラップします。
         * 
         * @param iterator
         *            反復子
         * @return ラップした反復子
         */
        protected Iterator wrappedIterator(Iterator iterator) {
            return new UnmodifiableIterator(iterator) {
                public Object next() {
                    return wrappedEntry((Group.Entry) super.next());
                }
            };
        }

        /**
         * グループエントリを変更不可能なビューでラップします。
         * 
         * @param entry
         *            グループエントリ
         * @return ラップしたグループエントリ
         */
        protected Group.Entry wrappedEntry(Group.Entry entry) {
            return new UnmodifiableEntry(entry);
        }
    }

    /**
     * <code>Group.Entry</code> をラッピングする変更不可能なビューを定義します。
     */
    protected class UnmodifiableEntry extends WrappedGroupEntry {

        /**
         * ラップするグループエントリを指定して初期化します。
         * 
         * @param e
         *            ラップするグループエントリ
         */
        protected UnmodifiableEntry(Group.Entry e) {
            super(e);
        }

        /**
         * 基となるグループエントリの同メソッドの戻り値を <code>UnmodifiableCollection</code> でラッピングしたコレクションを返します。
         * 
         * @see shohaku.core.collections.Group.Entry#getValues()
         */
        public Collection getValues() {
            return new UnmodifiableCollection(e.getValues());
        }
    }
}