/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Set;

import shohaku.core.collections.Group;

/**
 * 指定されたグループを基にする同期 (スレッドセーフな) ビューを定義します。 <br>
 * 確実に直列アクセスを実現するには、基になるグループへのアクセスはすべて、返されたグループを介して行う必要があります。 <br>
 * <br>
 * 返されたグループのコレクションビューでの繰り返し処理を行う場合、ユーザは、次に示すように手動で同期をとる必要があります。 <br>
 * 
 * <pre>
 *       Group g = new SynchronizedGroup(new HashListGroup());
 *           ...
 *       Set s = g.keySet();  // Needn't be in synchronized block
 *           ...
 *       synchronized(g) {  // Synchronizing on m, not s!
 *           Iterator i = s.iterator(); // Must be in synchronized block
 *           while (i.hasNext())
 *              foo(i.next());
 *       }
 * </pre>
 * 
 * これを行わない場合、動作は保証されません。 <br>
 * <br>
 * ラッピングされたグループが直列化可能の場合は直列化可能です。
 */
public class SynchronizedGroup extends WrappedGroup {

    /** 同期オブジェクト。 */
    protected final Object mutex;

    /**
     * ラップするグループを指定して初期化します。
     * 
     * @param g
     *            ラップするグループ
     */
    public SynchronizedGroup(Group g) {
        super(g);
        this.mutex = this;
    }

    /**
     * 基となるグループの同メソッドの戻り値を 同期コレクションでラッピングしたコレクションを返します。
     * 
     * @see shohaku.core.collections.Group#get(java.lang.Object)
     */
    public Collection get(Object key) {
        synchronized (mutex) {
            return Collections.synchronizedCollection(group.get(key));
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#size()
     */
    public int size() {
        synchronized (mutex) {
            return group.size();
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#size(java.lang.Object)
     */
    public int size(Object key) {
        synchronized (mutex) {
            return group.size(key);
        }
    }

    /**
     * 基となるグループの同メソッドの戻り値を 同期セットでラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#keySet()
     */
    public Set keySet() {
        synchronized (mutex) {
            if (keySet == null) {
                keySet = Collections.synchronizedSet(group.keySet());
            }
            return keySet;
        }
    }

    /**
     * 基となるグループの同メソッドの戻り値を 同期セットでラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#entrySet()
     */
    public Set entrySet() {
        synchronized (mutex) {
            if (entrySet == null) {
                entrySet = Collections.synchronizedSet(group.entrySet());
            }
            return entrySet;
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#isEmpty()
     */
    public boolean isEmpty() {
        synchronized (mutex) {
            return group.isEmpty();
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key) {
        synchronized (mutex) {
            return group.containsKey(key);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#containsValue(java.lang.Object, java.lang.Object)
     */
    public boolean containsValue(Object key, Object value) {
        synchronized (mutex) {
            return group.containsValue(key, value);
        }
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator()
     */
    public Iterator iterator() {
        synchronized (mutex) {
            return group.iterator();//iterator Must be manually synched by
            // user!
        }
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator(java.lang.Object)
     */
    public Iterator iterator(Object key) {
        synchronized (mutex) {
            return group.iterator(key);//iterator Must be manually synched by
            // user!
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object)
     */
    public boolean add(Object key) {
        synchronized (mutex) {
            return group.add(key);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object, java.lang.Object)
     */
    public boolean add(Object key, Object value) {
        synchronized (mutex) {
            return group.add(key, value);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#addAll(shohaku.core.collections.Group)
     */
    public void addAll(Group g) {
        synchronized (mutex) {
            group.addAll(g);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#addAll(java.lang.Object, java.util.Collection)
     */
    public boolean addAll(Object key, Collection c) {
        synchronized (mutex) {
            return group.addAll(key, c);
        }
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object)
     */
    public Collection remove(Object key) {
        synchronized (mutex) {
            return group.remove(key);//Synchronism is unnecessary to the
            // deleted collection.
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object, java.lang.Object)
     */
    public boolean remove(Object key, Object value) {
        synchronized (mutex) {
            return group.remove(key, value);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#clear()
     */
    public void clear() {
        synchronized (mutex) {
            group.clear();
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        synchronized (mutex) {
            return group.equals(o);
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        synchronized (mutex) {
            return group.hashCode();
        }
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        synchronized (mutex) {
            return group.toString();
        }
    }

}
