/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Set;

import shohaku.core.collections.Group;

/**
 * 唯一のキーとコレクションをマッピングする不変のグループを定義します。返されるグループは直列化可能です。
 */
public class SingletonGroup implements Group, Serializable {

    /* 唯一のグループエントリを保存します。 */
    private final Group.Entry entry;

    /**
     * 唯一のキーとコレクションの対を指定して初期化します。
     * 
     * @param key
     *            返されるグループに格納されている唯一のキー
     * @param values
     *            返されるグループによって key がマッピングされるコレクション
     */
    public SingletonGroup(Object key, Collection values) {
        this.entry = new SingletonEntry(key, new ArrayList(values));
    }

    /*
     * UnsupportedOperation
     */

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object)
     */
    public boolean add(Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object, java.lang.Object)
     */
    public boolean add(Object key, Object value) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#addAll(shohaku.core.collections.Group)
     */
    public void addAll(Group g) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#addAll(java.lang.Object, java.util.Collection)
     */
    public boolean addAll(Object key, Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object)
     */
    public Collection remove(Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object, java.lang.Object)
     */
    public boolean remove(Object key, Object value) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see shohaku.core.collections.Group#clear()
     */
    public void clear() {
        throw new UnsupportedOperationException();
    }

    /*
     * Singleton
     */

    private transient Set keySet = null;

    private transient Set entrySet = null;

    /**
     * 唯一のキーを <code>Collections.singleton(Object o)</code> でラッピングした変更不可能なセットを返します。
     * 
     * @see shohaku.core.collections.Group#keySet()
     */
    public Set keySet() {
        if (keySet == null) {
            keySet = Collections.singleton(entry.getKey());
        }
        return keySet;
    }

    /**
     * 唯一のグループエントリを <code>Collections.singleton(Object o)</code> でラッピングした変更不可能なセットを返します。
     * 
     * @see shohaku.core.collections.Group#entrySet()
     */
    public Set entrySet() {
        if (entrySet == null) {
            entrySet = Collections.singleton(entry);
        }
        return entrySet;
    }

    public Collection get(Object key) {
        return (entry.getKey().equals(key)) ? entry.getValues() : null;
    }

    public Iterator iterator() {
        return entrySet().iterator();
    }

    public Iterator iterator(Object key) {
        return (entry.getKey().equals(key)) ? entry.getValues().iterator() : null;
    }

    public boolean isEmpty() {
        return false;
    }

    public boolean containsKey(Object key) {
        return (entry.getKey().equals(key));
    }

    public boolean containsValue(Object key, Object value) {
        return (entry.getKey().equals(key) && entry.getValues().contains(value));
    }

    public int size() {
        return 1;
    }

    public int size(Object key) {
        return (entry.getKey().equals(key)) ? entry.getValues().size() : 0;
    }

    public int hashCode() {
        return entry.hashCode();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof Group)) {
            return false;
        }
        Group t = (Group) o;
        if (t.size() != size()) {
            return false;
        }
        return (entrySet().equals(t.entrySet()));
    }

    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("{");
        buf.append(entry.getKey());
        buf.append('=');
        buf.append(entry.getValues());
        buf.append("}");
        return buf.toString();
    }

    /*
     * グループが唯一保有する変更不可能なグループエントリを定義します。
     */
    private class SingletonEntry implements Group.Entry, Serializable {

        private final Object key;

        private final Collection values;

        private SingletonEntry(Object key, Collection values) {
            this.key = key;
            this.values = new UnmodifiableCollection(values);
        }

        public Object getKey() {
            return key;
        }

        public Collection getValues() {
            return values;
        }

        public int hashCode() {
            return key.hashCode() + values.hashCode();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            return (key.equals(e.getKey()) && values.equals(e.getValues()));
        }

        public String toString() {
            return key + "=" + values;
        }
    }

}