/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.group;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;

import shohaku.core.collections.Group;
import shohaku.core.collections.SetUtils;

/**
 * <code>java.util.LinkedHashMap</code> と <code>java.util.Set</code> を用いた <code>Group</code> インターフェースの実装を提供します。
 * <br>
 * キーの登録順序を保管する <code>Group</code> の実装です。 <br>
 * 以外の特性は <code>HashSetGroup</code> と同様です。
 *  
 */
public class LinkedHashSetGroup extends HashSetGroup {

    /**
     * デフォルトの容量 (16) と負荷係数 (0.75) とデフォルトのコレクション容量 (8) とで LinkedHashSetGroup インスタンスを作成します。
     */
    public LinkedHashSetGroup() {
        super();
    }

    /**
     * 指定された初期容量とデフォルトの負荷係数 (0.75) とデフォルトのコレクション容量 (8) で LinkedHashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @throws IllegalArgumentException
     *             初期容量が負の場合
     */
    public LinkedHashSetGroup(int initialCapacity) {
        super(initialCapacity);
    }

    /**
     * 指定された Group と同じグルーピングで LinkedHashSetGroup インスタンスを作成します。 <br>
     * LinkedHashSetGroup インスタンスは、 指定された Group のグルーピングを保持するのに十分なデフォルトの負荷係数 (0.75) 、および初期容量で作成されます。
     * 
     * @param group
     *            このグループに配置されるグループ
     * @throws NullPointerException
     *             指定されたグループがNullの場合
     */
    public LinkedHashSetGroup(Group group) {
        super(group);
    }

    /**
     * 指定された初期容量と負荷係数とデフォルトのコレクション容量 (8) で LinkedHashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public LinkedHashSetGroup(int initialCapacity, float loadFactor) {
        super(initialCapacity, loadFactor);
    }

    /**
     * 指定された初期容量と負荷係数とコレクション容量で LinkedHashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param setInitialCapacity
     *            コレクションの初期容量
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public LinkedHashSetGroup(int initialCapacity, float loadFactor, int setInitialCapacity) {
        this(new LinkedHashMap(initialCapacity, loadFactor));
        this.initialCapacity = initialCapacity;
        this.loadFactor = loadFactor;
        this.collInitialCapacity = setInitialCapacity;
    }

    /**
     * @param groupMap
     */
    protected LinkedHashSetGroup(Map groupMap) {
        super(groupMap);
    }

    /**
     * <code>java.util.LinkedHashSet</code> 型の実装を持つコレクションを生成して返します。
     * 
     * @return コレクションのコレクション
     * @see shohaku.core.collections.group.AbstractMapGroup#newEntryCollection()
     */
    protected Collection newEntryCollection() {
        return SetUtils.wrappedSet(new LinkedHashSet(this.collInitialCapacity));
    }

    /**
     * LinkedHashSetGroup のクローンオブジェクトを生成して返却します。 <br>
     * <code>
     * よって以下の処理が有効です
     *  LinkedHashSetGroup g = (LinkedHashSetGroup) group.clone();
     * </code>
     * 
     * @return LinkedHashSetGroup のクローンオブジェクト
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        LinkedHashSetGroup result = new LinkedHashSetGroup(this.initialCapacity, this.loadFactor, this.collInitialCapacity);
        result.addAll(this);
        return result;
    }

}
