/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.group;

import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.collections.Group;

/**
 * <code>java.util.LinkedHashMap</code> と <code>java.util.List</code> を用いた <code>Group</code> インターフェースの実装を提供します。
 * <br>
 * キーの登録順序を保管する <code>Group</code> の実装です。 <br>
 * 以外の特性は <code>HashListGroup</code> と同様です。
 *  
 */
public class LinkedHashListGroup extends HashListGroup {

    /**
     * デフォルトの容量 (16) と負荷係数 (0.75) とデフォルトのコレクション容量 (8) とで LinkedHashListGroup インスタンスを作成します。
     */
    public LinkedHashListGroup() {
        super();
    }

    /**
     * 指定された初期容量とデフォルトの負荷係数 (0.75) とデフォルトのコレクション容量 (8) で LinkedHashListGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @throws IllegalArgumentException
     *             初期容量が負の場合
     */
    public LinkedHashListGroup(int initialCapacity) {
        super(initialCapacity);
    }

    /**
     * 指定された Group と同じグルーピングで LinkedHashListGroup インスタンスを作成します。 <br>
     * LinkedHashListGroup インスタンスは、 指定された Group のグルーピングを保持するのに十分なデフォルトの負荷係数 (0.75) 、および初期容量で作成されます。
     * 
     * @param group
     *            このグループに配置されるグループ
     * @throws NullPointerException
     *             指定されたグループがNullの場合
     */
    public LinkedHashListGroup(Group group) {
        super(group);
    }

    /**
     * 指定された初期容量と負荷係数とデフォルトのコレクション容量 (8) で LinkedHashListGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public LinkedHashListGroup(int initialCapacity, float loadFactor) {
        super(initialCapacity, loadFactor);
    }

    /**
     * 指定された初期容量と負荷係数とコレクション容量で LinkedHashListGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param listInitialCapacity
     *            コレクションの初期容量
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public LinkedHashListGroup(int initialCapacity, float loadFactor, int listInitialCapacity) {
        this(new LinkedHashMap(initialCapacity, loadFactor));
        this.initialCapacity = initialCapacity;
        this.loadFactor = loadFactor;
        this.collInitialCapacity = listInitialCapacity;
    }

    /**
     * @param groupMap
     */
    protected LinkedHashListGroup(Map groupMap) {
        super(groupMap);
    }

    /**
     * LinkedHashListGroup のクローンオブジェクトを生成して返却します。 <br>
     * よって以下の処理が有効です。 <br>
     * <code>
     *  LinkedHashListGroup g = (LinkedHashListGroup) group.clone();
     * </code>
     * 
     * @return LinkedHashListGroup のクローンオブジェクト
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        LinkedHashListGroup result = new LinkedHashListGroup(this.initialCapacity, this.loadFactor, this.collInitialCapacity);
        result.addAll(this);
        return result;
    }

}
