/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.group;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import shohaku.core.collections.Group;
import shohaku.core.collections.SetUtils;

/**
 * <code>java.util.HashMap</code> と <code>java.util.Set</code> を用いた <code>Group</code> インターフェースの実装を提供します。 <br>
 * 要素の重複が許可されない大半の文脈において最速である可能性が高い <code>Group</code> の実装です。 <br>
 * 各グループの要素の保管に <code>java.util.Set</code> が使用されているため、順序を持たず要素の重複はありません。 <br>
 * また以外にもセットと同様の特性をもちます。 <br>
 * よって以下の処理が有効です。 <br>
 * <ul>
 * <li><code>Set s = (Set) group.get(key);</code></li>
 * <li><code>Set s = (Set) group.remove(key);</code></li>
 * <li><code>Set s = (Set) entry.getValues();</code></li>
 * </ul>
 * <br>
 * また取得されたセット要素に対して各オペレーションを実行する事が出来ます。<br>
 * ただし参照のセットへの変更は基のグループに連動する事に注意が必要です。 
 * 
 */
public class HashSetGroup extends AbstractMapGroup implements Cloneable, Serializable {

    /*
     * static fields
     */

    /** デフォルトの初期容量。 */
    protected static final int DEFAULT_INITIAL_CAPACITY = 16;

    /** デフォルトの負荷係数。 */
    protected static final float DEFAULT_LOAD_FACTOR = 0.75f;

    /** デフォルトのコレクションの初期容量。 */
    protected static final int DEFAULT_COLL_INITIAL_CAPACITY = 8;

    /*
     * instance fields
     */

    /** 初期容量。 */
    protected int initialCapacity;

    /** 負荷係数。 */
    protected float loadFactor;

    /** コレクションの初期容量。 */
    protected int collInitialCapacity;

    /**
     * デフォルトの容量 (16) と負荷係数 (0.75) とデフォルトのコレクション容量 (8) とで HashSetGroup インスタンスを作成します。
     */
    public HashSetGroup() {
        this(DEFAULT_INITIAL_CAPACITY, DEFAULT_LOAD_FACTOR);
    }

    /**
     * 指定された初期容量とデフォルトの負荷係数 (0.75) とデフォルトのコレクション容量 (8) で HashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @throws IllegalArgumentException
     *             初期容量が負の場合
     */
    public HashSetGroup(int initialCapacity) {
        this(initialCapacity, DEFAULT_LOAD_FACTOR);
    }

    /**
     * 指定された Group と同じグルーピングで HashSetGroup インスタンスを作成します。 <br>
     * HashSetGroup インスタンスは、 指定された Group のグルーピングを保持するのに十分なデフォルトの負荷係数 (0.75) 、および初期容量で作成されます。
     * 
     * @param group
     *            このグループに配置されるグループ
     * @throws NullPointerException
     *             指定されたグループがNullの場合
     */
    public HashSetGroup(Group group) {
        this(Math.max((int) (group.size() / DEFAULT_LOAD_FACTOR) + 1, DEFAULT_INITIAL_CAPACITY), DEFAULT_LOAD_FACTOR);
        addAll(group);
    }

    /**
     * 指定された初期容量と負荷係数とデフォルトのコレクション容量 (8) で HashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public HashSetGroup(int initialCapacity, float loadFactor) {
        this(initialCapacity, loadFactor, DEFAULT_COLL_INITIAL_CAPACITY);
    }

    /**
     * 指定された初期容量と負荷係数とコレクション容量で HashSetGroup インスタンスを作成します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param setInitialCapacity
     *            コレクションの初期容量
     * @throws IllegalArgumentException
     *             初期容量が負であるか、負荷係数が正ではない場合
     */
    public HashSetGroup(int initialCapacity, float loadFactor, int setInitialCapacity) {
        this(new HashMap(initialCapacity, loadFactor));
        this.initialCapacity = initialCapacity;
        this.loadFactor = loadFactor;
        this.collInitialCapacity = setInitialCapacity;
    }

    /**
     * @param groupMap
     */
    protected HashSetGroup(Map groupMap) {
        super(groupMap);
    }

    /**
     * <code>java.util.HashSet</code> 型の実装を持つコレクションを生成して返します。
     * 
     * @return コレクションのコレクション
     * @see shohaku.core.collections.group.AbstractMapGroup#newEntryCollection()
     */
    protected Collection newEntryCollection() {
        return SetUtils.wrappedSet(new HashSet(this.collInitialCapacity));
    }

    /**
     * HashSetGroup のクローンオブジェクトを生成して返却します。 <br>
     * <code>
     * よって以下の処理が有効です
     *  HashSetGroup g = (HashSetGroup) group.clone();
     * </code>
     * 
     * @return HashSetGroup のクローンオブジェクト
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        HashSetGroup result = new HashSetGroup(this.initialCapacity, this.loadFactor, this.collInitialCapacity);
        result.addAll(this);
        return result;
    }

}
