/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.ListIterator;
import java.util.NoSuchElementException;

import shohaku.core.collections.wrapper.UnmodifiableListIterator;

/**
 * <code>ListIterator</code> インタフェースのユーティリティメソッドを提供します。
 */
public class ListIteratorUtils {

    /** 空要素の反復子。 */
    public static final ListIterator EMPTY_LIST_ITERATOR = new ListIterator() {

        public int nextIndex() {
            return -1;
        }

        public boolean hasNext() {
            return false;
        }

        public Object next() {
            throw new NoSuchElementException();
        }

        public int previousIndex() {
            return -1;
        }

        public boolean hasPrevious() {
            return false;
        }

        public Object previous() {
            throw new NoSuchElementException();
        }

        public void add(Object o) {
            throw new UnsupportedOperationException();
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public void set(Object o) {
            throw new UnsupportedOperationException();
        }

    };

    /**
     * 更新不可のリストイテレータにラッピングして返却します。
     * 
     * @param listiterator
     *            ラップするリストイテレータ
     * @return 更新不可のリストイテレータ
     */
    public static ListIterator unmodifiableListIterator(ListIterator listiterator) {
        return new UnmodifiableListIterator(listiterator);
    }

    /**
     * ラッピングしたリストイテレータを返却します。 実装型に固有の機能へのアクセスを制限します。
     * 
     * @param listiterator
     *            ラップするリストイテレータ
     * @return 実装をラッピングしたリストイテレータ
     */
    public static ListIterator wrappedListIterator(ListIterator listiterator) {
        return new UnmodifiableListIterator(listiterator);
    }

}
