/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * <p>
 * キーと値の対とするデータ構造の反復子を提供します。 <br>
 * キーと値を対とするデータ構造を持つ複数の実装に対して、統一されたインターフェースからのアクセスを実現します。
 * </p>
 */
public interface KayValueIterator extends Iterator {

    /**
     * 次の要素に移動しそのキーを返します。
     * 
     * @return 次の要素のキー
     * @throws NoSuchElementException
     *             次の要素が存在しない場合
     */
    Object next();

    /**
     * このインターフェースは削除オペレーションを提供しません。 <br>
     * 実装クラスは必ず <code>UnsupportedOperationException</code> を発生させてください。
     * 
     * @see java.util.Iterator#remove()
     */
    void remove();

    /**
     * キーを返します。 <br>
     * <code>next()</code> が呼び出された後で利用可能です、複数回に同一の要素にアクセス出来ます。
     * 
     * @return キー
     * @throws NoSuchElementException
     *             次の要素が存在しない場合
     * @throws IllegalStateException
     *             <code>next()</code> の呼出が行われていない場合
     */
    Object getKey();

    /**
     * 値を返します。 <br>
     * <code>next()</code> が呼び出された後で利用可能です、複数回に同一の要素にアクセス出来ます。
     * 
     * @return 値
     * @throws NoSuchElementException
     *             次の要素が存在しない場合
     * @throws IllegalStateException
     *             <code>next()</code> の呼出が行われていない場合
     */
    Object getValue();

    /**
     * 値を変更して既存の値を返します。 <br>
     * <code>next()</code> が呼び出された後で利用可能です、複数回に同一の要素にアクセス出来ます。 <br>
     * 任意オペレーション。
     * 
     * @param value
     *            値
     * @return 既存の値
     * @throws NoSuchElementException
     *             次の要素が存在しない場合
     * @throws IllegalStateException
     *             <code>next()</code> の呼出が行われていない場合
     */
    Object setValue(Object value);

    /**
     * 要素数を返します。
     * 
     * @return 要素数
     */
    int size();

}