/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections;

import java.lang.reflect.Array;
import java.util.Iterator;
import java.util.NoSuchElementException;

import shohaku.core.collections.wrapper.UnmodifiableIterator;
import shohaku.core.collections.wrapper.WrappedIterator;
import shohaku.core.lang.Eval;

/**
 * <code>Iterator</code> インタフェースのユーティリティメソッドを提供します。
 */
public class IteratorUtils {

    /** 空要素の反復子。 */
    public static final Iterator EMPTY_ITERATOR = new Iterator() {
        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            return false;
        }

        public Object next() {
            throw new NoSuchElementException();
        }

    };

    private static class ArrayIterator implements Iterator {
        private final Object a;

        private final int size;

        private int index = -1;

        public ArrayIterator(Object a) {
            if (a == null) {
                throw new NullPointerException();
            }
            if (!Eval.isArray(a)) {
                throw new IllegalArgumentException("is not Array:" + a);
            }
            this.a = a;
            this.size = Array.getLength(a);
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            if ((index + 1) < size) {
                index++;
                return true;
            }
            return false;
        }

        public Object next() {
            if (0 > index) {
                throw new IllegalStateException();
            }
            if (index < size) {
                return Array.get(a, index);
            }
            throw new NoSuchElementException();
        }
    }

    /**
     * 配列の要素を反復する更新不可のイテレータを生成して返却します。
     * 
     * @param array
     *            反復する配列
     * @return 配列の要素を反復する更新不可のイテレータ
     */
    public static Iterator arrayIterator(Object array) {
        return new ArrayIterator(array);
    }

    /**
     * 更新不可のイテレータにラッピングして返却します。
     * 
     * @param iterator
     *            ラップするイテレータ
     * @return 更新不可のイテレータ
     */
    public static Iterator unmodifiableIterator(Iterator iterator) {
        return new UnmodifiableIterator(iterator);
    }

    /**
     * ラッピングしたイテレータを返却します。 実装型に固有の機能へのアクセスを制限します。
     * 
     * @param iterator
     *            ラップするイテレータ
     * @return 実装をラッピングしたイテレータ
     */
    public static Iterator wrappedIterator(Iterator iterator) {
        return new WrappedIterator(iterator);
    }

}
