/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import java.lang.reflect.Method;

import shohaku.core.lang.Boxing;
import shohaku.core.util.Format;
import shohaku.core.util.UnmodifiableStateException;

/**
 * メソッドの情報を格納して指定された種別に応じて呼出を実行する機能を提供します。
 */
public final class MethodInfo extends MethodFeatureInfo {

    /** メソッド種別：通常のメソッド。 */
    public static final int METHOD = 0;

    /** メソッド種別：セッタープロパティ。 */
    public static final int SET_PROPERTY = 1;

    /** メソッド種別：ゲッタープロパティ。 */
    public static final int GET_PROPERTY = 2;

    /** メソッド種別：コンストラクタ。 */
    public static final int CONSTRUCTOR = 3;

    /* メソッド種別。 */
    private int type = METHOD;

    /**
     * デフォルトコンストラクタ。
     */
    public MethodInfo() {
        //no op
    }

    /**
     * 引数のプロパティを格納して初期化します。
     * 
     * @param methodName
     *            メソッド名
     * @param params
     *            パラメータ情報のリスト
     * @param type
     *            メソッド種別
     */
    public MethodInfo(String methodName, ParametersInfo params, int type) {
        setMethodName(methodName);
        setParameteres(params);
        setType(type);
    }

    /*
     * protected
     */

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     *            クラス
     * @param obj
     *            インスタンス
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     * @see shohaku.core.beans.MethodFeatureInfo#invokeMethod(java.lang.Class, java.lang.Object)
     */
    protected Object invokeMethod(Class c, Object obj) throws InvocationBeansException {
        if (c == null) {
            throw new NullPointerException("argument class is null.");
        }
        try {

            Method method = getMethod(c);
            return BeanUtilities.invokeMethod(c, obj, method, getParameterValues());

        } catch (NoSuchMethodException e) {
            throw new InvocationBeansException("Class#getMethod class:" + c, e);
        }
    }

    /**
     * 指定されたクラスから処理の対象となるメソッドオブジェクトを返却します。
     * 
     * @param c
     *            クラス
     * @return 処理の対象となるメソッドオブジェクト
     * @throws NoSuchMethodException
     *             メソッドが発見出来なかった場合に発生します
     */
    protected Method getMethod(Class c) throws NoSuchMethodException {
        Method method = null;
        switch (getType()) {
        case METHOD:
            method = BeanUtilities.getMatchingAccessibleMethod(c, getMethodName(), getParameterTypes());
            break;
        case SET_PROPERTY:
            method = BeanUtilities.getMatchingAccessibleSetProperty(c, getMethodName(), getParameterTypes());
            break;
        case GET_PROPERTY:
            method = BeanUtilities.getMatchingAccessibleGetProperty(c, getMethodName(), getParameterTypes());
            break;
        default:
            break;
        }

        if (method == null) {
            throw new NoSuchMethodException(Format.format("type:{0}, class:{1}, method:{2}, args:{3,array}",
                    new Object[] { Boxing.box(getType()), c, getMethodName(), getParameterTypes() }));
        }
        return method;
    }

    /*
     * Property
     */

    /**
     * コンストラクタの場合は true を返す。
     * 
     * @return コンストラクタの場合は true
     */
    public boolean isConstructor() {
        return (getType() == CONSTRUCTOR);
    }

    /**
     * メソッド種別を返却します。
     * 
     * @return メソッド種別
     */
    public int getType() {
        return type;
    }

    /**
     * メソッド種別を格納します。
     * 
     * @param i
     *            メソッド種別
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setType(int i) {
        checkUnmodifiable();

        if (METHOD > i || i > CONSTRUCTOR) {
            throw new IllegalArgumentException();
        }
        type = i;
    }

}
