/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import java.lang.reflect.Method;

import shohaku.core.util.UnmodifiableStateException;

/**
 * メソッドの情報を格納し呼出を実行する機能の一般化された定義を提供します。
 */
public class MethodFeatureInfo extends ClassFeatureInfo {

    /* メソッド名。 */
    private String methodName;

    /* パラメータの情報のリスト。 */
    private ParametersInfo parameteres;

    /*
     * public
     */

    /**
     * 登録されているクラスからメソッドを呼び出し結果を返します。
     * 
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object invoke() throws InvocationBeansException {
        return invoke(getObjectType());
    }

    /**
     * 指定されたクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     *            クラス
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object invoke(Class c) throws InvocationBeansException {
        return invoke(c, null);
    }

    /**
     * 指定されたインスタンスからメソッドを呼び出し結果を返します。
     * 
     * @param obj
     *            インスタンス
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object invoke(Object obj) throws InvocationBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return invoke(obj.getClass(), obj);
    }

    /*
     * protected
     */

    /**
     * 指定されたインスタンスとクラスからメソッド又はコンストラクタを呼び出し結果を返します。
     * 
     * @param c
     *            クラス
     * @param obj
     *            インスタンス
     * @return 生成されたインスタンス
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    protected Object invoke(Class c, Object obj) throws InvocationBeansException {
        if (c == null) {
            throw new NullPointerException("argument is null.");
        }
        if (isConstructor()) {
            return newInstance(c);
        } else {
            return invokeMethod(c, obj);
        }
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     *            クラス
     * @param obj
     *            インスタンス
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    protected Object invokeMethod(Class c, Object obj) throws InvocationBeansException {
        return BeanUtilities.invokeMethod(c, obj, getMethodName(), getParameterTypes(), getParameterValues());
    }

    /**
     * 指定されたクラスからコンストラクタを呼び出しインスタンスを生成して返します。
     * 
     * @param c
     *            クラス
     * @return 生成されたインスタンス
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    protected Object newInstance(Class c) throws InvocationBeansException {
        return BeanUtilities.newInstance(c, getParameterTypes(), getParameterValues());
    }

    /*
     * Property
     */

    /**
     * コンストラクタの場合は true を返す。
     * 
     * @return コンストラクタの場合は true
     */
    public boolean isConstructor() {
        return (getMethodName() == null || getMethodName().length() == 0);
    }

    /**
     * メソッド名を返却します。
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * メソッド名を格納します。
     * 
     * @param name
     *            メソッド名
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setMethodName(String name) {
        checkUnmodifiable();

        if (name == null) {
            throw new NullPointerException("argument is null.");
        }
        methodName = name;
    }

    /**
     * パラメータ情報のリストを返却します。
     * 
     * @return パラメータ情報のリスト
     */
    public ParametersInfo getParameteres() {
        return parameteres;
    }

    /**
     * パラメータの値を返却します。
     * 
     * @return パラメータの値
     */
    public Object[] getParameterValues() {
        return (parameteres == null) ? new Object[0] : parameteres.getParameterValues();
    }

    /**
     * パラメータの型情報を返却します。
     * 
     * @return パラメータの型情報
     */
    public Class[] getParameterTypes() {
        return (parameteres == null) ? new Class[0] : parameteres.getParameterTypes();
    }

    /**
     * パラメータ情報のリストを格納します。
     * 
     * @param params
     *            パラメータ情報のリスト
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setParameteres(ParametersInfo params) {
        checkUnmodifiable();

        if (params == null) {
            throw new NullPointerException("argument is null.");
        }
        this.parameteres = params;
    }

}
