/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

/**
 * <code>BindMethod</code> のインスタンスを構築するビルダーを提供します。 
 */
public class BindMethodBuilder {

    /* 実行基のオブジェクト型。 */
    private Class objectType;

    /* 実行基のインスタンス。 */
    private Object srcObject;

    /* メソッド名。 */
    private String methodName;

    /* パラメータの拘束情報。 */
    private BindParametersInfo bindParameters;

    /**
     * デフォルトコンストラクタ。
     */
    public BindMethodBuilder() {
        //no op
    }

    /**
     * メソッドの拘束を実行し生成されたオブジェクトを返却する。
     * 
     * @return 拘束されたメソッド情報
     * @throws InvocationBeansException
     *             メソッド生成または拘束に失敗した場合
     */
    public BindMethod bind() throws InvocationBeansException {
        BindMethodImpl bindMethodImpl = new BindMethodImpl();
        bindMethodImpl.setMethodName(methodName);
        bindMethodImpl.setObjectType(objectType);
        bindMethodImpl.setSrcObject(srcObject);
        bindMethodImpl.setParameterTypes(bindParameters.getParameterTypes());
        bindMethodImpl.setParameterValues(bindParameters.getParameterValues());
        bindMethodImpl.setParameterNames(bindParameters.geParameterNames());
        bindMethodImpl.setParameterRules(bindParameters.getParameterRules());
        bindMethodImpl.bind();
        return bindMethodImpl;
    }

    /**
     * メソッドの情報を設定します。
     * 
     * @param objectType
     *            実行基のオブジェクト型
     * @param srcObject
     *            実行基のインスタンス
     * @param methodName
     *            メソッド名
     * @param bindParameters
     *            パラメータの拘束情報
     */
    public void setMethod(Class objectType, Object srcObject, String methodName, BindParametersInfo bindParameters) {
        this.objectType = objectType;
        this.srcObject = srcObject;
        this.methodName = methodName;
        this.bindParameters = bindParameters;
    }

    /**
     * メソッド名を返却します．
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * メソッド名を格納します．
     * 
     * @param methodName
     *            メソッド名
     */
    public void setMethodName(String methodName) {
        this.methodName = methodName;
    }

    /**
     * 実行基のオブジェクト型を返却します．
     * 
     * @return 実行基のオブジェクト型
     */
    public Class getObjectType() {
        return objectType;
    }

    /**
     * 実行基のオブジェクト型を格納します．
     * 
     * @param objectType
     *            実行基のオブジェクト型
     */
    public void setObjectType(Class objectType) {
        this.objectType = objectType;
    }

    /**
     * 実行基のインスタンスを返却します．
     * 
     * @return 実行基のインスタンス
     */
    public Object getSrcObject() {
        return srcObject;
    }

    /**
     * 実行基のインスタンスを格納します．
     * 
     * @param srcObject
     *            実行基のインスタンス
     */
    public void setSrcObject(Object srcObject) {
        this.srcObject = srcObject;
    }

    /**
     * パラメータの拘束情報を返却します．
     * 
     * @return パラメータの拘束情報
     */
    public BindParametersInfo getBindParameters() {
        return bindParameters;
    }

    /**
     * パラメータの拘束情報を格納します．
     * 
     * @param bindParameters
     *            パラメータの拘束情報
     */
    public void setBindParameters(BindParametersInfo bindParameters) {
        this.bindParameters = bindParameters;
    }
}
