(* pp camlp4r *)

open ScmDynnum

type environment = {
  stack: ((string list) * (datum ref list)) list;
  top: (string,datum ref) Hashtbl.t;
}
and datum =
    Symbol of string
  | SymbolInEnv of string * environment
  | Ref of datum ref
  | Pair of pair
  | Vector of datum array
  | Nil
  | Number of ScmDynnum.t
  | String of string
  | Character of char
  | Boolean of bool
  | Procedure of datum * datum * environment * (datum ref list)
  | ML_EagerProcedure of (datum -> datum)
  | ML_LazyProcedure of (environment -> (datum -> datum) -> datum -> datum)
  | Delayed of (unit -> datum)
  | Values of pair
  | InputPort of in_channel 
  | OutputPort of out_channel
  | Eof
  | Unspecified
and pair = { 
  mutable car: datum;
  mutable cdr: datum
}

let scm_nil = Nil

let scm_false = Boolean false
let scm_true = Boolean true
let scm_zero = Number ScmDynnum.zero
let scm_one = Number ScmDynnum.one
(* let inexact_one = Number Dynnum.inexact_one *)

(* 

   Schoca uses a symbol table to decrease the memomry usage.
   A second reason for this table is a fast implementation
   of `eq?' (pointer comparison).

*)

let symbol_tbl = Hashtbl.create ScmConfig.initial_symbol_table_size

(* 
   A new symbol has to be created by the function `gen_symbol'.
   Otherwise memory is wasted and `eq?' and `eqv?' will misbehave.
*)

let gen_symbol name =
  try Hashtbl.find symbol_tbl name
  with Not_found -> 
    let sym = Symbol name in
      Hashtbl.add symbol_tbl name sym;
      sym


let is_pair = 
  function
      Pair _ -> true
    | _ -> false

let  is_boolean =
  function
      Boolean _ -> true
    | _ -> false

let is_symbol = 
  function
      Symbol _ -> true
    | _ -> false

let is_number =
  function
    | Number _ -> true
    | _ -> false

let is_vector =
  function
    | Vector _ -> true
    | _ -> false

let is_procedure = 
  function 
    | Procedure _
    | ML_LazyProcedure _ 
    | ML_EagerProcedure _ -> true
    | _ -> false

let is_port =
  function
    | InputPort _ | OutputPort _ -> true
    | _ -> false 

let is_input_port =
  function
    | InputPort _ -> true
    | _ -> false

let is_output_port =
  function
    | OutputPort _ -> true
    | _ -> false

let is_eof =
  function
    | Eof -> true
    | _ -> false


let is_string =
  function
    | String _ -> true
    | _ -> false

let is_list list =
  let rec test clist =
    function
	Nil -> true
      | Pair p -> 
	  if List.exists ((==) p) clist then false
	  else test (p::clist) p.cdr
      | _ -> false
  in test [] list

let is_char =
  function
    | Character _ -> true
    | _ -> false

let char_of_datum =
  function
    | Character c -> c
    | _ -> invalid_arg "char_of"

let datum_of_char c = Character c
let char_of_int i = Character (Char.chr i)
  
let escape_string s = 
  let b = Buffer.create 0 in
  String.iter (function
      '"' -> Buffer.add_string b "\\\""
    | '\\' -> Buffer.add_string b "\\\\"
    | c -> Buffer.add_char b c) s;
  Buffer.contents b
  

let string_of_datum =
  let rec string_of in_list clist =
    function
      | Ref r -> string_of in_list clist !r
      | Symbol s | SymbolInEnv (s,_) -> s
      | String s -> "\"" ^ (escape_string s) ^ "\""
      | Pair c -> 
	  if List.exists ((==) c) clist then "(...))" else
	    let nclist = c::clist in
	    let car_str = string_of false nclist c.car 
	    and sep_str, cdr_str, rpar_str = 
	      if is_pair c.cdr then " ", string_of true nclist c.cdr, ""
	      else if c.cdr = Nil then "", "", ")" 
	      else " . ", string_of false nclist c.cdr, ")" 
	    and lpar_str = 
	      if in_list then ""
	      else "(" in
	      lpar_str ^ car_str ^ sep_str ^ cdr_str ^  rpar_str
      | Values p -> string_of in_list clist p.car 
      | Vector v ->
	  let content =
	    Array.fold_left (fun s d ->
			       let dstr = string_of false clist d in
				 if s = "" then dstr
				 else s ^ " " ^ dstr) "" v in
	    "#(" ^ content ^ ")"
      | Nil -> "()"
      | Number n -> ScmDynnum.string_of n 
      | Boolean true -> "#t"
      | Boolean false -> "#f"
      | Character c -> 
	  (match c with
	     | ' ' -> "#\\space"
	     | '\n' -> "#\\newline"
	     | _ -> "#\\" ^ Char.escaped c)
      | ML_EagerProcedure _
      | ML_LazyProcedure _
      | Procedure _ -> "#procedure"
      | InputPort _ | OutputPort _ -> "#port"
      | Eof -> "#eof"
      | Delayed _ -> "#delayed"
      | Unspecified -> "#unspecified"
  in string_of false []


let cons car cdr = Pair { car = car; cdr = cdr }

  

let string_of_symbol =
  function
      Symbol s -> s
    | _ -> invalid_arg "string_of_symbol"

let in_channel_of_datum =
  function
    | InputPort ich -> ich
    | _ -> invalid_arg "channel_of_input_port"
	

let out_channel_of_datum =
  function
    | OutputPort och -> och
    | _ -> invalid_arg "channel_of_input_port"


let car =
  function
    | Pair c -> c.car
    | a -> failwith ("car of " ^ string_of_datum a)


let cdr = 
  function
      Pair c -> c.cdr
    | a -> failwith ("cdr of " ^ string_of_datum a)

let cadr l = car (cdr l)
let caar l = car (car l)
let cddr l = cdr (cdr l)
let cdar l = cdr (car l)

let caaar l = car (car (car l))
let caadr l = car (car (cdr l))
let cadar l = car (cdr (car l))
let cdaar l = cdr (car (car l))
let caddr l = car (cdr (cdr l))
let cdadr l = cdr (car (cdr l))
let cddar l = cdr (cdr (car l))
let cdddr l = cdr (cdr (cdr l))

let caaaar l = car (car (car (car l)))
let caaadr l = car (car (car (cdr l)))
let caadar l = car (car (cdr (car l)))
let cadaar l = car (cdr (car (car l)))
let cdaaar l = cdr (car (car (car l)))
let caaddr l = car (car (cdr (cdr l)))
let cadadr l = car (cdr (car (cdr l)))
let caddar l = car (cdr (cdr (car l)))
let cdaadr l = cdr (car (car (cdr l)))
let cdadar l = cdr (car (cdr (car l)))
let cddaar l = cdr (cdr (car (car l)))
let cadddr l = car (cdr (cdr (cdr l)))
let cdaddr l = cdr (car (cdr (cdr l)))
let cddadr l = cdr (cdr (car (cdr l)))
let cdddar l = cdr (cdr (cdr (car l)))
let cddddr l = cdr (cdr (cdr (cdr l)))

let set_car l car =
  match l with
    | Pair p -> p.car <- car
    | _ -> invalid_arg "set_car"

let set_cdr l cdr =
  match l with 
    | Pair p -> p.cdr <- cdr
    | _ -> invalid_arg "set_cdr"

let is_true =
  function
      Boolean false -> false
    | _ -> true

let rec list_of_datum =
  function
    | Nil -> []
    | Pair c -> 
	c.car::list_of_datum c.cdr
    | _ -> failwith "list_of_datum"

let rec list_of_improper_list =
  function
    | Nil -> []
    | Pair c ->
	c.car::list_of_improper_list c.cdr
    | e -> [e]



let bool_of_datum = 
  function 
      Boolean false -> false
    | _ -> true

let simple_cell =
  function
      Pair c -> c.cdr = Nil
    | _ -> false

let string_of_scm_string = 
  function
    | String s -> s
    | d -> invalid_arg ("string_of_scm_string:" ^ string_of_datum d)

let number_of_datum =
  function
    | Number n -> n
    | e -> invalid_arg ("number_of: "^ string_of_datum e)

let symbol_of_datum =
  function
    | Symbol s -> String s
    | _ -> invalid_arg "symbol_of"

let datum_of_int i = Number (ScmDynnum.of_int i)
let datum_of_number n = Number n

let datum_of_in_channel ch = InputPort ch
let datum_of_out_channel ch = OutputPort ch

let datum_of_bool b = if b then scm_true else scm_false
let datum_of_string s = String s

(* let string_set s k c =
  match s with
    | String s -> String.set s k c
    | _ -> invalid_arg "string_set"

let string_get s k =
  match s with
    | String s -> String.get s k
    | _ -> invalid_arg "string_get"*)


let exact_int_of_datum =
  function
    | Number n when n.exactness = ScmDynnum.Exact -> 
	ScmDynnum.int_of n
    | e -> invalid_arg ("exact_int_of: "^ string_of_datum e)



let rec reduce f init =
  function
      Nil -> init
    | Pair p -> reduce f (f init p.car) p.cdr
    | _ -> invalid_arg "reduce"

let exists predicate tests =
  let rec iter accum =
    function
      | Nil -> accum
      | Pair p -> 
	  let r = predicate p.car in
	  if is_true r then r
	  else iter r p.cdr
      | _ -> invalid_arg "exits"
  in iter scm_false tests

let for_all predicate tests =
  let rec iter accum =
    function
      |	Nil -> accum
      | Pair p ->
	  let r = predicate p.car in
	  if is_true r then iter r p.cdr
	  else scm_false
      | _ -> invalid_arg "for_all"
  in iter scm_true tests

let reduce2 f init list1 list2 =
  let rec iter acc = 
    function
      | Nil,Nil -> acc
      | Pair c1,Pair c2 -> iter (f acc c1.car c2.car) (c1.cdr,c2.cdr)
      | _ -> invalid_arg "reduce2"
  in iter init (list1,list2)

let append l1 l2 =
  let rec iter =
    function
      | Nil -> l2
      | Pair p -> Pair { p with cdr = iter p.cdr }
      | _ -> invalid_arg  "append"
  in iter l1

let is_values = 
  function
    | Values _ -> true
    | _ -> false

let list_of_values = 
  function
    | Values p -> Pair p 
    | _ -> invalid_arg "list_of_values"

let values_of_list = 
  function
    | Pair p -> Values p 
    | _ -> invalid_arg "values_of_list"

let rec map f =
  function
    | Nil -> Nil
    | Pair c -> 
	if is_values c.car then
	  append (map f (list_of_values c.car)) (map f (c.cdr))
	else
	  cons (f c.car) (map f c.cdr)
    | atom -> f atom

let reverse =
  let rec iter acc =
    function
      | Nil -> acc
      | Pair p -> iter (cons p.car acc) p.cdr
      | _ -> invalid_arg "reverse"
  in iter Nil 
	

let copy = map (fun x -> x)

let rec iter f =
  function 
      Nil -> ()
    | Pair p -> f p.car; iter f p.cdr
    | e -> f e

let rec mem a =
  function
      Nil -> false
    | Pair p -> 
	if p.car = a then true
	else mem a p.cdr 
    | e -> a = e 

let length = reduce (fun a _ -> a+1) 0 


let reverse = 
  let rec iter accum = 
    function
      | Nil -> accum 
      | Pair p -> iter (cons p.car accum) p.cdr
      | _ -> invalid_arg "reverse"
  in iter Nil

let rec list_tail l k =
  if k = 0 then l
  else list_tail (cdr l) (k-1)

let nth list k = car (list_tail list k)

let eqv a b =
  match a with
      Pair _ 
    | Procedure _
    | ML_EagerProcedure _
    | ML_LazyProcedure _
    | Vector _ 
    | Symbol _ -> a == b
    | _ -> a = b

let eq a b = 
  match a, b with
    | Number a, Number b -> ScmDynnum.equal a b
    | _ -> a == b


let equal = (=)

let member_search equal_p element list =
  let rec iter =
    function
      | Nil -> scm_false
      | Pair p as tail ->
	  if equal_p element p.car then tail
	  else iter p.cdr
      | _ -> invalid_arg "member_search"
  in iter list

let memq = member_search eq
let memv = member_search eqv
let member = member_search equal

let assoc_search equal_p element pair_list =
  let rec iter =
    function
      | Nil -> scm_false
      | Pair p ->
	  if equal_p (car p.car) element then p.car
	  else iter p.cdr
      | _ -> invalid_arg "assoc_seach"
  in iter pair_list

let assq = assoc_search eq
let assv = assoc_search eqv
let assoc = assoc_search equal



let vector_of_datum =
  function
    | Vector v -> v
    | _ -> invalid_arg "vector_of"

let vector_of_list list = 
  let len = length list in
  let a = Array.create len Nil in
  let idx = ref ~-1 in
    iter (fun elem -> 
	    incr idx; 
	    a.(!idx) <- elem) list;
    Vector a

let list_of_vector =
  function 
    | Vector vec ->
	Array.fold_right
	(fun value acc ->
	   cons value acc) vec Nil
    | _ -> invalid_arg "list_of_vector"


(*
let string_set s c =
  function
      String s -> String.set s c *)
