(**************************************************************************)
(*  Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.    *)
(*  Copyright (C) 2003, 2004, 2005  Yamagata Yoriyuki                     *)
(*                                                                        *)
(*  This program is free software; you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation; either version 2 of the License, or (at  *)
(*  your option) any later version.                                       *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful, but   *)
(*  WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *)
(*  General Public License for more details.                              *)
(**************************************************************************)

let rec scm_list = function
    [] -> ScmTypes.Nil
  | v :: r ->
      ScmTypes.cons v (scm_list r)

let rec scm_path = function
  | [] -> ScmTypes.Nil
  | (mrph, p, l, state, cost) :: path ->
      let converted = Mrph.surface_form mrph in
      let x = 
	scm_list
	  [ScmTypes.String converted;
	   ScmTypes.datum_of_int p;
	   ScmTypes.datum_of_int l;
	   ScmTypes.datum_of_int state;
	   ScmTypes.datum_of_int cost] in
      ScmTypes.cons x (scm_path path)

let rec scm_candidates = function
  | [] -> ScmTypes.Nil
  | (mrph, p, l) :: path ->
      let converted = Mrph.surface_form mrph in
      let x = 
	scm_list
	  [ScmTypes.String converted;
	   ScmTypes.datum_of_int p;
	   ScmTypes.datum_of_int l] in
      ScmTypes.cons x (scm_candidates path)

let rec ml_path = function
    ScmTypes.Nil -> []
  | (ScmTypes.Pair _) as v  ->
      let e = ScmTypes.car v in
      let string = ScmTypes.string_of_scm_string (ScmTypes.car e) in
      let pos = ScmTypes.exact_int_of_datum (ScmTypes.cadr e) in
      let len = ScmTypes.exact_int_of_datum (ScmTypes.caddr e) in
      (string, pos, len) :: ml_path (ScmTypes.cdr v)
  | _ -> assert false

let rec scm_ptree ptree =
  let scm_best_path_from_parse_obj = 
    let f args =
      let state = ScmTypes.exact_int_of_datum (ScmTypes.car args) in
      let pos = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
      let path = Parse.best_path_from_parse_obj ptree ~state ~pos in 
      scm_path path in
      ScmTypes.ML_EagerProcedure f
  in
  let scm_list_candidates_from_parse_obj =
    let f args =
      let state = ScmTypes.exact_int_of_datum (ScmTypes.car args) in
      let pos = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
      let mrph_len = ScmTypes.exact_int_of_datum (ScmTypes.caddr args) in
      let candidates = 
	try
	  let f (mrph, _, _) = Mrph.surface_form mrph in
	  List.map f
	    (Parse.list_candidates_from_parse_obj  
	       ptree ~state ~pos ~mrph_len) 
	with
	  Not_found -> [] in
      scm_list (List.map (fun s -> ScmTypes.String s) candidates) in
    ScmTypes.ML_EagerProcedure f
  in
  let scm_learn =
    let f args =
      let state = ScmTypes.exact_int_of_datum (ScmTypes.car args) in
      let pos = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
      let path = ml_path (ScmTypes.caddr args) in
      Parse.learn ptree ~state ~pos path;
      ScmTypes.Nil in
    ScmTypes.ML_EagerProcedure f
  in
  let a = Array.init 3 (function 
			    0 -> scm_best_path_from_parse_obj
			  | 1 -> scm_list_candidates_from_parse_obj
			  | 2 -> scm_learn
			  | _ -> assert false)
  in ScmTypes.Vector a

let scm_parse args =
  let string = ScmTypes.string_of_scm_string (ScmTypes.car args) in
  let state = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
  let pos = ScmTypes.exact_int_of_datum (ScmTypes.caddr args) in
  let len = ScmTypes.exact_int_of_datum (ScmTypes.cadddr args) in
  scm_ptree (Parse.parse string ~state ~pos ~len)

let scm_list_candidates args =
  let string = ScmTypes.string_of_scm_string (ScmTypes.car args) in
  let state = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
  let pos = ScmTypes.exact_int_of_datum (ScmTypes.caddr args) in
  let mrph_len = ScmTypes.exact_int_of_datum (ScmTypes.cadddr args) in
  let len = ScmTypes.exact_int_of_datum (ScmTypes.nth args 4) in
  scm_candidates 
    (Parse.list_candidates string ~state ~pos ~mrph_len ~len)


let scm_add_new_word args =
  let kaki = ScmTypes.string_of_scm_string (ScmTypes.car args) in
  let yomi = ScmTypes.string_of_scm_string (ScmTypes.cadr args) in
  Mrph.add_new_word ~kaki ~yomi;
  ScmTypes.Nil

(* Some code comes from Schoca *)
let _ =
  let prompt v =
    print_string (ScmTypes.string_of_datum v ^ "\n"); 
    flush stdout;
    v in
  let env = ScmEval.init_environment () in
  let env, _= ScmEval.register_scm_fun env "mana-parse" scm_parse in
  let env, _= 
    ScmEval.register_scm_fun env "mana-list-candidates" scm_list_candidates in
  let env, _= 
    ScmEval.register_scm_fun env "mana-add-new-word" scm_add_new_word in

  let scm_stream = ScmSchoca.scheme_stream_from_channel 
      (open_in Config.prelude) in
  ignore(ScmSchoca.parse_stream env scm_stream);

  let scm_stream = ScmSchoca.scheme_stream_from_channel stdin in
  let rec exc_loop () =
    try ignore(ScmSchoca.parse_stream env ~prompt scm_stream);
    with 
    | ScmEval.Quit | Exit -> ()
  in exc_loop ()
