;; Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.  
;; Copyright (C) 2003, 2004, 2005  Yamagata Yoriyuki

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;;; The file is a derivative of
;;; egg/anthy.el --- ANTHY Support (high level interface) in Egg
;;;                Input Method Architecture

;; Copyright (C) 2002 The Free Software Initiative of Japan

;; Author: NIIBE Yutaka <gniibe@m17n.org>

;; Maintainer: NIIBE Yutaka <gniibe@m17n.org>

;; Keywords: mule, multilingual, input method

;; This file is part of EGG.

;; EGG is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; EGG is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:


;;; Code:

(require 'egg)
(require 'egg-edep)

(defgroup egg-mana nil
  "Mana interface for Tamago 4."
  :group 'egg)

(setplist 'egg-mana-conversion-backend
	  '(egg-start-conversion          egg-mana-convert
	    egg-get-bunsetsu-source       egg-mana-get-bunsetsu-source
	    egg-get-bunsetsu-converted    egg-mana-get-bunsetsu-converted
	    egg-list-candidates           egg-mana-get-candidates
	    egg-decide-candidate          egg-mana-select-candidate
	    egg-change-bunsetsu-length    egg-mana-resize-segment
	    egg-end-conversion            egg-mana-commit
	    ;;
	    egg-get-source-language       egg-mana-get-source-language
	    egg-get-converted-language    egg-mana-get-converted-language
	    egg-word-registration         egg-mana-word-registration))

(defconst egg-mana-backend-alist '((Japanese ((egg-mana-conversion-backend)))))

(egg-set-finalize-backend '(egg-mana-finalize-backend))

;; <ptid> ::= <integer>

(defvar egg-mana-proc nil
  "Process of MANA helper agent.")

(make-local-variable 'egg-mana-cmd-pending)
(setq-default egg-mana-cmd-pending nil)

(defun egg-mana-cmd-prim (val)
  (let ((buffer (process-buffer egg-mana-proc)))
    (if (and (eq (process-status egg-mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (let ((p (point)))
	     (newline)
	     (prin1 val buffer)
	     (newline)
	     (process-send-region egg-mana-proc p (point))
	     (setq egg-mana-cmd-pending t)
;;	     (accept-process-output egg-mana-proc)
	     ))
      (egg-error "process %s was killed" egg-mana-proc))))

(defun egg-mana-init-proc () nil)

(defun egg-mana-start-proc ()
  (if (null egg-mana-proc)
      (let ((buf (generate-new-buffer "*EGG-MANA*"))
	    (process-connection-type nil)) ; avoid using pty
	(setq egg-mana-proc
	      (start-process "mana" buf "mana"))
	(process-kill-without-query egg-mana-proc)
	(set-process-coding-system egg-mana-proc 'euc-jp-unix 'euc-jp-unix)
	(set-process-sentinel egg-mana-proc 'egg-mana-proc-sentinel)
	(set-marker-insertion-type (process-mark egg-mana-proc) t)
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (buffer-disable-undo))
	(egg-mana-init-proc))))

;;; XXX: Don't kill buffer (for now) so that I can debug this program
(defun egg-mana-proc-sentinel (proc reason)
;  (kill-buffer (process-buffer proc))
  (setq egg-mana-proc nil))

(defun egg-mana-cmd (val)
  (egg-mana-start-proc)
  (egg-mana-cmd-prim val))

(defun egg-mana-eval (val)
  (egg-mana-start-proc)
  (let ((buffer (process-buffer egg-mana-proc)))
    (if (and (eq (process-status egg-mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (goto-char (point-max))
	   (let ((p (point)))
	     (when egg-mana-cmd-pending 
	       (accept-process-output egg-mana-proc 0 10)
	       (setq egg-mana-cmd-pending nil))
	     (prin1 val buffer)
	     (newline)
	     (process-send-region egg-mana-proc p (point))
	     (accept-process-output egg-mana-proc)
	     (goto-char (point-max))
	     (forward-line -1)
	     (beginning-of-line)
	     (read buffer)))
      (egg-error "process %s was killed" egg-mana-proc))))

(defun egg-mana-best-path (yomi state pos len) 
  (egg-mana-eval `(mana-best-path ,yomi ,state ,pos ,len)))

(defun egg-mana-list-candidates (yomi state pos mrph-len len)
  (egg-mana-eval `(mana-list-candidates ,yomi ,state ,pos ,mrph-len ,len)))

(defun egg-mana-add-new-word (kaki yomi)
  (egg-mana-cmd `(mana-add-new-word ,kaki ,yomi)))

(defun egg-mana-learn (string state pos len path)
  (egg-mana-cmd `(mana-learn ,string ,state ,pos ,len ',path)))

;;
;; <egg-mana-bunsetsu> ::=
;;  [ <string> <pos> <len> <state> <converted> <candidates> <candidate-pos> ]
(defun egg-mana-make-bunsetsu (string pos len state converted)
  (egg-bunsetsu-create
   'egg-mana-conversion-backend
   (vector string pos len state converted nil 0)))

(defun egg-manabunsetsu-get-string (b)
  (aref (egg-bunsetsu-get-info b) 0))
(defun egg-manabunsetsu-get-pos (b)
  (aref (egg-bunsetsu-get-info b) 1))
(defun egg-manabunsetsu-get-len (b)
  (aref (egg-bunsetsu-get-info b) 2))
(defun egg-manabunsetsu-get-state (b)
  (aref (egg-bunsetsu-get-info b) 3))

(defun egg-manabunsetsu-get-source (b)
  (let ((string (aref (egg-bunsetsu-get-info b) 0))
	(pos (aref (egg-bunsetsu-get-info b) 1))
	(len (aref (egg-bunsetsu-get-info b) 2)))
    (substring string pos (+ pos len))))

(defun egg-manabunsetsu-get-converted (b)
  (aref (egg-bunsetsu-get-info b) 4))
(defun egg-manabunsetsu-get-candidates (b)
  (aref (egg-bunsetsu-get-info b) 5))
(defun egg-manabunsetsu-set-candidates (b z)
  (aset (egg-bunsetsu-get-info b) 5 z))
(defun egg-manabunsetsu-get-candidate-pos (b)
  (aref (egg-bunsetsu-get-info b) 6))
(defun egg-manabunsetsu-set-candidate-pos (b zp)
  (aset (egg-bunsetsu-get-info b) 6 zp))

(defun egg-mana-get-bunsetsu-source (b)
  (egg-manabunsetsu-get-source b))

(defun egg-mana-get-bunsetsu-converted (b)
  (let ((cands (egg-manabunsetsu-get-candidates b)))
    (if cands
	(car (nth (egg-manabunsetsu-get-candidate-pos b) cands))
      (egg-manabunsetsu-get-converted b))))

(defun egg-mana-get-source-language (b) 'Japanese)
(defun egg-mana-get-converted-language (b) 'Japanese)

;;
;; Returns list of bunsetsu
;;

(defun egg-mana-make-bunsetsu-list (string path)
  (mapcar (lambda (node)
	    (let* ((converted (elt node 0))
		   (pos (elt node 1))
		   (len (elt node 2))
		   (state (elt node 3)))
	      (egg-mana-make-bunsetsu string pos len state converted)))
	  path))

(defun egg-mana-convert (backend yomi &optional context)
  "Convert YOMI string to kanji, and enter conversion mode.
Return the list of bunsetsu."
  (set-text-properties 0 (length yomi) nil yomi)
  (egg-mana-make-bunsetsu-list yomi 
			   (egg-mana-best-path yomi 0 0 (length yomi))))

;;
;;
;;

(defun egg-mana-commit (bunsetsu-list abort)
  (let ((path
	 (mapcar (lambda (b) 
		   (let ((converted (egg-mana-get-bunsetsu-converted b))
			 (pos (egg-manabunsetsu-get-pos b))
			 (len (egg-manabunsetsu-get-len b)))
		     (list converted pos len)))
		 bunsetsu-list)))
    (if bunsetsu-list
	(let* ((string (egg-manabunsetsu-get-string (car bunsetsu-list)))
	      (pos (egg-manabunsetsu-get-pos (car bunsetsu-list)))
	      (len (length string)))
	  (egg-mana-learn string 0 pos len path)))))

;;
;; Returns ( <pos> <candidates> )
;;
(defun egg-mana-get-candidates (bunsetsu-list prev-bunsetsu next-bunsetsu major)
  (let ((bunsetsu (car bunsetsu-list)))

    (unless (egg-manabunsetsu-get-candidates bunsetsu)
      (let* ((state (if prev-bunsetsu 
			(egg-manabunsetsu-get-state (car prev-bunsetsu)) 
		      0))
	     (source (egg-manabunsetsu-get-source bunsetsu))
	     (len (egg-manabunsetsu-get-len bunsetsu))
	     (cands (egg-mana-list-candidates source state 0 len len)))
	(cons (egg-manabunsetsu-set-candidate-pos bunsetsu 0)
	      (egg-manabunsetsu-set-candidates bunsetsu cands))))

      (cons (egg-manabunsetsu-get-candidate-pos bunsetsu)
	    (mapcar 'car (egg-manabunsetsu-get-candidates bunsetsu)))))

;; Helper function
(defun egg-mana-get-yomi (bunsetsu-list) 
  (mapconcat 'egg-manabunsetsu-get-source bunsetsu-list ""))

;; Returns list of list of bunsetsu

(defun egg-mana-select-candidate (bunsetsu-list candidate-pos prev-b next-b)
  (let ((bunsetsu (car bunsetsu-list)))
    (egg-manabunsetsu-set-candidate-pos bunsetsu candidate-pos)
    (list (list bunsetsu))))

;; Returns list of list of bunsetsu
(defun egg-mana-resize-segment (bunsetsu-list prev-b next-b len major)
  (let* ((bunsetsu (car bunsetsu-list))
	 (string (egg-manabunsetsu-get-string bunsetsu))
	 (pos (egg-manabunsetsu-get-pos bunsetsu))
	 (state (egg-manabunsetsu-get-state bunsetsu))
	 (candidate (car (egg-mana-list-candidates string state pos len len)))
	 (conv (nth 0 candidate))
	 (next-state (nth 1 candidate))
	 (next-pos (+ pos len))
	 (next-len (- (length string) next-pos))
	 (path (egg-mana-best-path string next-state next-pos next-len)))
    (list (list (egg-mana-make-bunsetsu string pos len state conv))
	  nil 
	  (egg-mana-make-bunsetsu-list string path))))

(defun egg-mana-finalize-backend ()
  (if egg-mana-proc
      (progn
	(delete-process egg-mana-proc)
	(setq egg-mana-proc nil))))

;;Register new word
(defun egg-mana-word-registration (backend str yomi)
  (let ((str (copy-sequence str))
	(yomi (copy-sequence yomi)))
    (set-text-properties 0 (length str) nil str)
    (set-text-properties 0 (length yomi) nil yomi)
    (egg-mana-add-new-word str yomi)
    '("̤" "Ŀͼ")))

;;; setup

(run-hooks 'egg-mana-load-hook)

;;;###autoload
(defun egg-activate-mana (&rest arg)
  "Activate MANA backend of Tamago 4."
  (apply 'egg-mode (append arg egg-mana-backend-alist)))

;;; egg-mana.el ends here.