/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*            Francois Rouaix, projet Cristal, INRIA Rocquencourt      */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  en Automatique.  All rights reserved.  This file is distributed    */
/*  under the terms of the GNU Library General Public License, with    */
/*  the special exception on linking described in file ../../LICENSE.  */
/*                                                                     */
/***********************************************************************/

/* $Id: cldbm.c,v 1.8 2003/07/08 13:50:31 xleroy Exp $ */

#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <caml/mlvalues.h>
#include <caml/alloc.h>
#include <caml/memory.h>
#include <caml/fail.h>
#include <caml/callback.h>
#include <gdbm.h>

/* Quite close to sys_open_flags, but we need RDWR */
static int dbm_open_flags[] = {
  O_RDONLY, O_WRONLY, O_RDWR, O_CREAT
};

static void raise_dbm (char *errmsg) Noreturn;

static void raise_dbm(char *errmsg)
{
  static value * dbm_exn = NULL;
  if (dbm_exn == NULL)
    dbm_exn = caml_named_value("dbmerror");
  raise_with_string(*dbm_exn, errmsg);
}

static void raise_dbm2(char *errmsg, int linnum, int ernum){
  char msg[256];

  snprintf(msg, sizeof(char)*256, errmsg, linnum, ernum);
  raise_dbm(msg);
}


#define DBM_val(v) *((GDBM_FILE *) &Field(v, 0))

static value alloc_dbm(GDBM_FILE db)
{
  value res = alloc_small(1, Abstract_tag);
  DBM_val(res) = db;
  return res;
}

static GDBM_FILE extract_dbm(value vdb)
{
  if (DBM_val(vdb) == NULL) raise_dbm("DBM has been closed");
  return DBM_val(vdb);
}

static void lock_db(GDBM_FILE db)
{
  int fd;
  off_t pos;

  fd = gdbm_fdesc(db);

  /* Do not work with multi-thred app.! */
  pos = lseek(fd, 0, SEEK_CUR);	/* back up the current position */
  if (pos < 0) {
    raise_dbm2("Unable lock_db: %d errno=%d", __LINE__, errno);
  }

  if (lseek(fd, 0, SEEK_SET) < 0) {
    raise_dbm2("Unable lock_db: %d errno=%d", __LINE__, errno);
  }

  if (lockf(fd, F_LOCK, 0) != 0) {
    raise_dbm2("Unable lock_db: %d errno=%d", __LINE__, errno);
  }

  if (lseek(fd, pos, SEEK_SET) < 0){	/* recover the back-uped position */
    raise_dbm2("Unable lock_db: %d errno=%d", __LINE__, errno);
  }
}

static void unlock_db(GDBM_FILE db)
{
  int fd;
  off_t pos;

  fd = gdbm_fdesc(db);

  /* Do not work with multi-thred app.! */
  pos = lseek(fd, 0, SEEK_CUR);	/* back up the current position */
  if (pos < 0) {
    raise_dbm2("Unable unlock_db: %d errno=%d", __LINE__, errno);
  }

  if (lseek(fd, 0, SEEK_SET) < 0) {
    raise_dbm2("Unable unlock_db: %d errno=%d", __LINE__, errno);
  }

  if (lockf(fd, F_ULOCK, 0) != 0) {
    raise_dbm2("Unable unlock_db: %d errno=%d", __LINE__, errno);
  }

  if (lseek(fd, pos, SEEK_SET) < 0){	/* recover the back-uped position */
    raise_dbm2("Unable unlock_db: %d errno=%d", __LINE__, errno);
  }
}

/* Gdbm.lock_db : t -> unit */
value caml_lock_db(value vdb){
  lock_db(extract_dbm(vdb));
  return Val_unit;
}

/* Gdbm.unlock_db : t -> unit */
value caml_unlock_db(value vdb){
  unlock_db(extract_dbm(vdb));
  return Val_unit;
}

/* Dbm.open : string -> Sys.open_flag list -> int -> t */
value caml_dbm_open(value vfile, value vflags, value vmode) /* ML */
{
  char *file = String_val(vfile);
  int flags = convert_flag_list(vflags, dbm_open_flags);
  int mode = Int_val(vmode);
  GDBM_FILE db;

  db = gdbm_open(file, 512, 
		 flags | GDBM_NOLOCK | GDBM_SYNC,mode, NULL);

  if (db == NULL) 
    raise_dbm("Can't open file");
  else
    return (alloc_dbm(db));
}

/* Dbm.close: t -> unit */
value caml_dbm_close(value vdb)       /* ML */
{
  dbm_close(extract_dbm(vdb));
  DBM_val(vdb) = NULL;
  return Val_unit;
}

/* Dbm.fetch: t -> string -> string */
value caml_dbm_fetch(value vdb, value vkey)  /* ML */
{
  datum key,answer;
  key.dptr = String_val(vkey);
  key.dsize = string_length(vkey);
  answer = gdbm_fetch(extract_dbm(vdb), key);
  if (answer.dptr) {
    value res = alloc_string(answer.dsize);
    memmove (String_val (res), answer.dptr, answer.dsize);
    return res;
  }
  else raise_not_found();
}


value caml_dbm_insert(value vdb, value vkey, value vcontent) /* ML */
{
  int fd, rt;
  datum key, content;
  
  key.dptr = String_val(vkey);
  key.dsize = string_length(vkey);
  content.dptr = String_val(vcontent);
  content.dsize = string_length(vcontent);

  rt = gdbm_store(extract_dbm(vdb), key, content, GDBM_INSERT);

  switch(rt) {
  case 0:
    return Val_unit;
  case 1:                       /* DBM_INSERT and already existing */
    raise_dbm("Entry already exists");
  default:
    raise_dbm("dbm_store failed");
  }
}

value caml_dbm_replace(value vdb, value vkey, value vcontent) /* ML */
{
  int fd, rt;
  datum key, content;
  
  key.dptr = String_val(vkey);
  key.dsize = string_length(vkey);
  content.dptr = String_val(vcontent);
  content.dsize = string_length(vcontent);

  rt = dbm_store(extract_dbm(vdb), key, content, GDBM_REPLACE);

  switch(rt) {
  case 0:
    return Val_unit;
  default:
    raise_dbm("dbm_store failed");
  }
}

value caml_dbm_delete(value vdb, value vkey)         /* ML */
{
  int fd, rt;
  datum key;
  key.dptr = String_val(vkey);
  key.dsize = string_length(vkey);

  rt = dbm_delete(extract_dbm(vdb), key); 

  if (rt < 0)
    raise_dbm("dbm_delete");
  else return Val_unit;
}

value caml_dbm_firstkey(value vdb)            /* ML */
{
  datum key = gdbm_firstkey(extract_dbm(vdb));

  if (key.dptr) {
    value res = alloc_string(key.dsize);
    memmove (String_val (res), key.dptr, key.dsize);
    return res;
  }
  else raise_not_found();
}

value caml_dbm_nextkey(value vdb, value vkey)             /* ML */
{
  datum prev, key;

  prev.dptr = String_val(vkey);
  prev.dsize = string_length(vkey);
  key = gdbm_nextkey(extract_dbm(vdb), prev);

  if (key.dptr) {
    value res = alloc_string(key.dsize);
    memmove (String_val (res), key.dptr, key.dsize);
    return res;
  }
  else raise_not_found();
}
