(**************************************************************************)
(*  Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.    *)
(*  Copyright (C) 2003, 2004, 2005  Yamagata Yoriyuki                     *)
(*                                                                        *)
(*  This program is free software; you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation; either version 2 of the License, or (at  *)
(*  your option) any later version.                                       *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful, but   *)
(*  WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *)
(*  General Public License for more details.                              *)
(**************************************************************************)

(* $Id: personalDict.ml 132 2005-08-07 18:05:59Z yori $ *)

let get_lock () =
  let lock_file = 
    let flags = [Unix.O_WRONLY; Unix.O_CREAT] in
    Unix.openfile Config.personal_dict_lock flags 0o600 in
  Unix.lockf lock_file Unix.F_LOCK 0;
  lock_file

let release_lock lock_file =
  Unix.lockf lock_file Unix.F_ULOCK 0;
  Unix.close lock_file

let db = 
  if not (Sys.file_exists Config.personal_dict_file) then begin
    let file = 
      Dbm.opendbm Config.personal_dict_file [Dbm.Dbm_create] 0o600 in
    Dbm.close file
  end;
  let flags = [Dbm.Dbm_rdonly] in
  Dbm.opendbm Config.personal_dict_file flags 0o600

let () =
  at_exit (fun () -> Dbm.close db)

let get_entry_list key = 
  try
    let scanbuf = Scanf.Scanning.from_string (Dbm.find db key) in
    let s = Scanf.bscanf scanbuf "1,[%S;" (fun s -> s) in
    let rec scan_loop () =
      try Scanf.bscanf scanbuf "%S;" (fun s -> s :: scan_loop ()) with 
	Scanf.Scan_failure _ | Failure _ | End_of_file -> [] in
    s :: scan_loop ()
  with
    Scanf.Scan_failure _ | Failure _ | End_of_file | Not_found -> []

let encode_entry_list entry_list =
  let b = Buffer.create 0 in
  Buffer.add_string b "1,[";
  List.iter (fun entry -> Printf.bprintf b "%S;" entry) entry_list;
  Buffer.add_string b "]";
  Buffer.contents b

let lookup string pos len =
  let key = EucString.sub string ~pos ~len in
  List.map 
    Chasen.deserialize_mrph (get_entry_list key)

let rec lookup_prefix string pos len =
  if len <= 0 then [] else
  (List.map (fun m -> (m, len)) (lookup string pos len)) @
  (lookup_prefix string pos (len - 1))

let add mrph =
  let keyword = Chasen.keyword mrph in
  let entry_list = get_entry_list keyword in
  let string = 
    encode_entry_list (Chasen.serialize_mrph mrph :: entry_list) in
  let lock = get_lock () in
  let db_wr = 
    let flags = [Dbm.Dbm_rdwr] in
    Dbm.opendbm Config.personal_dict_file flags 0o600 in
  Dbm.replace db_wr keyword string;
  Dbm.close db_wr;
  release_lock lock
  
let remove mrph =
  let keyword = Chasen.keyword mrph in
  let mrph_list = 
    List.map Chasen.deserialize_mrph (get_entry_list keyword) in
  let mrph_list = 
    List.filter 
      (fun mrph' -> 
	Chasen.posid mrph' <> Chasen.posid mrph ||
	let mrph = Chasen.mrph_data_of_mrph mrph in
	let mrph' = Chasen.mrph_data_of_mrph mrph' in
	Chasen.headword mrph' <> Chasen.headword mrph)
      mrph_list in
  let lock = get_lock () in
  let db_rw =
    let flags = [Dbm.Dbm_rdwr] in
    Dbm.opendbm Config.personal_dict_file flags 0o600 in
  if mrph_list = [] then
    try Dbm.remove db_rw keyword with Dbm.Dbm_error "dbm_delete" -> ()
  else
    Dbm.replace db_rw keyword 
      (encode_entry_list 
	 (List.map Chasen.serialize_mrph mrph_list));
  Dbm.close db_rw;
  release_lock lock
    
let add_new_word ~kaki ~yomi =
  List.iter add (Chasen.new_word ~kaki ~yomi)
