(**************************************************************************)
(*  Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.    *)
(*  Copyright (C) 2003, 2004  Yamagata Yoriyuki                           *)
(*                                                                        *)
(*  This program is free software; you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation; either version 2 of the License, or (at  *)
(*  your option) any later version.                                       *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful, but   *)
(*  WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *)
(*  General Public License for more details.                              *)
(**************************************************************************)

let rec scm_list = function
    [] -> ScmTypes.Nil
  | v :: r ->
      ScmTypes.cons v (scm_list r)

let rec scm_path = function
    [] -> ScmTypes.Nil
  | mrph :: p ->
      let converted = Conv.surface_form mrph in
      let source = Chasen.keyword mrph in
      let source = Chasen.hiragana source in
      let x = ScmTypes.cons 
		(ScmTypes.String converted) 
		(ScmTypes.String source) 
      in
	ScmTypes.cons x (scm_path p)

let catch_all f x = try f x with _ -> ScmTypes.Nil

let catch_all_2 f x y = try f x y with _ -> ScmTypes.Nil

let rec scm_ptree ptree =
  let scm_best_path = 
    let f _ =
      let path = Parse.best_path ptree in scm_path path
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure f
  in
  let scm_split_best = 
    let f _ =
      let path, pt = Parse.split_best ptree in
	ScmTypes.cons (scm_path path) (scm_ptree pt)
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure f
  in
  let scm_p_cost = 
    let f _ = ScmTypes.Number (ScmDynnum.of_int (Parse.p_cost ptree)) in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure f
  in
  let scm_fix_firstmrph =
    let f arg =
      let word = ScmTypes.string_of_scm_string arg in
      let pt = Parse.fix_firstmrph word ptree in
	scm_ptree pt
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure (ScmEval.simple_cmd_1 f)
  in	
  let scm_fix_firstmrph_len =
    let f arg =
      let len = ScmDynnum.int_of (ScmTypes.number_of_datum arg) in
      let pt = Parse.fix_firstmrph_len len ptree in
	scm_ptree pt
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure (ScmEval.simple_cmd_1 f)
  in	
  let scm_fix_firstmrph_len_hira =
    let f arg =
      let len = ScmDynnum.int_of (ScmTypes.number_of_datum arg) in
      let pt = Parse.fix_firstmrph_len_hira len ptree in
	scm_ptree pt
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure (ScmEval.simple_cmd_1 f)
  in	
  let scm_list_candidates =
    let f arg =
      let len = ScmDynnum.int_of (ScmTypes.number_of_datum arg) in
      let path = Parse.list_candidates len ptree in
	scm_list (List.map (fun s -> ScmTypes.String s) path)
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure (ScmEval.simple_cmd_1 f)
  in
  let scm_list_candidates_hira =
    let f arg =
      let len = ScmDynnum.int_of (ScmTypes.number_of_datum arg) in
      let path = Parse.list_candidates_hira len ptree in
	scm_list (List.map (fun s -> ScmTypes.String s) path)
    in
    let f = catch_all f in
      ScmTypes.ML_EagerProcedure (ScmEval.simple_cmd_1 f)
  in
  let a = Array.init 8 (function 
			    0 -> scm_best_path
			  | 1 -> scm_split_best
			  | 2 -> scm_p_cost
			  | 3 -> scm_fix_firstmrph
			  | 4 -> scm_fix_firstmrph_len
			  | 5 -> scm_fix_firstmrph_len_hira
			  | 6 -> scm_list_candidates
			  | 7 -> scm_list_candidates_hira
			  | _ -> assert false)
  in ScmTypes.Vector a

let scm_parse =
  let f arg =
    let s = ScmTypes.string_of_scm_string arg in
    let pt = Parse.f s in
      scm_ptree pt
  in
    ScmEval.simple_cmd_1 f

let scm_parse_hiragana =
  let f arg =
    let s = ScmTypes.string_of_scm_string arg in
    let pt = Parse.hiragana s in
      scm_ptree pt
  in
    ScmEval.simple_cmd_1 f

let prelude =
  "(define (mana_best_path pt) ((vector-ref pt 0) ()))\
   (define (mana_split_best pt) ((vector-ref pt 1) ()))\
   (define (mana_p_cost pt) ((vector-ref pt 2) ()))\
   (define (mana_fix_firstmrph word pt) ((vector-ref pt 3) word))\
   (define (mana_fix_firstmrph_len len pt) ((vector-ref pt 4) len))\
   (define (mana_fix_firstmrph_len_hira len pt) ((vector-ref pt 5) len))\
   (define (mana_list_candidates len pt) ((vector-ref pt 6) len))\
   (define (mana_list_candidates_hira len pt) ((vector-ref pt 7) len))"

open ScmTypes
open ScmEval
open Lexing

(* Some code comes from Schoca *)
let _ =
  let prompt v =
    print_string (string_of_datum v ^ "\n"); 
    flush stdout;
    v
  in
  let lexbuf = Lexing.from_channel stdin in
  let env = ScmEval.init_environment () in
  let env, _= ScmEval.register_scm_fun env "mana_parse" scm_parse in
  let env, _ = ScmEval.register_scm_fun env 
	      "mana_parse_hiragana" 
	      scm_parse_hiragana
  in
    ignore(ScmSchoca.parse_string env prelude);
    let rec exc_loop () =
      try ignore( ScmSchoca.parse_stream_with_environment env ~prompt lexbuf );
      with 
	| ScmEval.Quit | Exit -> ()
	| e ->
	    (print_string ("Exception: " ^ Printexc.to_string e ^ "\n? ");
	     flush stdout)
    in exc_loop ()
