;;; egg/anthy.el --- ANTHY Support (high level interface) in Egg
;;;                Input Method Architecture

;; Copyright (C) 2002 The Free Software Initiative of Japan

;; Author: NIIBE Yutaka <gniibe@m17n.org>

;; Maintainer: NIIBE Yutaka <gniibe@m17n.org>

;; Keywords: mule, multilingual, input method

;; This file is part of EGG.

;; EGG is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; EGG is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:


;;; Code:

(require 'egg)
(require 'egg-edep)

(defgroup mana nil
  "Anthy interface for Tamago 4."
  :group 'egg)

(setplist 'mana-conversion-backend
	  '(egg-start-conversion          mana-convert
	    egg-get-bunsetsu-source       mana-get-bunsetsu-source
	    egg-get-bunsetsu-converted    mana-get-bunsetsu-converted
	    egg-list-candidates           mana-get-candidates
	    egg-decide-candidate          mana-select-candidate
	    egg-change-bunsetsu-length    mana-resize-segment
	    egg-end-conversion            mana-commit
	    ;;
	    egg-get-source-language       mana-get-source-language
	    egg-get-converted-language    mana-get-converted-language))

(defconst mana-backend-alist '((Japanese ((mana-conversion-backend)))))

(egg-set-finalize-backend '(mana-finalize-backend))

;; <ptid> ::= <integer>

(defvar mana-ptid-counter 0)
(defvar mana-ptid-pool nil)

(defun mana-get-new-ptid ()
  (if mana-ptid-pool
      (let ((ptid (car mana-ptid-pool)))
	(setq mana-ptid-pool (cdr mana-ptid-pool))
	ptid)
    (setq mana-ptid-counter (+ 1 mana-ptid-counter))
    (make-symbol (format "egg-p%d" mana-ptid-counter))))

(defvar mana-proc nil
  "Process of MANA helper agent.")

(defun mana-cmd-prim (val)
  (let ((buffer (process-buffer mana-proc)))
    (if (and (eq (process-status mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (let ((p (point)))
	     (newline)
	     (prin1 val buffer)
	     (newline)
	     (process-send-region mana-proc p (point))
	     (accept-process-output mana-proc)))
      (egg-error "process %s was killed" mana-proc))))

(defun mana-init-proc ()
  (mana-cmd-prim '(define egg-p0 (mana_parse_hiragana ""))))

(defun mana-start-proc ()
  (if (null mana-proc)
      (let ((buf (generate-new-buffer "*MANA*"))
	    (process-connection-type nil)) ; avoid using pty
	(setq mana-proc
	      (start-process "mana" buf "mana"))
	(process-kill-without-query mana-proc)
	(set-process-coding-system mana-proc 'euc-jp-unix 'euc-jp-unix)
	(set-process-sentinel mana-proc 'mana-proc-sentinel)
	(set-marker-insertion-type (process-mark mana-proc) t)
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (buffer-disable-undo))
	(mana-init-proc))))

;;; XXX: Don't kill buffer (for now) so that I can debug this program
(defun mana-proc-sentinel (proc reason)
;  (kill-buffer (process-buffer proc))
  (setq mana-proc nil))

(defun mana-cmd (val)
  (mana-start-proc)
  (mana-cmd-prim val))

(defun mana-eval (val)
  (mana-start-proc)
  (let ((buffer (process-buffer mana-proc)))
    (if (and (eq (process-status mana-proc) 'run)
	      (buffer-live-p buffer))
	 (save-excursion
	   (set-buffer buffer)
	   (goto-char (point-max))
	   (let ((p (point)))
	     (prin1 val buffer)
	     (newline)
	     (process-send-region mana-proc p (point))
	     (accept-process-output mana-proc)
	     (goto-char (point-max))
	     (forward-line -1)
	     (beginning-of-line)
	     (read buffer)))
      (egg-error "process %s was killed" mana-proc))))

(defun mana-release-ptid (ptid)
  (mana-cmd `(set! ,ptid egg-p0)))

(defun mana-parse (yomi)
  (let ((p (mana-get-new-ptid)))
	(mana-cmd `(define ,p (mana_parse_hiragana ,yomi)))
	p))

(defun mana-best-path (ptid) 
  (mana-eval `(mana_best_path ,ptid)))

(defun mana-fix-firstword (s ptid)
  (if s 
      (let ((p (mana-get-new-ptid)))
	(mana-cmd `(define ,p (mana_fix_firstmrph ,s ,ptid)))
	p)
    ptid))

(defun mana-list-candidates (len ptid)
  (mana-eval `(mana_list_candidates_hira ,len ,ptid)))

;;
;; <mana-bunsetsu> ::=
;;  [ <ptid> <source> <converted> <candidates> <candidate-pos> ]
(defsubst mana-make-bunsetsu (ptid source converted)
  (egg-bunsetsu-create
   'mana-conversion-backend
   (vector ptid source converted nil 0)))

(defsubst manabunsetsu-get-ptid (b)
  (aref (egg-bunsetsu-get-info b) 0))
(defsubst manabunsetsu-get-source (b)
  (aref (egg-bunsetsu-get-info b) 1))
(defsubst manabunsetsu-get-converted (b)
  (aref (egg-bunsetsu-get-info b) 2))
(defsubst manabunsetsu-get-candidates (b)
  (aref (egg-bunsetsu-get-info b) 3))
(defsubst manabunsetsu-set-candidates (b z)
  (aset (egg-bunsetsu-get-info b) 3 z))
(defsubst manabunsetsu-get-candidate-pos (b)
  (aref (egg-bunsetsu-get-info b) 4))
(defsubst manabunsetsu-set-candidate-pos (b zp)
  (aset (egg-bunsetsu-get-info b) 4 zp))

(defun mana-get-bunsetsu-source (b)
  (manabunsetsu-get-source b))

(defun mana-get-bunsetsu-converted (b)
  (let ((cands (manabunsetsu-get-candidates b)))
    (if cands
	(nth (manabunsetsu-get-candidate-pos b) cands)
      (manabunsetsu-get-converted b))))

(defun mana-get-source-language (b) 'Japanese)
(defun mana-get-converted-language (b) 'Japanese)

;;
;; Returns list of bunsetsu
;;

(defun mana-make-bunsetsu-list (ptid path)
  (if path
      (let* ((node (car path))
	     (converted (car node))
	     (source (cdr node))
	     (bunsetsu (mana-make-bunsetsu ptid source converted)))
	(cons bunsetsu
	      (mana-make-bunsetsu-list (mana-fix-firstword converted ptid) 
				       (cdr path))))
    nil))

(defun mana-convert (backend yomi &optional context)
  "Convert YOMI string to kanji, and enter conversion mode.
Return the list of bunsetsu."
  (set-text-properties 0 (length yomi) nil yomi)
  (let*  ((ptid (mana-parse yomi))
	  (conv-list (mana-best-path ptid)))
    (mana-make-bunsetsu-list ptid conv-list)))

;;
;;
;;

(defun mana-release-bunsetsu-list (bunsetsu-list)
  (mapcar 
   '(lambda (bunsetsu) 
      (mana-release-ptid (manabunsetsu-get-ptid bunsetsu)))
   bunsetsu-list))

(defun mana-commit (bunsetsu-list abort)
  (mana-release-bunsetsu-list bunsetsu-list))

;;
;; Returns ( <pos> <candidates> )
;;
(defun mana-get-candidates (bunsetsu-list prev-bunsetsu next-bunsetsu major)
  (let ((bunsetsu (car bunsetsu-list)))
    (if (manabunsetsu-get-candidates bunsetsu)
	(cons (manabunsetsu-get-candidate-pos bunsetsu)
	      (manabunsetsu-get-candidates bunsetsu))
      (let* ((ptid (manabunsetsu-get-ptid bunsetsu))
	     (len (length (manabunsetsu-get-source bunsetsu)))
	     (cands (mana-list-candidates len ptid)))
	(cons (manabunsetsu-set-candidate-pos bunsetsu 0)
	      (manabunsetsu-set-candidates bunsetsu cands))))))

;; Returns list of list of bunsetsu
(defun mana-select-candidate (bunsetsu-list candidate-pos prev-b next-b)
  (let* ((bunsetsu (car bunsetsu-list))
	 (ptid (manabunsetsu-get-ptid bunsetsu))
	 (candidate-list (manabunsetsu-get-candidates bunsetsu))
	 (candidate (nth candidate-pos candidate-list))
	 (next-ptid (mana-fix-firstword candidate ptid))
	 (path (mana-best-path next-ptid))
	 (next-b-list (mana-make-bunsetsu-list next-ptid path)))
    (mana-release-bunsetsu-list (cdr bunsetsu-list))
    (manabunsetsu-set-candidate-pos bunsetsu candidate-pos)
    (list (list bunsetsu) nil next-b-list)))

(defun mana-get-yomi (len bunsetsu-list)
  (if (or (< len 0) (not bunsetsu-list)) ""
    (let* ((bunsetsu (car bunsetsu-list))
	   (source (manabunsetsu-get-source bunsetsu))
	   (source-len (length source)))
      (if (<= len source-len)
	  (substring source 0 len)
	(concat source 
	      (mana-get-yomi (- len source-len) (cdr bunsetsu-list)))))))

;; Returns list of list of bunsetsu
(defun mana-resize-segment (bunsetsu-list prev-b next-b len major)
  (let* ((bunsetsu (car bunsetsu-list))
	 (ptid (manabunsetsu-get-ptid bunsetsu))
	 (cands (mana-list-candidates len ptid))
	 (conv (nth 0 cands))
	 (source (mana-get-yomi len bunsetsu-list))
	 (next-ptid (mana-fix-firstword conv ptid))
	 (bunsetsu (mana-make-bunsetsu ptid source conv))
	 (path (mana-best-path next-ptid)))
    (list (list bunsetsu) nil (mana-make-bunsetsu-list next-ptid path))))

(defun mana-finalize-backend ()
  (if mana-proc
      (progn
	(delete-process mana-proc)
	(setq mana-proc nil))))

;;; setup

(run-hooks 'mana-load-hook)

;;;###autoload
(defun egg-activate-mana (&rest arg)
  "Activate MANA backend of Tamago 4."
  (apply 'egg-mode (append arg mana-backend-alist)))

;;; egg-mana.el ends here.
