/*
SRBrowserController.h

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import <Cocoa/Cocoa.h>
#import <Carbon/Carbon.h>
#import <WebKit/WebKit.h>
#import "WebInspector.h"
#import <HMDTAppKit/HMDTAppKit.h>
#import <HMDTWebKit/HMDTWebKit.h>

#import "SRBrowserControllerProxy.h"

// Open action type
typedef enum _SROpenActionType {
    SROpenAction, 
    SROpenInNewTabAction, 
    SROpenInNewBackgroundTabAction, 
    SROpenInNewWindowAction, 
    SROpenInNewBackgroundWindowAction, 
    SROpenOptionAction, 
} SROpenActionType;

typedef enum _SRTabStyleType {
    SRTabStylePageDock = 0, 
    SRTabStyleTab, 
} SRTabStyleType;

typedef enum _SRPageDockStyleType {
    SRPageDockStyleHorizontal = 0, 
    SRPageDockStyleVertical, 
} SRPageDockStyleType;

// Nofitication
extern NSString*    SRBrowserDidCommitLoad;
extern NSString*    SRBrowserDidReceiveServerRedirect;
extern NSString*    SRBrowserTitleReceived;
extern NSString*    SRBrowserProgressStart;
extern NSString*    SRBrowserProgressFinish;
extern NSString*    SRBrowserDidBecomeMain;
extern NSString*    SRBrowserDidResignMain;
extern NSString*    SRBrowserSelectedTabChanged;

// Toolbar identifier
extern NSString*    SRBrowserToolbarIdentifier;
extern NSString*    SRBrowserToolbarLabelTable;

// Toolbar items
extern NSString*    SRBrowserGoBackItem;
extern NSString*    SRBrowserGoForwardItem;
extern NSString*    SRBrowserReloadPageItem;
extern NSString*    SRBrowserStopLoadingItem;
extern NSString*    SRBrowserGoHomeItem;
extern NSString*    SRBrowserURLItem;
extern NSString*    SRBrowserSearchItem;
extern NSString*    SRBrowserNewTabItem;
extern NSString*    SRBrowserCloseTabItem;
extern NSString*    SRBrowserShelfItem;
extern NSString*    SRBrowserFullScreenItem;
extern NSString*    SRBrowserPrintPageItem;
extern NSString*    SRBrowserAddBookmarkItem;
extern NSString*    SRBrowserBiggerTextItem;
extern NSString*    SRBrowserSmallerTextItem;
extern NSString*    SRBrowserBookmarkPanelItem;
extern NSString*    SRBrowserHistoryPanelItem;
extern NSString*    SRBrowserPageInfoPanelItem;
extern NSString*    SRBrowserFindPanelItem;
extern NSString*    SRBrowserDownloadPanelItem;
extern NSString*    SRBrowserRSSPanelItem;

// Toolbar icons
extern NSDictionary*    SRAquaIconImageDict;
extern NSDictionary*    SRMetalIconImageDict;

@class SRBookmark;

@class SRBrowserDocument;
@class SRPageController;
@class SRURLCompleteDataSource;
@class SRSearchFieldController;
@class SRTabExposeController;
@class SRFullScreenController;
@class SRFullScreenImageController;

@class SRBookmarkBar;
@class SRStatusBar;
@class SRSearchField;

@interface SRBrowserController : NSWindowController <SRBrowserController>
{
    NSMutableArray*                 _pageControllers;
    SRSearchFieldController*        _searchFieldController;
    SRTabExposeController*          _tabExposeController;
    
    HMTabViewItem*                  _willSelectTabViewItem;
    
    SRURLCompleteDataSource*        _URLDataSource;
    SRURLCompleteDataSource*        _locationURLDataSource;
    
    NSButton*                       _pageDockButton;
    NSButton*                       _tabExposeButton;
    NSButton*                       _feedButton;
    NSButton*                       _IDNButton;
    NSButton*                       _lockedButton;
    NSButton*                       _cookieButton;
    
    BOOL                            _isResizedByJavaScript;
    
    SRFullScreenController*         _fullScreenController;
    HMWindow*                       _fullScreenWindow;
    BOOL                            _fullScreenTabItemViewShown;
    
    SRFullScreenImageController*    _fullScreenImageController;
    
    IBOutlet SRBookmarkBar*         _bookmarkBar;
    IBOutlet HMTabView*             _tabView;
    IBOutlet SRStatusBar*           _statusBar;
    
    IBOutlet NSObjectController*    _browserController;
    IBOutlet NSMutableDictionary*   _browserContent;
    
    IBOutlet id                     _addBookmarkPanel;
    IBOutlet id                     _addBookmarkTextField;
    IBOutlet id                     _addBookmarkPopup;
    NSString*                       _addBookmarkURLString;
    
    IBOutlet id                     _bookmarkAllPanel;
    IBOutlet id                     _bookmarkAllPopup;
    
    IBOutlet id                     _newFolderPanel;
    IBOutlet id                     _newFolderTextField;
    
    IBOutlet id                     _addToBarPanel;
    IBOutlet id                     _addToBarTextField;
    
    IBOutlet id                     _locationPanel;
    IBOutlet id                     _locationURLView;
    
    IBOutlet id                     _searchPanel;
    IBOutlet id                     _searchFieldView;
}

// Initialize
+ (NSArray*)browserControllers;
+ (SRBrowserController*)mainBrowserController;
+ (BOOL)isFullScreen;

// Page controller
- (NSArray*)pageControllers;
- (SRPageController*)pageControllerAtIndex:(int)index;
- (SRPageController*)pageControllerForView:(NSView*)view;

// Search field controller
- (SRSearchFieldController*)searchFieldController;

// Tab management
- (HMTabView*)tabView;
- (SRPageController*)addNewTabWithLabel:(NSString*)label 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        select:(BOOL)select;
- (NSArray*)addNewTabsWithLabels:(NSArray*)labels 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        selectLast:(BOOL)selectLast;
- (SRPageController*)insertNewTabWithLabel:(NSString*)label 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        atIndex:(unsigned int)index 
        select:(BOOL)select;
- (void)removeTabAtIndex:(unsigned int)index;
- (void)removeTabAtIndexes:(NSIndexSet*)indexSet;
- (void)selectLeftTab;
- (void)selectRightTab;

- (BOOL)isPageDockVisible;
- (void)setPageDockVisible:(BOOL)isVisible;

// Web view and toolbar items
- (SRPageController*)selectedPageController;
- (HMImageComboBox*)URLComboBox;
- (HMImageComboBox*)URLComboBoxOnSheet;
- (SRSearchField*)searchField;
- (SRSearchField*)searchFieldOnSheet;

// Key value coding
- (id)browserContent;

// Opening page with request
- (SRPageController*)openRequest:(NSURLRequest*)request;
- (SRPageController*)openRequest:(NSURLRequest*)request atIndex:(int)index;
- (id)openRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        withOpenAction:(SROpenActionType)openAction;
- (SRPageController*)openInNewTabRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName;
- (SRPageController*)openInNewTabRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        select:(BOOL)select;
- (NSArray*)openInNewTabRequests:(NSArray*)requests 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        select:(BOOL)select;
- (SRBrowserDocument*)openInNewWindowRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName;
- (SRBrowserDocument*)openInNewBackgroundWindowRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName;

// Opening page with URL string
- (NSURLRequest*)createRequestWithURL:(NSURL*)url;
- (NSURLRequest*)createRequestWithURLString:(NSString*)URLString;
- (SRPageController*)openURLString:(NSString*)URLString;
- (SRPageController*)openURLString:(NSString*)URLString atIndex:(int)index;
- (id)openURLString:(NSString*)URLString withOpenAction:(SROpenActionType)openAction;
- (SRPageController*)openInNewTabURLString:(NSString*)URLString;
- (SRPageController*)openInNewTabURLString:(NSString*)URLString select:(BOOL)select;
- (SRBrowserDocument*)openInNewWindowURLString:(NSString*)URLString;
- (SRBrowserDocument*)openInNewBackgroundWindowURLString:(NSString*)URLString;
- (NSArray*)openInNewTabsURLStrings:(NSArray*)URLStrings select:(BOOL)select;

// Opening page with URL
- (SRPageController*)openURL:(NSURL*)url;
- (SRPageController*)openURL:(NSURL*)url atIndex:(int)index;
- (id)openURL:(NSURL*)url withOpenAction:(SROpenActionType)openAction;
- (SRPageController*)openInNewTabURL:(NSURL*)url;
- (SRPageController*)openInNewTabURL:(NSURL*)url select:(BOOL)select;
- (SRBrowserDocument*)openInNewWindowURL:(NSURL*)url;
- (SRBrowserDocument*)openInNewBackgroundWindowURL:(NSURL*)url;

// Opening page with bookmark
- (SRPageController*)openBookmark:(SRBookmark*)bookmark;
- (SRPageController*)openBookmark:(SRBookmark*)bookmark atIndex:(int)index;
- (id)openBookmark:(SRBookmark*)bookmark withOpenAction:(SROpenActionType)openAction;
- (SRPageController*)openInNewTabBookmark:(SRBookmark*)bookmark;
- (SRPageController*)openInNewTabBookmark:(SRBookmark*)bookmark select:(BOOL)select;
- (SRBrowserDocument*)openInNewWindowBookmark:(SRBookmark*)bookmark;
- (SRBrowserDocument*)openInNewBackgroundWindowBookmark:(SRBookmark*)bookmark;
- (NSArray*)openInTabsBookmark:(SRBookmark*)bookmark;
- (NSArray*)openInTabsBookmarks:(NSArray*)bookmarks;

// Opening page with history
- (SRPageController*)openHistory:(WebHistoryItem*)historyItem;
- (SRPageController*)openHistory:(WebHistoryItem*)historyItem atIndex:(int)index;
- (id)openHistory:(WebHistoryItem*)historyItem withOpenAction:(SROpenActionType)openAction;
- (SRPageController*)openInNewTabHistory:(WebHistoryItem*)historyItem;
- (SRPageController*)openInNewTabHistory:(WebHistoryItem*)historyItem select:(BOOL)select;
- (SRBrowserDocument*)openInNewWindowHistory:(WebHistoryItem*)historyItem;
- (SRBrowserDocument*)openInNewBackgroundWindowHistory:(WebHistoryItem*)historyItem;
- (NSArray*)openInTabsHistory:(NSArray*)historyItems;

// Opening shelf
- (SRPageController*)openShelf:(NSString*)shelf;

// Opening URL
- (WebView*)reloadURLIfExists:(NSURL*)url;

// Close page
- (void)closePageController:(SRPageController*)pageController;

// Toolbar icon
+ (NSArray*)toolbarItemIdentifiers;
+ (NSArray*)defaultToolbarItemIdentifiers;
+ (void)updateToolbarIcons;

// Outlet accessors
- (id)addBookmarkPanel;
- (id)addBookmarkTextField;
- (id)addBookmarkPopup;
- (id)bookmarkAllPanel;
- (id)bookmarkAllPopup;
- (id)newFolderPanel;
- (id)newFolderTextField;
- (id)addToBarPanel;
- (id)addToBarTextField;
- (id)locationPanel;
- (id)locationURLView;
- (id)searchPanel;
- (id)searchFieldView;

// Appearnace
- (BOOL)isBookmarkBarVisible;
- (void)setBookmarkBarVisible:(BOOL)isVisible display:(BOOL)display;
- (BOOL)isStatusBarVisible;
- (void)setStatusBarVisible:(BOOL)isVisible display:(BOOL)display;
- (BOOL)isTabVisible;
- (void)setTabVisible:(BOOL)isVisible display:(BOOL)display;
- (BOOL)isResizedByJavaScript;
- (void)setResizedByJavaScript:(BOOL)flag;
- (NSRect)tabViewFrame;

- (SRTabStyleType)tabStyle;
- (void)setTabStyle:(SRTabStyleType)tabStyle;
- (SRPageDockStyleType)pageDocSytle;
- (void)setPageDockStyle:(SRPageDockStyleType)pageDockStyle;

// Full screen
- (void)switchToFullScreenWithMenu:(BOOL)isMenuVisible;
- (void)backToNormalMode;
- (BOOL)isFullScreenMode;
- (HMWindow*)fullScreenWindow;

@end

@interface SRBrowserController (action)

// View menu action
- (void)reloadPageAction:(id)sender;
- (void)stopLoadingAction:(id)sender;
- (void)smallerTextAction:(id)sender;
- (void)biggerTextAction:(id)sender;
- (void)backToNormalModeAction:(id)sender;

// History menu action
- (void)goBackAction:(id)sender;
- (void)goForwardAction:(id)sender;
- (void)goHomeAction:(id)sender;
- (void)openHistoryItemAction:(id)sender;

@end

@interface SRBrowserController (delegate)
@end

@interface SRURLToolbarItem : NSToolbarItem
@end

// Utility
SROpenActionType SROpenActionTypeFromModifierFlags(
        unsigned modifierFlags, 
        BOOL enableTabbedBrowsing, 
        BOOL selectNewTabs);

NSWindow* SRGetFrontBrowserWindow();

NSString* SRErrorURLWithWebView(
        WebView* webView);
