/*
HMGLImage.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMGLImage.h"

@interface HMGLImage (private)
// Drawing
- (void)_generateTexture;
- (void)_deleteTexture;
@end

@implementation HMGLImage

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)initWithBitmapImageRep:(NSBitmapImageRep*)bitmapRep
{
	self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
	_bitmapRep = [bitmapRep retain];
	_textureId = 0;
    _red = 1.0f;
    _blue = 1.0f;
    _green = 1.0f;
    
    return self;
}

- (void) dealloc
{
	[self _deleteTexture];
	[_bitmapRep release];
    
	[super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Accessors --
//--------------------------------------------------------------//

- (NSSize)size
{
    return [_bitmapRep size];
}

- (NSBitmapImageRep*)bitmapImageRep
{
    return _bitmapRep;
}

- (void)setRed:(float)red blue:(float)blue greeen:(float)green
{
    _red = red;
    _blue = blue;
    _green = green;
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)_generateTexture
{
    // Delete current texture
    [self _deleteTexture];
    
    // Get image size
    NSSize  size;
    size = [_bitmapRep size];
    
    // Create texture
    
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glGenTextures(1, &_textureId);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, _textureId);
    glTexImage2D(
            GL_TEXTURE_RECTANGLE_EXT, 
            0, 
            [_bitmapRep hasAlpha] ? GL_RGBA8 : GL_RGB8, 
            [_bitmapRep pixelsWide], 
            [_bitmapRep pixelsHigh], 
            0, 
            [_bitmapRep hasAlpha] ? GL_RGBA : GL_RGB, 
            GL_UNSIGNED_BYTE, 
            [_bitmapRep bitmapData]);
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
}

- (void)_deleteTexture
{
    if (!_textureId) {
        return;
    }
    
    // Delete texture
    glDeleteTextures(1, &_textureId);
    _textureId = 0;
}

- (void)drawAtPoint:(NSPoint)point fraction:(float)alpha
{
    // Generate texture
    if (!_textureId) {
        [self _generateTexture];
    }
    
    // Draw with bounds
    NSSize  size;
    size = [_bitmapRep size];
    [self drawInRect:NSMakeRect(point.x, point.y, size.width, size.height) fraction:alpha];
}

- (void)drawInRect:(NSRect)rect fraction:(float)alpha
{
    // Set alpha
    if (alpha < 0) alpha = 0;
    if (alpha > 1.0f) alpha = 1.0f;
    
    // Set color
    if (alpha < 1.0f) {
        glColor4f(alpha, alpha, alpha, alpha);
    }
    else {
        glColor4f(_red, _blue, _green, 1.0f);
    }
    
    // Generate texture
    if (!_textureId) {
        [self _generateTexture];
    }
    
    // Get size
    NSSize  size;
    size = [_bitmapRep size];
    
    // Draw texture
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, _textureId);
    glBegin(GL_QUADS);
        // Upper left
        glTexCoord2f(0.0f, 0.0f);
        glVertex2f(rect.origin.x, rect.origin.y);
        
        // Lower left
        glTexCoord2f(0.0f, size.height);
        glVertex2f(rect.origin.x, rect.origin.y + rect.size.height);
        
        // Upper right
        glTexCoord2f(size.width, size.height);
        glVertex2f(rect.origin.x + rect.size.width, rect.origin.y + rect.size.height);
        
        // Lower right
        glTexCoord2f(size.width, 0.0f);
        glVertex2f(rect.origin.x + rect.size.width, rect.origin.y);
    glEnd();
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
}

@end
