/*
SRURLComboBox.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRURLComboBox.h"
#import "SRURLComboBoxCell.h"
#import "SRURLAutoComplete.h"

@interface NSKeyBindingManager
+ (id)sharedKeyBindingManager;
@end

#pragma mark -

@implementation SRURLComboBox

+ (Class)cellClass
{
    // Use SRURLComboBoxCell class
    return [SRURLComboBoxCell class];
}

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    [self setUsesDataSource:YES];
    
    // Create data source
    _completeDataSource = [SRURLAutoCompleteDataSource URLAutoCompleteDataSource];
    [self setDataSource:_completeDataSource];
    [_completeDataSource setComboBox:self];
    
    _buttons = [[NSMutableArray array] retain];
    
    return self;
}

- (void)dealloc
{
    [_completeDataSource setComboBox:nil];
	[_completeDataSource invalidate];
    [_buttons release];
    
	[super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Working with image --
//--------------------------------------------------------------//

- (void)setImage:(NSImage*)image
{
    // Set image to cell
    [[self cell] setImage:image];
}

- (NSImage*)image
{
    // Get image from cell
    return [[self cell] image];
}

//--------------------------------------------------------------//
#pragma mark -- Working wiht buttons --
//--------------------------------------------------------------//

- (NSArray*)buttons
{
    return _buttons;
}

- (NSButton*)addButtonWithSize:(NSSize)size
{
    // Get button frame;
    NSRect  buttonFrame;
    buttonFrame = [self buttonFrame];
    if (NSIsEmptyRect(buttonFrame)) {
        buttonFrame.origin.x = [self frame].origin.x + [self frame].size.width - 24;
    }
    
    // Create button
    NSRect      frame;
    NSButton*   button;
    frame.origin.x = buttonFrame.origin.x - size.width - 1;
    frame.origin.y = ([self frame].size.height - size.height) / 2;
    frame.size = size;
    button = [[NSButton alloc] initWithFrame:frame];
    [button autorelease];
    [button setAutoresizingMask:NSViewMinXMargin | NSViewMinYMargin];
    
    // Add button
    [self addSubview:button];
    [_buttons addObject:button];
    
    return button;
}

- (NSButton*)buttonWithTag:(int)tag
{
    // Get button
    NSEnumerator*   enumerator;
    NSButton*       button;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        if ([button tag] == tag) {
            return button;
        }
    }
    
    return nil;
}

- (void)removeButton:(NSButton*)button
{
    // Remove button
    [button removeFromSuperview];
    [_buttons removeObject:button];
}

- (NSRect)buttonFrame
{
    // Get union rect of existed buttons
    NSRect          unionRect = NSZeroRect;
    NSEnumerator*   enumerator;
    NSButton*       button;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        unionRect = NSUnionRect(unionRect, [button frame]);
    }
    
    return unionRect;
}

- (NSString*)absoluteURLStringOfCompletedString:(NSString*)completedString
{
    return [[self dataSource] 
            absoluteURLStringOfCompletedString:completedString];
}

#if 0
- (void)selectItemAtIndex:(int)index
{
    // Get old string
    NSString*   oldString;
    oldString = [self stringValue];
    
    [super selectItemAtIndex:index];
    
    // Get new string
    NSString*   newString;
    newString = [self stringValue];
    
    // Remove prefix not entered by user
    NSRange     range;
    range = [newString rangeOfString:oldString];
    if (range.location != NSNotFound && range.location < [newString length]) {
        NSString*   completedString;
        completedString = [newString substringFromIndex:range.location];
        [self setStringValue:completedString];
    }
}
#endif

- (void)mouseDown:(NSEvent *)event
{
	_mouseDownPoint=_mouseDownPoint_nowhere;
	//check mouse down point is in left image frame
    if ([[self cell] shouldImageTrackMouse:event inRect:[self bounds] ofView:self]) {
		_mouseDownPoint=_mouseDownPoint_leftImage;
		//select all
		[self selectText:self];
    }else{
		[super mouseDown:event];
	}
}

- (void)mouseDragged:(NSEvent *)event
{
    // Drag for image
	if(_mouseDownPoint==_mouseDownPoint_leftImage){
		[[self cell] imageTrackMouse:event inRect:[self bounds] ofView:self];
		_mouseDownPoint=_mouseDownPoint_nowhere;
	}else{
		[super mouseDragged:event];
	}
}

@end
