/*
SRRSSView.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRDefaultsKey.h"
#import "SRMainWindowController.h"
#import "SRPreferencesController.h"
#import "SRContextMenu.h"

#import "SRRSSView.h"

#import "WebKitEx.h"
#import "SRDOMFilter.h"
#import "SRUtil.h"

@implementation SRRSSView

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Load nib
    if (![NSBundle loadNibNamed:@"RSSView" owner:self]) {
        // Error
        NSLog(@"Can't load RSSView.nib");
        return nil;
    }
    
    // Configure itself
    [self setAutoresizingMask:NSViewWidthSizable | NSViewHeightSizable];
    
    return self;
}

- (void)dealloc
{
    [_document release];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- WebView accessors --
//--------------------------------------------------------------//

- (WebView*)webView
{
    return _webView;
}

- (NSString*)title
{
    return [[[[_webView mainFrame] dataSource] representation] title];
}

//--------------------------------------------------------------//
#pragma mark -- Display RSS --
//--------------------------------------------------------------//

- (void)updateRSS
{
    [self setFrame:[[self superview] frame]];
    
    // Add subviews
    if (![_view superview]) {
        [_view setFrame:[self frame]];
        [self addSubview:_view];
    }
    
    // Get RSS data
    NSData* data;
    data = [_dataSource data];
    
    // Create XML document
    [_document release];
    _document = [[NSClassFromString(@"SRXMLDocument") alloc] initWithData:data options:0 error:NULL];
    if (!_document) {
        return;
    }
    
    // Get XSL
    NSString*   XSLFilePath;
    XSLFilePath = [[NSBundle mainBundle] pathForResource:@"rss" ofType:@"xsl"];
    
    // Apply XSL
    id          result;
    NSError*    error;
    result = [_document objectByApplyingXSLTAtURL:[NSURL fileURLWithPath:XSLFilePath] 
            arguments:nil 
            error:&error];
    if (!result && error) {
        // Error
        NSLog([error localizedDescription]);
        return;
    }
    
    // Load HTML
    // For SRXMLDocument
    if ([result respondsToSelector:@selector(XMLData)]) {
        [[_webView mainFrame] loadData:[result XMLData] MIMEType:@"text/html" textEncodingName:@"utf-8" baseURL:nil];
    }
    // For NSData
    else {
        [[_webView mainFrame] loadData:result MIMEType:@"plain/text" textEncodingName:@"utf-8" baseURL:nil];
    }
}

//--------------------------------------------------------------//
#pragma mark -- Actions --
//--------------------------------------------------------------//

- (void)changeXSLTAction:(id)sender
{
    [self updateRSS];
}

//--------------------------------------------------------------//
#pragma mark -- NSView override --
//--------------------------------------------------------------//

- (void)drawRect:(NSRect)rect
{
    //NSDrawWindowBackground(rect);
}

//--------------------------------------------------------------//
#pragma mark -- WebFrameLoadDelegate protocol --
//--------------------------------------------------------------//

- (void)webView:(WebView*)webView didFinishLoadForFrame:(WebFrame*)frame
{
}

//--------------------------------------------------------------//
#pragma mark -- WebPolicyDelegate protocol --
//--------------------------------------------------------------//

- (void)webView:(WebView*)webView 
        decidePolicyForNavigationAction:(NSDictionary*)actionInformation 
        request:(NSURLRequest*)request 
        frame:(WebFrame*)frame 
        decisionListener:(id<WebPolicyDecisionListener>)listener
{
    // Get URL for request
    NSURL*      URL;
    URL = [request URL];
    
    // For about: and applewebdata:
    if ([[URL scheme] isEqualToString:@"about"] || 
        [[URL scheme] isEqualToString:@"applewebdata"])
    {
        // Use it
        [listener use];
        return;
    }
        
    // Open in parent web view
    [listener ignore];
    
    id  windowController;
    windowController = [[self window] windowController];
    if ([windowController respondsToSelector:
            @selector(webView:decidePolicyForNavigationAction:request:frame:decisionListener:)])
    {
        [(SRMainWindowController*)windowController webView:webView 
                decidePolicyForNavigationAction:actionInformation 
                request:request 
                frame:frame 
                decisionListener:nil];
    }
}

//--------------------------------------------------------------//
#pragma mark -- WebUIDelegate protocol --
//--------------------------------------------------------------//

- (NSArray*)webView:(WebView*)webView 
        contextMenuItemsForElement:(NSDictionary*)element 
        defaultMenuItems:(NSArray*)defaultMenuItems
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get element info
    NSURL*  linkURL;
    NSURL*  imageURL;
    BOOL    isSelected;
    
    linkURL = [element objectForKey:WebElementLinkURLKey];
    imageURL = [element objectForKey:WebElementImageURLKey];
    isSelected = [[element objectForKey:WebElementIsSelectedKey] boolValue];
    
    // Get modifier key flag
    unsigned int    modifierFlags;
    unsigned int    cmdFlag, optionFlag, shiftFlag;
    modifierFlags = [[NSApp currentEvent] modifierFlags];
    cmdFlag = modifierFlags & NSCommandKeyMask;
    optionFlag = modifierFlags & NSAlternateKeyMask;
    shiftFlag = modifierFlags & NSShiftKeyMask;
    
    // Check tab browsing
    BOOL    enableTabbedBrowsing, selectNewTabs;
    enableTabbedBrowsing = [defaults boolForKey:SRTabEnableTabbedBrowsing];
    selectNewTabs = [defaults boolForKey:SRTabSelectNewTabs];
    
    // Create array
    NSMutableArray* items;
    items = [NSMutableArray array];
    
    // Get web view context menu
    NSMenu*     menu;
    NSMenuItem* menuItem;
    menu = [SRContextMenu webViewContextMenu];
    
    NSMutableArray* allowedDefaultMenuItems;
    allowedDefaultMenuItems = [NSMutableArray array];
    
    // Check default menu items
    NSEnumerator*   enumerator;
    enumerator = [defaultMenuItems objectEnumerator];
    while (menuItem = [enumerator nextObject]) {
        int tag;
        tag = [menuItem tag];
        
        // For Spotlight, Google Search, and looking up in dictionary
        if (tag >= 1000) {
            // Do not include them
            continue;
        }
        
        // For separator
        if ([menuItem isSeparatorItem]) {
            if ([allowedDefaultMenuItems count] == 0 || 
                [[allowedDefaultMenuItems lastObject] isSeparatorItem])
            {
                continue;
            }
        }
        [allowedDefaultMenuItems addObject:menuItem];
        
        // For 'Open the link in a new window'
        if (tag == WebMenuItemTagOpenLinkInNewWindow) {
            // Change title, action and target
            [menuItem setAction:@selector(openLinkInNewWindowAction:)];
            [menuItem setTarget:nil];
            [menuItem setRepresentedObject:linkURL];
            continue;
        }
        
        // For 'Open the image in a new window'
        if (tag == WebMenuItemTagOpenImageInNewWindow) {
            // Change action and target
            [menuItem setAction:@selector(openLinkInNewWindowAction:)];
            [menuItem setTarget:nil];
            [menuItem setRepresentedObject:imageURL];
            continue;
        }
    }
    
    // Get selected links
    DOMRange*       range;
    DOMNode*        startContainer = nil;
    NSMutableArray* linkURLStrings;
    range = [webView selectedDOMRange];
    startContainer = [range startContainer];
    linkURLStrings = [NSMutableArray array];
    if (startContainer) {
        SRDOMRangeFilter*   filter;
        DOMNodeIterator*    iterator;
        DOMNode*            node;
        filter = [[SRDOMRangeFilter alloc] initWithDOMRange:range nodeName:@"A"];
        iterator = [[[webView mainFrame] DOMDocument] createNodeIterator:startContainer :DOM_SHOW_ALL :filter :NO];
        while (node = [iterator nextNode]) {
            // Get URL string
            NSString*   URLString = nil;
            if ([node respondsToSelector:@selector(href)]) {
                URLString = [(DOMHTMLLinkElement*)node href];
            }
            if (URLString) {
                // Check scheme
                NSString*   scheme;
                scheme = [[NSURL URLWithString:URLString] scheme];
                if (![scheme isEqualToString:@"javascript"]) {
                    [linkURLStrings addObject:URLString];
                }
            }
        }
        [filter release];
    }
    
    // Link is contained
    if (linkURL) {
        // Copy default menu item
        [items addObjectsFromArray:allowedDefaultMenuItems];
        
        // Add 'Add Link to Bookmarks'
        menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRAddLinkToBokkmarksTag target:nil];
        [menuItem setRepresentedObject:element];
        [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenLinkInNewWindow];
        
        // Check scheme
        if (![[linkURL scheme] isEqualToString:@"javascript"]) {
            // For tabbed browsing
            if (enableTabbedBrowsing) {
                if ((selectNewTabs && !shiftFlag) || 
                    (!selectNewTabs && shiftFlag))
                {
                    // Add 'Open Link in New Tab'
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenLinkInNewTabTag target:nil];
                    [menuItem setRepresentedObject:linkURL];
                    [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenLinkInNewWindow];
                    
                    // Add 'Open All Links in New Tabs'
                    if ([linkURLStrings count] > 1) {
                        menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenAllLinksInNewTabsTag target:nil];
                        [menuItem setTitle:[NSString stringWithFormat:[menuItem title], [linkURLStrings count]]];
                        [menuItem setRepresentedObject:linkURLStrings];
                        [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:SROpenLinkInNewTabTag];
                    }
                }
                else {
                    // Add 'Open Link in New Background Window'
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenLinkInNewBackgroundWindowTag target:nil];
                    [menuItem setRepresentedObject:linkURL];
                    [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenLinkInNewWindow];
                    
                    // Add 'Open Link in New Background Tab'
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenLinkInNewBackgroundTabTag target:nil];
                    [menuItem setRepresentedObject:linkURL];
                    [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:SROpenLinkInNewBackgroundWindowTag];
                    
                    // Add 'Open All Links in New Background Tabs'
                    if ([linkURLStrings count] > 1) {
                        menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenAllLinksInNewBackgroundTabsTag target:nil];
                        [menuItem setTitle:[NSString stringWithFormat:[menuItem title], [linkURLStrings count]]];
                        [menuItem setRepresentedObject:linkURLStrings];
                        [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:SROpenLinkInNewBackgroundTabTag];
                    }
                    
                    // Remove 'Open Link in New Window'
                    [SRContextMenu removeMenuItem:menuItem fromItems:items ofTag:WebMenuItemTagOpenLinkInNewWindow];
                }
            }
            else {
                if (shiftFlag) {
                    // Add 'Open Link in New Background Window'
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenLinkInNewBackgroundWindowTag target:nil];
                    [menuItem setRepresentedObject:linkURL];
                    [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenLinkInNewWindow];
                    
                    // Remove 'Open Link in New Window'
                    [SRContextMenu removeMenuItem:menuItem fromItems:items ofTag:WebMenuItemTagOpenLinkInNewWindow];
                }
            }
        }
    }
    
    // Link is not contained
    else {
        // Text is selected
        if (isSelected) {
            // Copy default menu item
            [items addObjectsFromArray:allowedDefaultMenuItems];
            
            // Get selected string
            NSView*         documentView;
            NSString*       selectedString = nil;
            documentView = [[[element objectForKey:WebElementFrameKey] frameView] documentView];
            if ([documentView respondsToSelector:@selector(selectedString)]) {
                selectedString = [documentView performSelector:@selector(selectedString)];
            }
            
#if 0
            // Get current search engine
            NSDictionary*   currentEngine;
            currentEngine = [_searchFieldController currentEngine];
            
            if (selectedString && currentEngine) {
                // Add 'Search by serach engine'
                NSMenuItem* searchMenuItem;
                searchMenuItem = [self _createSerachMenuItemByEngine:currentEngine 
                        searchString:selectedString];
                [items addObject:searchMenuItem];
                
                // Find other search engines
                NSMutableArray* otherEngines;
                NSArray*        engines;
                NSEnumerator*   enumerator;
                NSDictionary*   engine;
                otherEngines = [NSMutableArray array];
                engines = [[SRSearchEnginesManager sharedInstance] searchEngines];
                enumerator = [engines objectEnumerator];
                while (engine = [enumerator nextObject]) {
                    if ([[engine objectForKey:@"isUsing"] boolValue] && 
                        ![[engine objectForKey:@"title"] isEqualToString:[currentEngine objectForKey:@"title"]])
                    {
                        [otherEngines addObject:engine];
                    }
                }
                
                // Add search by other engines menu
                if ([otherEngines count] > 0) {
                    // Add 'Search by other'
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRFindByOtherSearchEngineTag target:nil];
                    [items addObject:menuItem];
                    
                    // Create submenu
                    NSMenu* otherEnginesSubmenu;
                    otherEnginesSubmenu = [[NSMenu alloc] initWithTitle:@"Other engine"];
                    [menuItem setSubmenu:otherEnginesSubmenu];
                    
                    enumerator = [otherEngines objectEnumerator];
                    while (engine = [enumerator nextObject]) {
                        // Add 'Search by serach engine'
                        NSMenuItem* searchMenuItem;
                        searchMenuItem = [self _createSerachMenuItemByEngine:engine 
                                searchString:selectedString];
                        [otherEnginesSubmenu addItem:searchMenuItem];
                    }
                }
            }
#endif
            
            // For link nodes
            if ([linkURLStrings count] > 1) {
                [items addObject:[NSMenuItem separatorItem]];
                
                if ((selectNewTabs && !shiftFlag) || 
                    (!selectNewTabs && shiftFlag))
                {
                    // Add open links in tabs
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenAllLinksInNewTabsTag target:nil];
                    [menuItem setTitle:[NSString stringWithFormat:[menuItem title], [linkURLStrings count]]];
                    [menuItem setRepresentedObject:linkURLStrings];
                    [items addObject:menuItem];
                }
                else {
                    // Add open links in tabs
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenAllLinksInNewBackgroundTabsTag target:nil];
                    [menuItem setTitle:[NSString stringWithFormat:[menuItem title], [linkURLStrings count]]];
                    [menuItem setRepresentedObject:linkURLStrings];
                    [items addObject:menuItem];
                }
            }
        }
        
        // Text is not selected
        else {
#if 0
            // Add navigation items
            if ([[webView mainFrame] dataSource]) {
                if ([webView canGoBack]) {
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRWebViewGoBackTag target:nil];
                    [items addObject:menuItem];
                }
                if ([webView canGoForward]) {
                    menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRWebViewGoForwardTag target:nil];
                    [items addObject:menuItem];
                }
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRWebViewReloadPageTag target:nil];
                [items addObject:menuItem];
                [items addObject:[NSMenuItem separatorItem]];
            }
#endif
            
            // Copy default menu item
            [items addObjectsFromArray:allowedDefaultMenuItems];
            
#if 0
            // Add tab menu
            if (enableTabbedBrowsing) {
                NSMenu* tabMenu;
                tabMenu = [SRContextMenu tabContextMenu];
                
                menuItem = [SRContextMenu copyMenuItemFrom:tabMenu ofTag:SRTabNewTabTag target:nil];
                [items addObject:menuItem];
                if ([_srTabView numberOfItems] > 1) {
                    menuItem = [SRContextMenu copyMenuItemFrom:tabMenu ofTag:SRTabCloseTabTag target:nil];
                    [items addObject:menuItem];
                }
            }
#endif
            
            // Add 'View Soruce'
            if ([[webView mainFrame] dataSource]) {
                if ([items count] > 0 && ![[items lastObject] isSeparatorItem]) {
                    [items addObject:[NSMenuItem separatorItem]];
                }
                
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRViewSourceTag target:nil];
                [menuItem setRepresentedObject:_document]; // SRXMLDocument
                [items addObject:menuItem];
            }
        }
    }
    
    // Image is selected
    if (imageURL) {
        // Add 'Copy Image Location to Clipboard'
        menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SRCopyImageURLTag target:nil];
        [menuItem setRepresentedObject:imageURL];
        [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagCopyImageToClipboard];
        
        // For tabbed browsing
        if (enableTabbedBrowsing) {
            if ((selectNewTabs && !shiftFlag) || 
                (!selectNewTabs && shiftFlag))
            {
                // Add 'Open Image in New Tab'
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenImageInNewTabTag target:nil];
                [menuItem setRepresentedObject:imageURL];
                [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenImageInNewWindow];
            }
            else {
                // Add 'Open Image in New Background Window'
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenImageInNewBackgroundWindowTag target:nil];
                [menuItem setRepresentedObject:imageURL];
                [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenImageInNewWindow];
                
                // Add 'Open Image in New Background Tab'
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenImageInNewBackgroundTabTag target:nil];
                [menuItem setRepresentedObject:imageURL];
                [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:SROpenImageInNewBackgroundWindowTag];
                
                // Remove 'Open Image in New Window'
                [SRContextMenu removeMenuItem:menuItem fromItems:items ofTag:WebMenuItemTagOpenImageInNewWindow];
            }
        }
        else {
            if (shiftFlag) {
                // Add 'Open Image in New Background Window'
                menuItem = [SRContextMenu copyMenuItemFrom:menu ofTag:SROpenImageInNewBackgroundWindowTag target:nil];
                [menuItem setRepresentedObject:imageURL];
                [SRContextMenu insertMenuItem:menuItem inItems:items afterTag:WebMenuItemTagOpenImageInNewWindow];
                
                // Remove 'Open Image in New Window'
                [SRContextMenu removeMenuItem:menuItem fromItems:items ofTag:WebMenuItemTagOpenImageInNewWindow];
            }
        }
    }
    
    return items;
}

- (void)webView:(WebView*)webView 
        mouseDidMoveOverElement:(NSDictionary*)info 
        modifierFlags:(unsigned int)modifierFlags
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    NSString*   status = nil;
    
    // Get window controller
    id  windowController;
    windowController = [[self window] windowController];
    
    // Get URL string
    NSString*   URLString;
    URLString = [[info objectForKey:WebElementLinkURLKey] _web_userVisibleString];
    
    if (URLString) {
#if 1
        // Go to URL
        status = NSLocalizedString(UTF8STR("Go to “%@”"), nil);
        goto finalize;
#else
        // Get scheme
        NSURL*      URL;
        NSString*   scheme;
        URL = [NSURL _web_URLWithUserTypedString:URLString];
        scheme = [URL scheme];
        
        // For mailto:
        if ([scheme isEqualToString:@"mailto"]) {
            status = NSLocalizedString(UTF8STR("Send e-mail to “%@”"), nil);
            goto finalize;
        }
        
        // Modifier flag is specified
        if (flags) {
            // Check key mask
            unsigned int    cmdFlag, optionFlag, shiftFlag;
            cmdFlag = flags & NSCommandKeyMask;
            optionFlag = flags & NSAlternateKeyMask;
            shiftFlag = flags & NSShiftKeyMask;
            
            // Download linked file
            if (!cmdFlag && optionFlag && !shiftFlag) {
                status = NSLocalizedString(UTF8STR("Download “%@”"), nil);
                goto finalize;
            }
            
            // Check tab browsing
            BOOL    enableTabbedBrowsing, selectNewTabs;
            enableTabbedBrowsing = [defaults boolForKey:SRTabEnableTabbedBrowsing];
            selectNewTabs = [defaults boolForKey:SRTabSelectNewTabs];
            
            // If tabbed browsing is enable
            if (enableTabbedBrowsing) {
                // If select new tabs
                if (selectNewTabs) {
                    // Open in new tabs and select it
                    if (cmdFlag && !optionFlag && !shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new tab"), nil);
                        goto finalize;
                    }
                    // Open in new tabs
                    else if (cmdFlag && !optionFlag && shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new background tab"), nil);
                        goto finalize;
                    }
                    // Open in new window
                    else if (cmdFlag && optionFlag && !shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new window"), nil);
                        goto finalize;
                    }
                    // Open in new window in background
                    else if (cmdFlag && optionFlag && shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new background window"), nil);
                        goto finalize;
                    }
                }
                // If not select new tabs
                else {
                    // Open in new tabs
                    if (cmdFlag && !optionFlag && !shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new background tab"), nil);
                        goto finalize;
                    }
                    // Open in new tabs and select it
                    else if (cmdFlag && !optionFlag && shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new tab"), nil);
                        goto finalize;
                    }
                    // Open in new window in background
                    else if (cmdFlag && optionFlag && !shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new background window"), nil);
                        goto finalize;
                    }
                    // Open in new window
                    else if (cmdFlag && optionFlag && shiftFlag) {
                        status = NSLocalizedString(UTF8STR("Open “%@” in a new window"), nil);
                        goto finalize;
                    }
                }
            }
            // If tabbed browsing is disable
            else {
                // Open in new window
                if (cmdFlag && !shiftFlag) {
                    status = NSLocalizedString(UTF8STR("Open “%@” in a new window"), nil);
                    goto finalize;
                }
                // Open in new window in background
                else if (cmdFlag && shiftFlag) {
                    status = NSLocalizedString(UTF8STR("Open “%@” in a new background window"), nil);
                    goto finalize;
                }
            }
        }
        
        // Get target
        WebFrame*   target;
        target = [info objectForKey:WebElementLinkTargetFrameKey];
        
        // Target is in this frame
        if (target) {
            // Go to URL
            status = NSLocalizedString(UTF8STR("Go to “%@”"), nil);
            goto finalize;
        }
        else {
            if([defaults boolForKey:SRTabEnableTabbedBrowsing] && [defaults boolForKey:SRTabTargetLinkUseTab]){
                //target link use tab
                status = NSLocalizedString(UTF8STR("Open “%@” in a new tab"), nil);
            }else{
                // Open page in new window
                status = NSLocalizedString(UTF8STR("Open “%@” in a new window"), nil);
            }
            goto finalize;
        }
#endif
    }
    
finalize:
    if (status) {
        [windowController setValue:[NSString stringWithFormat:status, URLString]
                forKey:SRMainWebViewStatus];
    }
    else {
        [windowController setValue:@"" forKey:SRMainWebViewStatus];
    }
}

//--------------------------------------------------------------//
#pragma mark -- WebDocumentView protocol --
//--------------------------------------------------------------//

- (void)setDataSource:(WebDataSource*)dataSource
{
    _dataSource = dataSource;
}

- (void)dataSourceUpdated:(WebDataSource*)dataSource
{
}

- (void)setNeedsLayout:(BOOL)flag
{
}

- (void)layout
{
}

- (void)viewDidMoveToHostWindow
{
}

- (void)viewWillMoveToHostWindow:(NSWindow*)window
{
}

@end
