/*
SRAutoFadeWindow.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRAutoFadePanel.h"

#pragma mark -
//--------------------------------------------------------------//
// _SRAutoFadeContentView class
//--------------------------------------------------------------//

static float    _backgroundRGB[] = { 0.93, 0.93, 0.93 };
static float    _borderRGB[] = { 0.75, 0.75, 0.75 };
static float    _lightEdgeRGB[] = { 0.98, 0.98, 0.98 };
static float    _darkEdgeRGB[] = { 0.73, 0.73, 0.73 };

@interface _SRAutoFadeContentView : NSView
{
    float   _alpha;
    BOOL    _halfTransparent;
}

- (float)alphaValue;
- (void)setAlphaValue:(float)alpha;

- (BOOL)halfTransparent;
- (void)setHalfTransparent:(BOOL)flag;

@end

@implementation _SRAutoFadeContentView

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    _alpha = 1.0f;
    _halfTransparent = NO;
    
    return self;
}

- (void)mouseEntered:(NSEvent*)event
{
    [[self window] mouseEntered:event];
}

- (void)mouseExited:(NSEvent*)event
{
    [[self window] mouseExited:event];
}

- (void)mouseDown:(NSEvent*)event
{
    [[self window] mouseDown:event];
}

- (float)alphaValue
{
    return _alpha;
}

- (void)setAlphaValue:(float)alpha
{
    _alpha = alpha;
    [self setNeedsDisplay:YES];
}

- (BOOL)halfTransparent
{
    return _halfTransparent;
}

- (void)_makeHalfTransparentForView:(NSView*)view
{
    // Invoke view's selector
    if (view != self && [view respondsToSelector:@selector(setHalfTransparent:)]) {
        objc_msgSend(view, @selector(setHalfTransparent:), _halfTransparent);
    }
    
    // For subview
    NSEnumerator*   enumerator;
    NSView*         subview;
    enumerator = [[view subviews] objectEnumerator];
    while (subview = [enumerator nextObject]) {
        [self _makeHalfTransparentForView:subview];
    }
}

- (void)setHalfTransparent:(BOOL)flag
{
    _halfTransparent = flag;
    
    // Change subview's string attribute
    [self _makeHalfTransparentForView:self];
    
    [self setNeedsDisplay:YES];
}

- (void)_drawHalfTransparentRect:(NSRect)rect
{
    // Create color
    NSColor*    color;
    color = [NSColor colorWithCalibratedWhite:0.5 alpha:_alpha];
    
    // Check with frame
    NSRect  frame;
    frame = [self frame];
    if (NSEqualRects(frame, rect)) {
        // Draw border
        [color set];
        
        static float    _radius = 12.0;
        float   left = rect.origin.x;
        float   right = rect.origin.x + rect.size.width;
        float   top = rect.origin.y + rect.size.height;
        float   bottom = rect.origin.y;
        
        NSBezierPath*   bezierPath;
        bezierPath = [NSBezierPath bezierPath];
        [bezierPath moveToPoint:NSMakePoint(left + _radius, bottom)];
        [bezierPath lineToPoint:NSMakePoint(right - _radius, bottom)];
        [bezierPath curveToPoint:NSMakePoint(right, bottom + _radius) 
                controlPoint1:NSMakePoint(right - _radius / 2.0, bottom) 
                controlPoint2:NSMakePoint(right, bottom + _radius / 2.0)];
        [bezierPath lineToPoint:NSMakePoint(right, top - _radius)];
        [bezierPath curveToPoint:NSMakePoint(right - _radius, top) 
                controlPoint1:NSMakePoint(right, top - _radius / 2.0) 
                controlPoint2:NSMakePoint(right - _radius / 2.0, top)];
        [bezierPath lineToPoint:NSMakePoint(left + _radius, top)];
        [bezierPath curveToPoint:NSMakePoint(left, top - _radius) 
                controlPoint1:NSMakePoint(left + _radius / 2.0, top) 
                controlPoint2:NSMakePoint(left, top - _radius / 2.0)];
        [bezierPath lineToPoint:NSMakePoint(left, bottom + _radius)];
        [bezierPath curveToPoint:NSMakePoint(left + _radius, bottom) 
                controlPoint1:NSMakePoint(left, bottom + _radius / 2.0) 
                controlPoint2:NSMakePoint(left + _radius / 2.0, bottom)];
        
        [bezierPath fill];
    }
    else {
        // Fill background
        [color set];
        NSRectFill(rect);
    }
}

- (void)drawRect:(NSRect)rect
{
    // Save graphics context
    [NSGraphicsContext saveGraphicsState];
    
    // For half transparent
    if (_halfTransparent) {
        [self _drawHalfTransparentRect:rect];
    }
    
    else {
        // Create colors
        NSColor*    backgroundColor;
        NSColor*    borderColor;
        NSColor*    lightEdgeColor;
        NSColor*    darkEdgeColor;
        int         index = 0;
        
        backgroundColor = [NSColor colorWithCalibratedRed:_backgroundRGB[index * 3] 
                green:_backgroundRGB[index * 3 + 1]  
                blue:_backgroundRGB[index * 3 + 2]  
                alpha:_alpha];
        borderColor = [NSColor colorWithCalibratedRed:_borderRGB[index * 3] 
                green:_borderRGB[index * 3 + 1]  
                blue:_borderRGB[index * 3 + 2]  
                alpha:_alpha];
        lightEdgeColor = [NSColor colorWithCalibratedRed:_lightEdgeRGB[index * 3] 
                green:_lightEdgeRGB[index * 3 + 1]  
                blue:_lightEdgeRGB[index * 3 + 2]  
                alpha:_alpha];
        darkEdgeColor = [NSColor colorWithCalibratedRed:_darkEdgeRGB[index * 3] 
                green:_darkEdgeRGB[index * 3 + 1]  
                blue:_darkEdgeRGB[index * 3 + 2]  
                alpha:_alpha];
        
        // Fill back ground
        [backgroundColor set];
        NSRectFill(rect);
        
        // Check with frame
        NSRect  frame;
        frame = [self frame];
        if (NSEqualRects(frame, rect)) {
            // Draw border
            [borderColor set];
            NSFrameRect(rect);
            
            // Draw edge
            NSRect  edge;
            edge = NSInsetRect(rect, 1.0, 1.0);
            
            [lightEdgeColor set];
            NSFrameRect(NSMakeRect(edge.origin.x, edge.origin.y + 1.0, 1.0, edge.size.height - 1.0));
            NSFrameRect(NSMakeRect(edge.origin.x, edge.origin.y + edge.size.height - 1.0, edge.size.width - 1.0, 1.0));
            
            [darkEdgeColor set];
            NSFrameRect(NSMakeRect(edge.origin.x + 1.0, edge.origin.y, edge.size.width - 1.0, 1.0));
            NSFrameRect(NSMakeRect(edge.origin.x + edge.size.width - 1.0, edge.origin.y, 1.0, edge.size.height - 1.0));
        }
    }
    
    // Restore graphics context
    [NSGraphicsContext restoreGraphicsState];
}

@end

#pragma mark -
//--------------------------------------------------------------//
// SRAutoFadePanel class
//--------------------------------------------------------------//

static NSTimeInterval   _SRAutoFadeIntervalTillFading = 5; // seconds
static NSTimeInterval   _SRAutoFadeIntervalFadingOut = 0.1; // seconds

static float    _SRAutoFadeDecreaseUnit = 0.2f;
static float    _SRAutoFadeHalfTransparentAlpha = 0.5f;

@implementation SRAutoFadePanel

- (void)_setTrackingRect
{
    NSView* contentView;
    contentView = [self contentView];
    [contentView addTrackingRect:[contentView frame] owner:self userData:nil assumeInside:NO];
}

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (id)initWithContentRect:(NSRect)contentRect 
        styleMask:(unsigned int)styleMask 
        backing:(NSBackingStoreType)backingType 
        defer:(BOOL)flag
{
    self = [super initWithContentRect:contentRect 
            styleMask:NSTexturedBackgroundWindowMask 
            backing:backingType 
            defer:flag];
    if (!self) {
        return nil;
    }
    
    [self setBackgroundColor:[NSColor clearColor]];
    [self setOpaque:NO];
    [self setLevel:NSStatusWindowLevel];
    
    return self;
}

- (void)awakeFromNib
{
    // Replace content view
    NSView*         contentView;
    NSArray*        subviews;
    NSEnumerator*   enumerator;
    NSView*         subview;
    contentView = [[_SRAutoFadeContentView alloc] initWithFrame:NSZeroRect];
    [contentView autorelease];
    subviews = [[self contentView] subviews];
    [self setContentView:contentView];
    
    enumerator = [subviews objectEnumerator];
    while (subview = [enumerator nextObject]) {
        [contentView addSubview:subview];
    }
    
    // Set tracking rect
    [self _setTrackingRect];
    
    // Set fade status
    _fadeStatus = SRAutoFadeKeepAppeared;
}

- (void)setFrame:(NSRect)frameRect 
        display:(BOOL)displayFlag 
        animate:(BOOL)animationFlag
{
    [super setFrame:frameRect display:displayFlag animate:animationFlag];
    
    // Update tracking rect
    [self _setTrackingRect];
}

- (void)mouseEntered:(NSEvent*)event
{
    // Set fade status
//    [self setFadeStatus:SRAutoFadeKeepAppeared];
}

- (void)mouseExited:(NSEvent*)event
{
    // Set fade status
    if ([self fadeStatus] == SRAutoFadeKeepAppeared) {
        [self setFadeStatus:SRAutoFadeApperaed];
    }
}

- (void)mouseDown:(NSEvent*)event
{
    // Set fade status
    [self setFadeStatus:SRAutoFadeKeepAppeared];
}

#pragma mark -
//--------------------------------------------------------------//
// Fade status
//--------------------------------------------------------------//

- (void)_timerTillFadingExpired:(id)userInfo
{
    // Set fade status
    [self setFadeStatus:SRAutoFadeFadingOut];
}

- (void)_timerFadintOutExpired:(id)userInfo
{
#if 1
    // Get alpha value
    float   alpha;
    alpha = [self alphaValue];
    
    if (alpha > 0.0f) {
        // Decrease alpha value
        alpha -= _SRAutoFadeDecreaseUnit;
        if (alpha < 0.0f) {
            alpha = 0.0f;
        }
        [self setAlphaValue:alpha];
    }
#else
    // Get alpha value
    float   alpha;
    alpha = [[self contentView] alphaValue];
    
    if (alpha > 0.0f) {
        // Decrease alpha value
        alpha -= _SRAutoFadeDecreaseUnit;
        if (alpha < 0.0f) {
            alpha = 0.0f;
        }
        [[self contentView] setAlphaValue:alpha];
    }
#endif
    
    // Stop timer
    if (alpha <= 0.0f) {
        // Check delegate policy
        BOOL    shouldRemain = YES;
        if (_autoFadeDelegate && 
            [_autoFadeDelegate respondsToSelector:@selector(autoFadePanelShouldRemainAfterFadingOut)])
        {
            shouldRemain = [_autoFadeDelegate autoFadePanelShouldRemainAfterFadingOut];
        }
        
        if (shouldRemain) {
            [self setFadeStatus:SRAutoFadeHalfTransparent];
        }
        else {
            [self setFadeStatus:SRAutoFadeDisappeared];
        }
    }
}

- (int)fadeStatus
{
    return _fadeStatus;
}

- (void)setFadeStatus:(int)fadeStatus
{
    // Get content view
    _SRAutoFadeContentView* contentView;
    contentView = (_SRAutoFadeContentView*)[self contentView];
    
    // Cancel all timers
    if (_timerTillFading) {
        [_timerTillFading invalidate];
    }
    _timerTillFading = nil;
    
    if (_timerFadingOut) {
        [_timerFadingOut invalidate];
    }
    _timerFadingOut = nil;
    
    // Set fade status
    _fadeStatus = fadeStatus;
    
    switch (fadeStatus) {
    case SRAutoFadeKeepAppeared: {
        // Show itself
        if (![self isVisible]) { [self orderFront:self]; }
        // Set alpha value
        if ([self alphaValue] < 1.0f) { [self setAlphaValue:1.0f]; }
        if ([contentView alphaValue] < 1.0f) { [contentView setAlphaValue:1.0f]; }
        // Set half transparent
        if ([contentView halfTransparent]) { [contentView setHalfTransparent:NO]; }
        // Show shadow
        if (![self hasShadow]) { [self setHasShadow:YES]; }
        
        break;
    }
    case SRAutoFadeApperaed: {
        // Show itself
        if (![self isVisible]) { [self orderFront:self]; }
        // Set alpha value
        if ([self alphaValue] < 1.0f) { [self setAlphaValue:1.0f]; }
        if ([contentView alphaValue] < 1.0f) { [contentView setAlphaValue:1.0f]; }
        // Set half transparent
        if ([contentView halfTransparent]) { [contentView setHalfTransparent:NO]; }
        // Show shadow
        if (![self hasShadow]) { [self setHasShadow:YES];  }
        
        // Restart timer
        _timerTillFading = [NSTimer scheduledTimerWithTimeInterval:_SRAutoFadeIntervalTillFading 
                target:self 
                selector:@selector(_timerTillFadingExpired:) 
                userInfo:nil 
                repeats:NO];
        
        // Notify to delegate
        if (_autoFadeDelegate && 
            [_autoFadeDelegate respondsToSelector:@selector(autoFadePanelAppeared)])
        {
            [_autoFadeDelegate autoFadePanelAppeared];
        }
        
        break;
    }
    case SRAutoFadeFadingOut: {
        // Set half transparent
        if ([contentView halfTransparent]) { [contentView setHalfTransparent:NO]; }
        // Show shadow
        if (![self hasShadow]) { [self setHasShadow:YES]; }
        
        // Start fading out timer
        _timerFadingOut = [NSTimer scheduledTimerWithTimeInterval:_SRAutoFadeIntervalFadingOut 
                target:self 
                selector:@selector(_timerFadintOutExpired:) 
                userInfo:nil 
                repeats:YES];
        
        break;
    }
    case SRAutoFadeHalfTransparent: {
        // Set half transparent
        if (![contentView halfTransparent]) { [contentView setHalfTransparent:YES]; }
        [contentView setAlphaValue:_SRAutoFadeHalfTransparentAlpha];
        // Set alpha value
        if ([self alphaValue] < 1.0f) { [self setAlphaValue:1.0f]; }
        // Hide shadow
        if ([self hasShadow]) { [self setHasShadow:NO]; }
        
        break;
    }
    case SRAutoFadeDisappeared: {
        // Close itself
        [self orderOut:self];
        
        break;
    }
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Delegate
//--------------------------------------------------------------//

- (id)autoFadeDelegate
{
    return _autoFadeDelegate;
}

- (void)setAutoFadeDelegate:(id)delegate
{
    _autoFadeDelegate = delegate;
}

@end
