/*
SRMainWindowController.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRMainWindowController.h"

// Toolbar items
static NSArray* _toolbarItemIdentifiers = nil;
static NSArray* _toolbarDefaultItemIdentifiers = nil;

// Toolbar identifier
NSString*   SRMainWindowToolbarIdentifier = @"SRMainWindowToolbarIdentifier";

// Toolbar item identifier
NSString*   SRGoBackIdentifier = @"SRGoBack";
NSString*   SRGoForwardIdentifier = @"SRGoForward";
NSString*   SRReloadPageIdentifier = @"SRReloadPage";
NSString*   SRStopLoadingIdentifier = @"SRStopLoading";
NSString*   SRGoHomeIdentifier = @"SRGoHome";
NSString*   SRURLIdentifier = @"SRURL";
NSString*   SRSidebarIdentifier = @"SRSidebar";
NSString*   SRHideSidebarIdentifier = @"SRHideSidebar";
NSString*   SRBookmarkPageIdentifier = @"SRBookmarkPage";
NSString*   SRNewTabIdentifier = @"SRNewTab";
NSString*   SRCloseTabIdentifier = @"SRCloseTab";
NSString*   SRBiggerTextIdentifier = @"SRBiggerText";
NSString*   SRSmallerTextIdentifier = @"SRSmallerText";
NSString*   SRCustomizeIdentifier = @"SRCustomize";
NSString*   SRSearchIdentifier = @"SRSearch";

NSString*   SRMainToolbarLabelTable = @"MainToolbar";

// Key path
NSString*   SRMainWebViewTitle = @"webViewTitle";
NSString*   SRMainWebViewURLString = @"webViewURLString";
NSString*   SRMainWebViewIsLoading = @"webViewIsLoading";
NSString*   SRMainWebViewIsIndeterminate = @"webViewIsIndeterminate";
NSString*   SRMainWebViewEstimatedProgress = @"webViewEstimatedProgress";
NSString*   SRMainWebViewStatus = @"webViewStatus";
NSString*   SRMainWebViewIsSecured = @"webViewIsSecured";

// Toolbar icons
static NSDictionary*    _iconImageDict = nil;

// URL location
int         SRURLMinWidth = 48;
int         SRURLMaxWidth = 1024;

// Search field
int         SRSearchMinWidth = 48;
int         SRSearchMaxWidth = 128;

// URL button tags
enum {
    SRURLButtonIDN = 0, 
};

#define _OK_BUTTON_TAG   0

@interface SRMainWindowController (private)
- (void)_setUserAgentOfWebView:(WebView*)webView;
@end

@implementation SRMainWindowController

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (id)initWithWindowNibName:(NSString*)nibName 
        initialFrameName:(NSString*)frameName 
        initialGroupName:(NSString*)groupName
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    self = [super initWithWindowNibName:nibName];
    if (!self) {
        return nil;
    }
    
    static  BOOL    _isInitialized = NO;
    if (!_isInitialized) {
        // Create toolbar item identifiers
        _toolbarItemIdentifiers = [[NSArray arrayWithObjects:
            SRGoBackIdentifier, SRGoForwardIdentifier, SRReloadPageIdentifier, SRStopLoadingIdentifier, 
            SRGoHomeIdentifier, SRURLIdentifier, SRSidebarIdentifier, SRBookmarkPageIdentifier, 
            SRNewTabIdentifier, SRCloseTabIdentifier, SRBiggerTextIdentifier, SRSmallerTextIdentifier, 
            SRCustomizeIdentifier, SRSearchIdentifier, 
            NSToolbarSeparatorItemIdentifier, NSToolbarSpaceItemIdentifier, 
            NSToolbarFlexibleSpaceItemIdentifier, nil] retain];
        _toolbarDefaultItemIdentifiers = [[NSArray arrayWithObjects:
            SRGoBackIdentifier, SRGoForwardIdentifier, SRReloadPageIdentifier, SRStopLoadingIdentifier, 
            SRURLIdentifier, SRSearchIdentifier, SRSidebarIdentifier, 
            nil] retain];
    }
    
    // Load nib file    
    _sideBarController = [[SRSideBarController alloc] initWithMainWindowController:self];
    if (![NSBundle loadNibNamed:@"SideBar" owner:_sideBarController]) {
        // Fatal error
        SR_FATAL(@"Could not load SideBar.nib");
    }
    
    // Configure tab view
    [self addNewTabWithLabel:NSLocalizedString(@"Untitled", @"Untitled") 
            frameName:frameName 
            groupName:groupName];
    [_srTabView selectItemAtIndex:0];
    [_srTabView setDelegate:self];
    [_srTabView setTabShown:[defaults boolForKey:SRTabEnableTabbedBrowsing]];
    [_srTabView setTabShownAlways:[defaults boolForKey:SRTabAlwaysShowTabBar]];
    [_srTabView setAccordingToStringWidth:[defaults boolForKey:SRTabAccordingToStringWidth]];
    
    // Initialize instance variables
    _isResizedByJavaScript = NO;
    _isInitializedSidebarVisibility = NO;
    
    SRSearchField*  searchField;
    searchField = [self searchField];
    _searchFieldController = [[SRSearchFieldController alloc] initWithMainWindowController:self];
    [_searchFieldController setSearchField:searchField];
    [searchField setTarget:_searchFieldController];
    
    _tabExposeController = [[SRTabExposeController alloc] init];
    
    // Initialize properties
    [self setValue:NSLocalizedString(@"Untitled", @"Untitled") forKey:SRMainWebViewTitle];
    [self setValue:@"" forKey:SRMainWebViewURLString];
    [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsLoading];
    [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsIndeterminate];
    [self setValue:[NSNumber numberWithDouble:0.0] forKey:SRMainWebViewEstimatedProgress];
    [self setValue:@"" forKey:SRMainWebViewStatus];
    [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsSecured];
    
    // Open default page
    int defaultPage;
    defaultPage = [defaults integerForKey:SRGeneralNewWindowsOpenWith];
    switch (defaultPage) {
    case SRGeneralNewWindowsOpenWithHomePage: {
        // Go Home
        [self goHomeAction:self];
        break;
    }
    case SRGeneralNewWindowsOpenWithSamePage: {
        // Get front window
        NSWindow*   frontWindow;
        frontWindow = SRGetFrontMainWindow();
        if (frontWindow) {
            // Get request
            NSURLRequest*   request;
            request = [[[[[frontWindow windowController] selectedWebView] 
                    mainFrame] dataSource] request];
            
            // Open request
            [self openRequest:request];
            break;
        }
        break;
    }
    }
    
    // Register notifications
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    [center addObserver:self 
            selector:@selector(webViewProgressStarted:) 
            name:WebViewProgressStartedNotification 
            object:nil];
    [center addObserver:self 
            selector:@selector(webViewProgressEstimateChanged:) 
            name:WebViewProgressEstimateChangedNotification 
            object:nil];
    [center addObserver:self 
            selector:@selector(webViewProgressFinished:) 
            name:WebViewProgressFinishedNotification 
            object:nil];
    
    [center addObserver:self 
            selector:@selector(webImageSizeChanged:) 
            name:SRWebImageSizeChangeNotification 
            object:nil];
    
    [center addObserver:self 
            selector:@selector(faviconAdded:) 
            name:SRFaviconAddedNotification 
            object:nil];
    
    // Register key value observation
    [defaults addObserver:self 
            forKeyPath:SRBookmarkBookmarksBar 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRTabEnableTabbedBrowsing 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRTabSelectNewTabs 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRTabAlwaysShowTabBar 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRTabAccordingToStringWidth 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRUserAgent 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRUserAgentOtherName 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRIconName 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRIconUseFavicon 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRIconUseFaviconBookmarkBar 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SREnableJavaScirptStatusMessage 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    
    [self addObserver:self 
            forKeyPath:SRMainWebViewURLString 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    
    return self;
}

- (void)windowDidLoad
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get default window size
    NSString*   sizeString;
    sizeString = [defaults objectForKey:SRMainWindowContentSize];
    
    // Bookmarks bar and status bar visiblity
    [self setBookmarksBarVisible:[defaults boolForKey:SRMainWindowBookmarksBarVisible] display:NO];
    [self setStatusBarVisible:[defaults boolForKey:SRMainWindowStatusBarVisible] display:NO];
    
    // Get window
    NSWindow*   window;
    window = [self window];
    
    // Set default window size
    if (sizeString) {
        // Restore size
        NSSize  size;
        size = NSSizeFromString(sizeString);
        
        if (size.width > 200 && size.height > 200) { // Minimum size
            // Make window default size
            [window setContentSize:size];
        }
    }
    
    // For first window
    if ([[[NSDocumentController sharedDocumentController] documents] count] == 1) {
        // Get default window origin
        NSString*   originString;
        originString = [defaults objectForKey:SRMainWindowFrameOrigin];
        if (originString) {
            // Restore origin
            NSPoint origin;
            origin = NSPointFromString(originString);
            
            if (!NSEqualPoints(origin, NSZeroPoint)) {
                // Make window default origin
                [window setFrameTopLeftPoint:origin];
            }
        }
    }
    // For other windows
    else {
        // Get front window
        NSWindow*   frontWindow;
        frontWindow = SRGetFrontMainWindow();
        if (frontWindow) {
            // Calculate cascaded point
            NSRect  frontWindowFrame;
            NSPoint frontWindowPoint;
            frontWindowFrame = [frontWindow frame];
            frontWindowPoint = frontWindowFrame.origin;
            frontWindowPoint.y += frontWindowFrame.size.height;
            [window setFrameTopLeftPoint:[window cascadeTopLeftFromPoint:frontWindowPoint]];
        }
    }
    
    // Get toolbar icons
    NSString*   iconName;
    iconName = [defaults stringForKey:SRIconName];
    _iconImageDict = [[[[NSApp delegate] iconInstaller] iconsOfName:iconName] retain];
    
    // Create toolbar
    NSToolbar*  toolbar;
    toolbar = [[NSToolbar alloc] 
            initWithIdentifier:SRMainWindowToolbarIdentifier];
    [toolbar autorelease];
    [toolbar setDelegate:self];
    [toolbar setAllowsUserCustomization:YES];
    [toolbar setAutosavesConfiguration:YES];
    [window setToolbar:toolbar];
    
    // Check window position
    NSRect  visibleFrame;
    NSRect  windowFrame, originalFrame;
    visibleFrame = [[window screen] visibleFrame];
    windowFrame = [window frame];
    originalFrame = windowFrame;
    
    if (windowFrame.origin.x < visibleFrame.origin.x ||
        windowFrame.origin.x + windowFrame.size.width > visibleFrame.origin.x + visibleFrame.size.width)
    {
        windowFrame.origin.x = visibleFrame.origin.x;
    }
    if (windowFrame.origin.y < visibleFrame.origin.y)
    {
        windowFrame.origin.y = visibleFrame.origin.y + visibleFrame.size.height - windowFrame.size.height;
    }
    
    if (!NSEqualRects(windowFrame, originalFrame)) {
        [window setFrame:windowFrame display:YES];
    }
    
    // Create lock image view
    _lockIconView = [[NSImageView alloc] initWithFrame:
            NSMakeRect(windowFrame.size.width - 44, windowFrame.size.height - 19, 14, 16)];
    [_lockIconView setImage:[NSImage imageNamed:@"Lock"]];
    [_lockIconView setAutoresizingMask:NSViewMinXMargin | NSViewMinYMargin];
    [[[window contentView] superview] addSubview:_lockIconView];
    [_lockIconView setHidden:YES];
    
    // Dose not use automatic cascading
    [self setShouldCascadeWindows:NO];
    
    // Update browser of bookmarks bar
    _bookmarksBarBrowser = -1; // Dummy
    [self setBookmarksBarBrowser:
            [defaults integerForKey:SRBookmarkBookmarksBar]];
    
    // Update file menu
    [[[[NSApp mainMenu] itemWithTag:SRFileTag] submenu] update];
}

- (void)dealloc
{
    // Release controllers
    [_sideBarController release];
    [_authController release];
    [_searchFieldController release];
    [_lockIconView release];
    [_tabExposeController release];
    
    // Remove observers
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    [defaults removeObserver:self forKeyPath:SRBookmarkBookmarksBar];
    [defaults removeObserver:self forKeyPath:SRTabEnableTabbedBrowsing];
    [defaults removeObserver:self forKeyPath:SRTabSelectNewTabs];
    [defaults removeObserver:self forKeyPath:SRTabAlwaysShowTabBar];
    [defaults removeObserver:self forKeyPath:SRTabAccordingToStringWidth];
    [defaults removeObserver:self forKeyPath:SRUserAgent];
    [defaults removeObserver:self forKeyPath:SRUserAgentOtherName];
    [defaults removeObserver:self forKeyPath:SRIconName];
    [defaults removeObserver:self forKeyPath:SRIconUseFavicon];
    [defaults removeObserver:self forKeyPath:SRIconUseFaviconBookmarkBar];
    
	[[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [_srTabView setDelegate:nil];
    [[[self window] toolbar] setDelegate:nil];
    [[self searchField] setTarget:nil];
    [[self searchFieldOnSheet] setTarget:nil];
    [[self URLComboBox] setDelegate:nil];
    [[self URLComboBox] setTarget:nil];
    [[self URLComboBoxOnSheet] setDelegate:nil];
    [[self URLComboBoxOnSheet] setTarget:nil];
    
    int count, i;
    count = [_srTabView numberOfItems];
    for (i = 0; i < count; i++) {
        WebView*    webView;
        webView = (WebView*)[_srTabView viewAtIndex:i];
        [webView setFrameLoadDelegate:nil];
        [webView setResourceLoadDelegate:nil];
        [webView setPolicyDelegate:nil];
        [webView setUIDelegate:nil];
    }
    
    [super dealloc];
}

#pragma mark -
//--------------------------------------------------------------//
// Controller
//--------------------------------------------------------------//

- (SRSideBarController*)sideBarController
{
    return _sideBarController;
}

#pragma mark -
//--------------------------------------------------------------//
// Outlets
//--------------------------------------------------------------//

- (void)_loadDialogs
{
    // Load MainWindowDialogs.nib
    if (![NSBundle loadNibNamed:@"MainWindowDialogs" owner:self]) {
        // Fatal error
        SR_FATAL(@"Could not load MainWindowDialogs.nib");
    }
}

- (id)addBookmarkPanel
{
    if (!_addBookmarkPanel) {
        [self _loadDialogs];
    }
    return _addBookmarkPanel;
}

- (id)addBookmarkTextField
{
    if (!_addBookmarkTextField) {
        [self _loadDialogs];
    }
    return _addBookmarkTextField;
}

- (id)addBookmarkPopup
{
    if (!_addBookmarkPopup) {
        [self _loadDialogs];
    }
    return _addBookmarkPopup;
}

- (id)bookmarkAllPanel
{
    if (!_bookmarkAllPanel) {
        [self _loadDialogs];
    }
    return _bookmarkAllPanel;
}

- (id)bookmarkAllPopup
{
    if (!_bookmarkAllPopup) {
        [self _loadDialogs];
    }
    return _bookmarkAllPopup;
}

- (id)newFolderPanel
{
    if (!_newFolderPanel) {
        [self _loadDialogs];
    }
    return _newFolderPanel;
}

- (id)newFolderTextField
{
    if (!_newFolderTextField) {
        [self _loadDialogs];
    }
    return _newFolderTextField;
}

- (id)addToBarPanel
{
    if (!_addToBarPanel) {
        [self _loadDialogs];
    }
    return _addToBarPanel;
}

- (id)addToBarTextField
{
    if (!_addToBarTextField) {
        [self _loadDialogs];
    }
    return _addToBarTextField;
}

- (id)locationPanel
{
    if (!_locationPanel) {
        [self _loadDialogs];
    }
    return _locationPanel;
}

- (id)locationURLView
{
    if (!_locationURLView) {
        [self _loadDialogs];
    }
    return _locationURLView;
}

- (id)searchPanel
{
    if (!_searchPanel) {
        [self _loadDialogs];
    }
    return _searchPanel;
}

- (id)searchFieldView
{
    if (!_searchFieldView) {
        [self _loadDialogs];
    }
    return _searchFieldView;
}

#pragma mark -
//--------------------------------------------------------------//
// Web view and toolbar items
//--------------------------------------------------------------//

- (WebView*)selectedWebView
{
    id  view;
    view = [_srTabView viewOfSelectedTabViewItem];
    if (!view) {
        return nil;
    }
    SR_VERIFY([view isKindOfClass:[WebView class]]);
    
    return view;
}

- (SRURLComboBox*)URLComboBox
{
    // Get URL combo box from toolbar items
    NSEnumerator*   enumerator;
    NSToolbarItem*  toolbarItem;
    enumerator = [[[[self window] toolbar] visibleItems] objectEnumerator];
    while (toolbarItem = [enumerator nextObject]) {
        if ([[toolbarItem itemIdentifier] isEqualToString:SRURLIdentifier]) {
            id  view;
            view = [toolbarItem view];
            SR_VERIFY([view isKindOfClass:[SRURLComboBox class]]);
            
            return view;
        }
    }
    
    return nil;
}

- (SRURLComboBox*)URLComboBoxOnSheet
{
    // Return first children of location URL view
    NSArray*    subviews;
    subviews = [[self locationURLView] subviews];
    
    if ([subviews count] < 1) {
        return nil;
    }
    
    id  view;
    view = [subviews objectAtIndex:0];
    SR_VERIFY([view isKindOfClass:[SRURLComboBox class]]);
    
    return view;
}

- (SRSearchField*)searchField
{
    // Get search field from toolbar items
    NSEnumerator*   enumerator;
    NSToolbarItem*  toolbarItem;
    enumerator = [[[[self window] toolbar] visibleItems] objectEnumerator];
    while (toolbarItem = [enumerator nextObject]) {
        if ([[toolbarItem itemIdentifier] isEqualToString:SRSearchIdentifier]) {
            id  view;
            view = [toolbarItem view];
            SR_VERIFY([view isKindOfClass:[SRSearchField class]]);
            
            return view;
        }
    }
    
    return nil;
}

- (SRSearchField*)searchFieldOnSheet
{
    // Return first children of search field view
    NSArray*    subviews;
    subviews = [[self searchFieldView] subviews];
    
    if ([subviews count] < 1) {
        return nil;
    }
    id  view;
    view = [subviews objectAtIndex:0];
    SR_VERIFY([view isKindOfClass:[SRSearchField class]]);
    
    return view;
}

#pragma mark -
//--------------------------------------------------------------//
// Tab management
//--------------------------------------------------------------//

- (SRTabView*)srTabView
{
    return _srTabView;
}

- (WebView*)addNewTabWithLabel:(NSString*)label
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName
{
    return [self addNewTabWithLabel:label 
            frameName:frameName 
            groupName:groupName 
            select:[[NSUserDefaults standardUserDefaults] boolForKey:SRTabEnableTabbedBrowsing]];
}

- (WebView*)addNewTabWithLabel:(NSString*)label 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        select:(BOOL)select
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Create WebView
    WebView*    webView;
    webView = [[WebView alloc] initWithFrame:[[_srTabView tabView] frame] 
            frameName:frameName groupName:groupName];
    [webView autorelease];
    [webView setHostWindow:[self window]];  // For Plug-in, JavaScript
    [webView setFrameLoadDelegate:self];
    [webView setResourceLoadDelegate:self];
    [webView setPolicyDelegate:self];
    [webView setUIDelegate:self];
    [webView setDownloadDelegate:[SRDownloadCenter sharedInstance]];
    [webView setPreferencesIdentifier:SRWebPreferencesIdentifier];
//    [[webView mainFrame] _setShouldCreateRenderers:select];
    
    // Set user agent
    [self _setUserAgentOfWebView:webView];
    
// For force custom encoding for Japanese character code
#if 0
    // Get default encoding
    NSStringEncoding    encoding;
    encoding = [[defaults objectForKey:SRDefaultTextEncoding] unsignedIntValue];
    
    // For Jpanese encoding
    if (encoding == SRConvertedShiftJISStringEncoding || 
        encoding == SRConvertedJapaneseEUCStringEncoding || 
        encoding == SRConvertedISO2022JPStringEncoding)
    {
        // Set it as custom web view encoding
        NSValueTransformer* transformer;
        NSString*           encodingName;
        transformer = [NSValueTransformer valueTransformerForName:SRIANAToEncodingTransformerName];
        encodingName = [transformer reverseTransformedValue:
                [NSNumber numberWithUnsignedInt:encoding]];
        [webView setCustomTextEncodingName:encodingName];
    }
#endif
    
    // Dummy
    WebPreferences* pref;
    int             defaultFontSize;
    pref = [webView preferences];
    defaultFontSize = [pref defaultFontSize];
    [pref setDefaultFontSize:defaultFontSize + 1];
    [pref setDefaultFontSize:defaultFontSize];
    
    // Set icon
    if (select) {
        [[self URLComboBox] setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
    }
    
    // Add new tab and select
    [_srTabView addItemWithLabel:label view:webView select:select];
    
    return webView;
}

- (void)selectLeftTab
{
    // Get selected index
    int index;
    index = [_srTabView selectedIndex];
    
    // Select left tab
    if (index == 0) {
        [_srTabView selectItemAtIndex:[_srTabView numberOfItems] - 1];
    }
    else {
        [_srTabView selectItemAtIndex:index - 1];
    }
}

- (void)selectRightTab
{
    // Get selected index
    int index;
    index = [_srTabView selectedIndex];
    
    // Select right tab
    if (index == [_srTabView numberOfItems] - 1) {
        [_srTabView selectItemAtIndex:0];
    }
    else {
        [_srTabView selectItemAtIndex:index + 1];
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Opening page with request
//--------------------------------------------------------------//

- (void)_openRequest:(NSURLRequest*)request atWebView:(WebView*)webView
{
    // Do JavaScript
    NSString*   URLString;
    URLString = [[request URL] absoluteString];
    if ([URLString hasPrefix:@"javascript:"]) {
        NSString*   script;
        NSString*   result;
        script = [URLString substringWithRange:NSMakeRange(11, [URLString length] - 11)]; // Remove 'javascript:'
        script = [script stringByReplacingPercentEscapesUsingEncoding:NSUTF8StringEncoding];
        result = [webView stringByEvaluatingJavaScriptFromString:script];
        
        return;
    }
    
    // Open request
    [[webView mainFrame] loadRequest:request];
    
    // Set icon
    if ([self selectedWebView] == webView) {
        [[self URLComboBox] setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
    }
}

- (WebView*)openRequest:(NSURLRequest*)request
{
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Open request
    [self _openRequest:request atWebView:webView];
    
    return webView;
}

- (WebView*)openRequest:(NSURLRequest*)request atIndex:(int)index
{
    // Get selected web view
    WebView*    webView;
    webView = (WebView*)[_srTabView viewAtIndex:index];
    
    // Open request
    [self _openRequest:request atWebView:webView];
    
    return webView;
}

- (id)openRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        withOpenAction:(SROpenActionType)openAction
{
    switch (openAction) {
    case SROpenAction: {
        return [self openRequest:request];
    }
    case SROpenInNewTabAction: {
        return [self openInNewTabRequest:request frameName:frameName groupName:groupName select:YES];
    }
    case SROpenInNewBackgroundTabAction: {
        return [self openInNewTabRequest:request frameName:frameName groupName:groupName select:NO];
    }
    case SROpenInNewWindowAction: {
        return [self openInNewWindowRequest:request frameName:frameName groupName:groupName];
    }
    case SROpenInNewBackgroundWindowAction: {
        return [self openInNewBackgroundWindowRequest:request frameName:frameName groupName:groupName];
    }
    case SROpenOptionAction: {
        // Download it
        SRDownloadCenter*   downloadCenter;
        NSURLDownload*      download;
        downloadCenter = [SRDownloadCenter sharedInstance];
        download = [[NSURLDownload alloc] initWithRequest:request delegate:downloadCenter];
        [download autorelease];
        
        return download;
    }
    }
    
    // Never reach here
    return nil;
}

- (WebView*)openInNewTabRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName
{
    return [self openInNewTabRequest:request 
            frameName:frameName 
            groupName:groupName 
            select:[[NSUserDefaults standardUserDefaults] boolForKey:SRTabSelectNewTabs]];
}

- (WebView*)openInNewTabRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName 
        select:(BOOL)select
{
    // Add new tab
    WebView*    webView;
    webView = [self addNewTabWithLabel:NSLocalizedString(@"Untitled", @"Untitled") 
            frameName:frameName 
            groupName:groupName 
            select:select];
    
    // Open request
    [self _openRequest:request atWebView:webView];
    
    return webView;
}

- (SRMainDocument*)openInNewWindowRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName
{
    // Create new document
    SRDocumentController*   documentController;
    SRMainDocument*         document;
    documentController = [NSDocumentController sharedDocumentController];
    document = [documentController makeUntitledDocumentOfType:SRHTMLDocumentType];
    [documentController addDocument:document];
    
    // Make window controllers
    [document makeWindowControllersWithInitialFrameName:frameName groupName:groupName];
    
    // Get main window controller
    SRMainWindowController* mainWindowController;
    mainWindowController = [document mainWindowController];
    
    // Get window and make it front
    [[mainWindowController window] makeKeyAndOrderFront:self];
    
    // Open request in new window
    [mainWindowController openRequest:request];
//    [document showWindows];
    
    return document;
}

- (SRMainDocument*)openInNewBackgroundWindowRequest:(NSURLRequest*)request 
        frameName:(NSString*)frameName 
        groupName:(NSString*)groupName
{
    // Create new document
    SRDocumentController*   documentController;
    SRMainDocument*         document;
    documentController = [NSDocumentController sharedDocumentController];
    document = [documentController makeUntitledDocumentOfType:SRHTMLDocumentType];
    [documentController addDocument:document];
    
    // Make window controllers
    [document makeWindowControllersWithInitialFrameName:frameName groupName:groupName];
    
    // Get main window controller
    SRMainWindowController* mainWindowController;
    mainWindowController = [document mainWindowController];
    
    // Get window and make it back
    [[mainWindowController window] orderWindow:NSWindowBelow relativeTo:[[self window] windowNumber]];
    
    // Open bookmark in new window
    [mainWindowController openRequest:request];
//    [document showWindows];
    
    return document;
}

#pragma mark -
//--------------------------------------------------------------//
// Opening page with URL string
//--------------------------------------------------------------//

- (NSURLRequest*)_createRequestWithURL:(NSURL*)url
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Create request
    return [NSURLRequest requestWithURL:url 
            cachePolicy:NSURLRequestUseProtocolCachePolicy 
            timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
}

- (NSURLRequest*)_createRequestWithURLString:(NSString*)URLString
{
    return [self _createRequestWithURL:[NSURL _web_URLWithUserTypedString:URLString]];
}

- (WebView*)openURLString:(NSString*)URLString
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURLString:URLString]];
}

- (WebView*)openURLString:(NSString*)URLString atIndex:(int)index
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURLString:URLString] atIndex:index];
}

- (id)openURLString:(NSString*)URLString withOpenAction:(SROpenActionType)openAction
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURLString:URLString] 
            frameName:nil 
            groupName:nil 
            withOpenAction:openAction];
}

- (WebView*)openInNewTabURLString:(NSString*)URLString
{
    // Create and open request
    return [self openInNewTabRequest:[self _createRequestWithURLString:URLString] 
            frameName:nil 
            groupName:nil];
}

- (WebView*)openInNewTabURLString:(NSString*)URLString select:(BOOL)select
{
    // Create and open request
    return [self openInNewTabRequest:[self _createRequestWithURLString:URLString] 
            frameName:nil 
            groupName:nil 
            select:select];
}

- (SRMainDocument*)openInNewWindowURLString:(NSString*)URLString
{
    // Create and open request
    return [self openInNewWindowRequest:[self _createRequestWithURLString:URLString] 
            frameName:nil 
            groupName:nil];
}

- (SRMainDocument*)openInNewBackgroundWindowURLString:(NSString*)URLString
{
    // Create and open request
    return [self openInNewBackgroundWindowRequest:[self _createRequestWithURLString:URLString] 
            frameName:nil 
            groupName:nil];
}

#pragma mark -
//--------------------------------------------------------------//
// Opening page with URL
//--------------------------------------------------------------//

- (WebView*)openURL:(NSURL*)url
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURL:url]];
}

- (WebView*)openURL:(NSURL*)url atIndex:(int)index
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURL:url] atIndex:index];
}

- (id)openURL:(NSURL*)url withOpenAction:(SROpenActionType)openAction
{
    // Create and open request
    return [self openRequest:[self _createRequestWithURL:url] 
            frameName:nil 
            groupName:nil 
            withOpenAction:openAction];
}

- (WebView*)openInNewTabURL:(NSURL*)url
{
    // Create and open request
    return [self openInNewTabRequest:[self _createRequestWithURL:url] 
            frameName:nil 
            groupName:nil];
}

- (WebView*)openInNewTabURL:(NSURL*)url select:(BOOL)select
{
    // Create and open request
    return [self openInNewTabRequest:[self _createRequestWithURL:url] 
            frameName:nil 
            groupName:nil 
            select:select];
}

- (SRMainDocument*)openInNewWindowURL:(NSURL*)url
{
    // Create and open request
    return [self openInNewWindowRequest:[self _createRequestWithURL:url] 
            frameName:nil 
            groupName:nil];
}

- (SRMainDocument*)openInNewBackgroundWindowURL:(NSURL*)url
{
    // Create and open request
    return [self openInNewBackgroundWindowRequest:[self _createRequestWithURL:url] 
            frameName:nil 
            groupName:nil];
}

#pragma mark -
//--------------------------------------------------------------//
// Opening page with bookmark
//--------------------------------------------------------------//

- (WebView*)openBookmark:(SRBookmark*)bookmark
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openURLString:[bookmark URLString]];
    }
    
    return nil;
}

- (WebView*)openBookmark:(SRBookmark*)bookmark atIndex:(int)index
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openURLString:[bookmark URLString] atIndex:(int)index];
    }
    
    return nil;
}

- (id)openBookmark:(SRBookmark*)bookmark withOpenAction:(SROpenActionType)openAction
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openURLString:[bookmark URLString] withOpenAction:openAction];
    }
    
    return nil;
}

- (WebView*)openInNewTabBookmark:(SRBookmark*)bookmark
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openInNewTabURLString:[bookmark URLString]];
    }
    
    return nil;
}

- (WebView*)openInNewTabBookmark:(SRBookmark*)bookmark select:(BOOL)select
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openInNewTabURLString:[bookmark URLString] select:select];
    }
    
    return nil;
}

- (SRMainDocument*)openInNewWindowBookmark:(SRBookmark*)bookmark
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openInNewWindowURLString:[bookmark URLString]];
    }
    
    return nil;
}

- (SRMainDocument*)openInNewBackgroundWindowBookmark:(SRBookmark*)bookmark
{
    // For HTML bookmark
    if ([bookmark type] == SRBookmarkTypeHTML) {
        // Open URL string
        return [self openInNewBackgroundWindowURLString:[bookmark URLString]];
    }
    
    return nil;
}

- (NSArray*)openInTabsBookmark:(SRBookmark*)bookmark
{
    // For bookmark folder
    if ([bookmark isFolderType]) {
        // Open children
        return [self openInTabsBookmarks:[bookmark children]];
    }
    
    return nil;
}

- (NSArray*)openInTabsBookmarks:(NSArray*)bookmarks
{
    // Check bookmarks
    if (!bookmarks || [bookmarks count] == 0) {
        return nil;
    }
    
    // Get current tab number
    int numberOfTab;
    numberOfTab = [_srTabView numberOfItems];
    
    // Enumerate bookmark
    NSMutableArray* webViews;
    NSEnumerator*   enumerator;
    SRBookmark*     bookmark;
    WebView*        webView;
    int             index = 0;
    webViews = [NSMutableArray arrayWithCapacity:numberOfTab];
    enumerator = [bookmarks objectEnumerator];
    while (bookmark = [enumerator nextObject]) {
        if (index < numberOfTab) {
            // Open bookmark in tab
            webView = [self openBookmark:bookmark atIndex:index];
        }
        else {
            // Open bookmark in new tab
            webView = [self openInNewTabBookmark:bookmark select:NO];
        }
        if (webView) {
            [webViews addObject:webView];
            index++;
        }
    }
    
    // Select first tab
    [_srTabView selectItemAtIndex:0];
    
    return webViews;
}

#pragma mark -
//--------------------------------------------------------------//
// Opening page with history
//--------------------------------------------------------------//

- (WebView*)openHistory:(WebHistoryItem*)historyItem
{
    // Open URL string
    return [self openURLString:[historyItem URLString]];
}

- (WebView*)openHistory:(WebHistoryItem*)historyItem atIndex:(int)index
{
    // Open URL string
    return [self openURLString:[historyItem URLString] atIndex:index];
}

- (id)openHistory:(WebHistoryItem*)historyItem withOpenAction:(SROpenActionType)openAction
{
    // Open URL string
    return [self openURLString:[historyItem URLString] withOpenAction:openAction];
}

- (WebView*)openInNewTabHistory:(WebHistoryItem*)historyItem
{
    // Open URL string
    return [self openInNewTabURLString:[historyItem URLString]];
}

- (WebView*)openInNewTabHistory:(WebHistoryItem*)historyItem select:(BOOL)select
{
    // Open URL string
    return [self openInNewTabURLString:[historyItem URLString] select:select];
}

- (SRMainDocument*)openInNewWindowHistory:(WebHistoryItem*)historyItem
{
    // Open URL string
    return [self openInNewWindowURLString:[historyItem URLString]];
}

- (SRMainDocument*)openInNewBackgroundWindowHistory:(WebHistoryItem*)historyItem
{
    // Open URL string
    return [self openInNewBackgroundWindowURLString:[historyItem URLString]];
}

- (NSArray*)openInTabsHistory:(NSArray*)historyItems
{
    // Check history items
    if (!historyItems || [historyItems count] == 0) {
        return nil;
    }
    
    // Get current tab number
    int numberOfTab;
    numberOfTab = [_srTabView numberOfItems];
    
    // Enumerate bookmark
    NSMutableArray* webViews;
    NSEnumerator*   enumerator;
    WebHistoryItem* historyItem;
    WebView*        webView;
    int             index = 0;
    webViews = [NSMutableArray arrayWithCapacity:numberOfTab];
    enumerator = [historyItems objectEnumerator];
    while (historyItem = [enumerator nextObject]) {
        if (![historyItem isKindOfClass:[WebHistoryItem class]]) {
            continue;
        }
        
        if (index < numberOfTab) {
            // Open history item in tab
            webView = [self openHistory:historyItem atIndex:index];
        }
        else {
            // Open history item in new tab
            webView = [self openInNewTabHistory:historyItem select:NO];
        }
        [webViews addObject:webView];
        
        index++;
    }
    
    // Select first tab
    [_srTabView selectItemAtIndex:0];
    
    return webViews;
}

#pragma mark -
//--------------------------------------------------------------//
// Realod URL
//--------------------------------------------------------------//

// from openURL
- (BOOL)reloadURLIfExists:(NSURL*)URL
{
    int i,cnt;
    NSArray*    tabViewItems=[[_srTabView tabView]tabViewItems];
    cnt=[tabViewItems count];
    for(i=0;i<cnt;i++){
        WebDataSource*  dataSource;
        NSURLRequest*   request;
        WebView*    webView;
        NSURL*      aURL;
        
        webView = (WebView*)[[tabViewItems objectAtIndex:i]view];
        if([webView isMemberOfClass:[WebView class]]){
            dataSource = [[webView mainFrame] dataSource];
            request = [dataSource request];
            aURL=[request URL];
            if([URL isEqual:aURL]){
                //use existed WebView
                [_srTabView selectItemAtIndex:i];
                [webView reload:self];
                [self showWindow:self];
                return YES;
            }
        }
    }
    return NO;
}

#pragma mark -
//--------------------------------------------------------------//
// Appearnace
//--------------------------------------------------------------//

- (BOOL)isBookmarksBarVisible
{
    return ![_bookmarksBarView isHidden];
}

- (void)setBookmarksBarVisible:(BOOL)isVisible 
        display:(BOOL)display
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Show bookmarks bar
    if (isVisible && [_bookmarksBarView isHidden]) {
        // Show bookmarks box
        [_bookmarksBarView setHidden:NO];
        
        // Get frames
        NSRect  barFrame;
        NSRect  windowFrame;
        NSRect  parentFrame;
        barFrame = [_bookmarksBarView frame];
        windowFrame = [[self window] frame];
        parentFrame = [_parentView frame];
        
        // Expand window
        int mask;
        mask = [_parentView autoresizingMask];
        [_parentView setAutoresizingMask:NSViewMaxYMargin];
        [[self window] setFrame:NSMakeRect(
                windowFrame.origin.x, 
                windowFrame.origin.y - barFrame.size.height, 
                windowFrame.size.width, 
                windowFrame.size.height + barFrame.size.height) display:display animate:YES]; 
        [_parentView setAutoresizingMask:mask];
    }
    
    // Hide bookmarks bar
    else if (!isVisible && ![_bookmarksBarView isHidden]) {
        // Get frames
        NSRect  barFrame;
        NSRect  windowFrame;
        NSRect  parentFrame;
        barFrame = [_bookmarksBarView frame];
        windowFrame = [[self window] frame];
        parentFrame = [_parentView frame];
        
        // Shrink window
        int mask;
        mask = [_parentView autoresizingMask];
        [_parentView setAutoresizingMask:NSViewMaxYMargin];
        [[self window] setFrame:NSMakeRect(
                windowFrame.origin.x, 
                windowFrame.origin.y + barFrame.size.height, 
                windowFrame.size.width, 
                windowFrame.size.height - barFrame.size.height) display:display animate:YES]; 
        [_parentView setAutoresizingMask:mask];
        
        // Hide bookmarks box
        [_bookmarksBarView setHidden:YES];
    }
    
    // Show or hide separator of tab view
    [_srTabView setSeparatorShow:isVisible];
    
    // Set it default database
    if (!_isResizedByJavaScript) {
        [defaults setBool:isVisible forKey:SRMainWindowBookmarksBarVisible];
    }
}

- (BOOL)isStatusBarVisible
{
    return ![_statusBox isHidden];
}

- (void)setStatusBarVisible:(BOOL)isVisible 
        display:(BOOL)display
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Show status bar
    if (isVisible && [_statusBox isHidden]) {
        // Show status box
        [_statusBox setHidden:NO];
        
        // Get frames
        NSRect  boxFrame;
        NSRect  windowFrame;
        NSRect  parentFrame;
        boxFrame = [_statusBox frame];
        windowFrame = [[self window] frame];
        parentFrame = [_parentView frame];
        
        // Expand window
        int mask;
        mask = [_parentView autoresizingMask];
        [_parentView setAutoresizingMask:NSViewMinYMargin];
        [[self window] setFrame:NSMakeRect(
                windowFrame.origin.x, 
                windowFrame.origin.y - boxFrame.size.height, 
                windowFrame.size.width, 
                windowFrame.size.height + boxFrame.size.height) display:display animate:YES]; 
        [_parentView setAutoresizingMask:mask];
    }
    
    // Hide status bar
    else if (!isVisible && ![_statusBox isHidden]) {
        // Get frames
        NSRect  boxFrame;
        NSRect  windowFrame;
        NSRect  parentFrame;
        boxFrame = [_statusBox frame];
        windowFrame = [[self window] frame];
        parentFrame = [_parentView frame];
        
        // Shrink window
        int mask;
        mask = [_parentView autoresizingMask];
        [_parentView setAutoresizingMask:NSViewMinYMargin];
        [[self window] setFrame:NSMakeRect(
                windowFrame.origin.x, 
                windowFrame.origin.y + boxFrame.size.height, 
                windowFrame.size.width, 
                windowFrame.size.height - boxFrame.size.height) display:display animate:YES]; 
        [_parentView setAutoresizingMask:mask];
        
        // Hide status box
        [_statusBox setHidden:YES];
    }
    
    // Set it default database
    if (!_isResizedByJavaScript) {
        [defaults setBool:isVisible forKey:SRMainWindowStatusBarVisible];
    }
}

- (BOOL)isTabVisible
{
    return [_srTabView isTabShown];
}

- (void)setTabVisible:(BOOL)isVisible 
        display:(BOOL)display
{
    [_srTabView setTabShown:isVisible];
}

- (BOOL)isSidebarVisible
{
    int state;
    state = [[_sideBarController drawer] state];
    return state == NSDrawerOpenState || state == NSDrawerOpeningState;
}

- (void)setSidebarVisible:(BOOL)isVisible
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    NSDrawer*   drawer;
    drawer = [_sideBarController drawer];
    
    // Set visible status
    if (isVisible) {
        BOOL    usePreferredEdge;
        int     edge;
        usePreferredEdge = [defaults boolForKey:SRSidebarPreferredEdge];
        edge = [defaults integerForKey:SRSidebarEdge];
        if (edge == SRSidebarEdgeLeft) {
            if (usePreferredEdge) {
                [drawer setPreferredEdge:NSMaxXEdge];
                [drawer open];
            }
            else {
                [drawer openOnEdge:NSMinXEdge];
            }
        }
        else if (edge == SRSidebarEdgeRight) {
            if (usePreferredEdge) {
                [drawer setPreferredEdge:NSMinXEdge];
                [drawer open];
            }
            else {
                [drawer openOnEdge:NSMaxXEdge];
            }
        }
        else {
            [drawer open];
        }
    }
    else {
        [drawer close];
    }
    
#if 0
    // Set it default database
    if (!_isResizedByJavaScript) {
        [defaults setBool:isVisible forKey:SRMainWindowSidebarVisible];
    }
#endif
}

- (BOOL)isResizedByJavaScript
{
    return _isResizedByJavaScript;
}

#pragma mark -
//--------------------------------------------------------------//
// Supported schemes
//--------------------------------------------------------------//

- (BOOL)isSchemeSupported:(NSString *)scheme
{
    // For javascript window.open
    if(!scheme || [scheme length]<=0){
        return YES;
    }

    return [scheme isEqualToString:@"http"] || 
           [scheme isEqualToString:@"https"] || 
           [scheme isEqualToString:@"file"];
}

#pragma mark -
//--------------------------------------------------------------//
// Bookmarks bar
//--------------------------------------------------------------//

- (void)setBookmarksBarBrowser:(int)browser
{
    if (_bookmarksBarBrowser != browser) {
        _bookmarksBarBrowser = browser;
        
        // Find bookmarks bar
        SRBookmark* bookmarksBar = nil;
        
        switch(_bookmarksBarBrowser) {
        case SRBrowserShiira: {
            bookmarksBar = [[SRBookmarkStorage sharedInstance] shiiraBookmarksBar];
            break;
        }
        case SRBrowserSafari: {
            bookmarksBar = [[SRBookmarkStorage sharedInstance] safariBookmarksBar];
            break;
        }
        }
        
        if (bookmarksBar) {
            // Set accept drag
            [_bookmarksBarView setAcceptDrag:[bookmarksBar isMutable]];
            
            // Set bookmarks bar
            [_bookmarksBarView setBookmarksBar:bookmarksBar];
#if 0
            // Clear all bookmarks
            [_bookmarksBarView removeAllBookmarks];
            
            // Set bookmarks
            NSArray*        children;
            NSEnumerator*   enumerator;
            SRBookmark*     bookmark;
            children = [bookmarksBar children];
            enumerator = [children objectEnumerator];
            while (bookmark = [enumerator nextObject]) {
                [_bookmarksBarView addBookmark:bookmark];
            }
            
            // Set accept drag
            [_bookmarksBarView setAcceptDrag:[bookmarksBar isMutable]];
            
            // Set bookmarks bar
            [_bookmarksBarView setBookmarksBar:bookmarksBar];
#endif
        }
    }
}

#pragma mark -
//--------------------------------------------------------------//
// User agent
//--------------------------------------------------------------//

- (void)_setUserAgentOfWebView:(WebView*)webView
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Set user agent
    int         userAgent;
    NSString*   customUserAgentName;
    userAgent = [defaults integerForKey:SRUserAgent];
    customUserAgentName = [defaults stringForKey:SRUserAgentOtherName];
    
    // For Shiira user agent
    if (userAgent == SRUserAgentShiira) {
        [webView setCustomUserAgent:nil];
        
        // Get version
        NSBundle*   bundle;
        NSString*   version;
        NSString*   userAgentName = @"Shiira";
        bundle = [NSBundle mainBundle];
        version = [[bundle infoDictionary] objectForKey:@"CFBundleShortVersionString"];
        if (version) {
            userAgentName = [NSString stringWithFormat:@"%@/%@", userAgentName, version];
        }
        [webView setApplicationNameForUserAgent:userAgentName];
        
        // Get composited user agent name
        NSString*   compUserAgentName;
        compUserAgentName = [webView userAgentForURL:[NSURL URLWithString:@"http://local"]];
        
        // If user agent name dose not contain 'Safari', add it after 'Gecko'
        if ([compUserAgentName rangeOfString:@"Safari"].location == NSNotFound) {
            NSMutableString*    string;
            NSRange             range;
            string = [NSMutableString stringWithString:compUserAgentName];
            range = [compUserAgentName rangeOfString:@"Gecko"];
            if (range.location != NSNotFound) {
                [string insertString:@", Safari" atIndex:range.location + range.length];
            }
            else {
                [string insertString:@"(like Safari)" atIndex:[string length]];
            }
            compUserAgentName = string;
        }
        [webView setCustomUserAgent:compUserAgentName];
        
        // Show user agent name on preferences panel
        [defaults setValue:compUserAgentName forKey:SRUserAgentShiiraName];
    }
    // For default user agents
    if (userAgent > SRUserAgentShiira && userAgent < SRUserAgentOther) {
        // Get user agents plist
        NSString*   userAgentsPath;
        NSArray*    userAgents;
        userAgentsPath = [[NSBundle mainBundle] pathForResource:@"UserAgent" ofType:@"plist"];
        userAgents = [NSArray arrayWithContentsOfFile:userAgentsPath];
        if (userAgents && [userAgents count] > userAgent - 1) {
            NSDictionary*   userAgentInfo;
            NSString*       userAgentName;
            userAgentInfo = [userAgents objectAtIndex:userAgent - 1];
            userAgentName = [userAgentInfo objectForKey:@"userAgent"];
            if (userAgentName) {
                [webView setCustomUserAgent:userAgentName];
                
                // Show user agent name on preferences panel
                [defaults setValue:userAgentName forKey:SRUserAgentShiiraName];
            }
        }
    }
    // For other
    if (userAgent == SRUserAgentOther) {
        [webView setCustomUserAgent:customUserAgentName];
        
        // Set other user agent name on preferences panel
        [defaults setValue:customUserAgentName forKey:SRUserAgentShiiraName];
    }
}

- (void)setUserAgent:(int)userAgent
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get custom user agent name
    NSString*   customUserAgentName;
    customUserAgentName = [defaults stringForKey:SRUserAgentOtherName];
    
    // Set it web view
    NSEnumerator*   enumerator;
    NSTabViewItem*  tabViewItem;
    enumerator = [[[_srTabView tabView] tabViewItems] objectEnumerator];
    while (tabViewItem = [enumerator nextObject]) {
        // Get web view
        WebView*    webView;
        webView = (WebView*)[tabViewItem view];
        
        // Set user agent
        [self _setUserAgentOfWebView:webView];
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Icon and texture
//--------------------------------------------------------------//

- (void)setTexture:(int)texture
{
}

#pragma mark -
//--------------------------------------------------------------//
// Authentication
//--------------------------------------------------------------//

- (SRAuthenticationController*)authenticationController
{
    if (!_authController) {
        // Load AuthenticationPanel.nib
        _authController = [[SRAuthenticationController alloc] initWithWindowNibName:@"AuthenticationPanel"];
        if (!_authController) {
            // Fatal
            SR_FATAL(@"Could not load AuthenticationPanel.nib");
        }
    }
    
    return _authController;
}

#pragma mark -
//--------------------------------------------------------------//
// Actions
//--------------------------------------------------------------//

- (void)setTextEncodingAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(tag)]) {
        // Get web view
        WebView*    webView;
        webView = [self selectedWebView];
        
        // Get encoding value
        int encoding;
        encoding = [sender tag];
        
        // For default
        if (encoding == SRTextEncodingDefaultTag) {
            [webView setCustomTextEncodingName:nil];
            return;
        }
        
        // Get encoding name
        NSString*   encodingName;
        encodingName = [[NSValueTransformer valueTransformerForName:SREncodingToIANATransformerName] 
                transformedValue:[NSNumber numberWithInt:encoding]];
        if (encodingName) {
            [webView setCustomTextEncodingName:encodingName];
        }
    }
}

- (void)goBackAction:(id)sender
{
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Send action to web view
    [webView goBack:sender];
}

- (void)goForwardAction:(id)sender
{
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Send action to web view
    [webView goForward:sender];
}

- (void)goHomeAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get home page URL
    NSString*       URLString;
    URLString = [defaults stringForKey:SRGeneralHomePgae];
    if (!URLString || [URLString length] == 0) {
        return;
    }
    
    [self openURLString:URLString];
}

- (void)showHistoryInSidebarAction:(id)sender
{
    if(([[_sideBarController drawer] state] == NSDrawerOpenState) && 
       ([_sideBarController selectedTab] == SRSidebarHistoryTab))
    {
        [[_sideBarController drawer] close];
    }
    else {
        // Show history
        [_sideBarController showTab:SRSidebarHistoryTab];
        
        // Show sidebar
        [[_sideBarController drawer] open];
    }
}

- (void)openHistoryItemAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            // Check in back forward list
            WebView*            webView;
            WebBackForwardList* list;
            webView = [self selectedWebView];
            list = [webView backForwardList];
            if ([list containsItem:representedObject]) {
                // Go to item
                [webView goToBackForwardItem:representedObject];
            }
            else {
                // Open URL
                [self openURLString:[representedObject URLString]];
            }
        }
    }
}

- (void)showBookmarkInSidebarAction:(id)sender
{
    if(([[_sideBarController drawer] state] == NSDrawerOpenState) && 
       ([_sideBarController selectedTab] == SRSidebarBookmarkTab))
    {
        [[_sideBarController drawer] close];
    }
    else {
        // Show bookmark
        [_sideBarController showTab:SRSidebarBookmarkTab];
        
        // Show sidebar
        [[_sideBarController drawer] open];
    }
}

- (void)takeStringURLFromAction:(id)sender
{
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Send action to web view
    [webView takeStringURLFrom:sender];
}

- (void)toggleToolbarAction:(id)sender
{
    // Toogle toolbar
    [[self window] toggleToolbarShown:sender];
}

- (void)toggleBookmarksBarAction:(id)sender
{
    // Toggle bookmarks bar
    [self setBookmarksBarVisible:![self isBookmarksBarVisible] display:YES];
}

- (void)toggleStatusBarAction:(id)sender
{
    // Toggle status bar
    [self setStatusBarVisible:![self isStatusBarVisible] display:YES];
}

- (void)toggleSidebarAction:(id)sender
{
    // Toogle sidebar
    [self setSidebarVisible:![self isSidebarVisible]];
}

- (void)customizeToolbarAction:(id)sender
{
    // Run toolbar customize panel
    [[[self window] toolbar] runCustomizationPalette:self];
}

- (void)stopLoadingAction:(id)sender
{
    // Stop loading
    [[self selectedWebView] stopLoading:sender];
}

- (void)reloadPageAction:(id)sender
{
    // Get modifier key
    unsigned    modifierFlags;
    unsigned    shiftFlag;
    modifierFlags = [[NSApp currentEvent] modifierFlags];
    shiftFlag = modifierFlags & NSShiftKeyMask;
    
    if (!shiftFlag) {
        // Reload page
        [[self selectedWebView] reload:sender];
    }
    else {
        // Reload without cache
        NSURLRequest*   request;
        NSURLRequest*   requestWithoutCache;
        request = [[[[self selectedWebView] mainFrame] dataSource] request];
        requestWithoutCache = [NSURLRequest requestWithURL:[request URL] 
                cachePolicy:NSURLRequestReloadIgnoringCacheData 
                timeoutInterval:[request timeoutInterval]];
        [self openRequest:requestWithoutCache];
    }
}

- (void)biggerTextAction:(id)sender
{
    // Bigger text
    [[self selectedWebView] makeTextLarger:sender];
}

- (void)smallerTextAction:(id)sender
{
    // Smaller text
    [[self selectedWebView] makeTextSmaller:sender];
}

- (void)findByToolbarAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get toolbar
    NSToolbar*  toolbar;
    toolbar = [[self window] toolbar];
    
    // When no search field on toolbar
    if (![toolbar isVisible] || 
        ![self searchField] || 
        [toolbar displayMode] == NSToolbarDisplayModeLabelOnly) 
    {
        if (![self searchFieldOnSheet]) {
            // Create search field
            SRSearchField*  searchFieldOnSheet;
            searchFieldOnSheet = [[SRSearchField alloc] initWithFrame:NSMakeRect(0, 0, 0, 0)];
            [searchFieldOnSheet autorelease];
#if 1
            [[searchFieldOnSheet cell] setControlSize:NSSmallControlSize];
            [searchFieldOnSheet setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
#else
            if ([defaults boolForKey:SRToolbarUseSmallURLAndSearchField]) {
                [[searchFieldOnSheet cell] setControlSize:NSSmallControlSize];
                [searchFieldOnSheet setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
            }
            else {
                [[searchFieldOnSheet cell] setControlSize:NSRegularControlSize];
                [searchFieldOnSheet setFont:[NSFont systemFontOfSize:[NSFont systemFontSize]]];
            }
#endif
            [searchFieldOnSheet sizeToFit];
            [searchFieldOnSheet setTag:0];
            [searchFieldOnSheet setTarget:self];
            [searchFieldOnSheet setAction:@selector(closeSearchPanelAction:)];
            [[searchFieldOnSheet cell] setWraps:NO];
            [[searchFieldOnSheet cell] setScrollable:YES];
            [[searchFieldOnSheet cell] setSendsWholeSearchString:YES];
            
            NSRect  frame;
            frame = [[self searchFieldView] frame];
            frame.origin = NSZeroPoint;
            [searchFieldOnSheet setFrame:frame];
            [searchFieldOnSheet setAutoresizingMask:NSViewWidthSizable | NSViewMinYMargin];
            
            [[self searchFieldView] addSubview:searchFieldOnSheet];
        }
        
        // Set search field to controller
        [_searchFieldController setSearchField:[self searchFieldOnSheet]];
        
        // Show search panel
        [NSApp beginSheet:[self searchPanel] 
                modalForWindow:[self window] 
                modalDelegate:self 
                didEndSelector:@selector(_searchPanelDidEnd:returnCode:contextInfo:) 
                contextInfo:NULL];
        
        return;
    }
    
    // Focus on search field
    [[self window] makeFirstResponder:[self searchField]];
}

- (void)closeSearchPanelAction:(id)sender
{
    // End sheet
    [NSApp endSheet:[self searchPanel] returnCode:[sender tag]];
}

- (void)_searchPanelDidEnd:(NSWindow*)sheet 
        returnCode:(int)returnCode 
        contextInfo:(void*)contextInfo
{
    // Close location panel
    [sheet orderOut:self];
    
    // Search
    if (returnCode == 0) {
        [_searchFieldController searchAction:self];
    }
    
    // Restore search field of controller
    [_searchFieldController setSearchField:[self searchField]];
}

- (void)viewPageSourceAction:(id)sender
{
    // Get page source
    WebDataSource*                  dataSource;
    
	//check from context menu
	id  repObj=[sender representedObject];
	if([repObj isKindOfClass:[WebFrame class]]){
		dataSource=[repObj dataSource];
	}else{
		dataSource = [[[self selectedWebView] mainFrame] dataSource];
	}
    if (!dataSource) {
        return;
    }

    // View page source
    SRSourceWindowController*   swc=[SRSourceWindowController sourceWindowWithWebDataSource:dataSource];
    if(swc){
        [swc showWindow:self];
    }
}

- (void)previousTabAction:(id)sender
{
    // Get selected index
    int selectedIndex;
    int numberOfItems;
    selectedIndex = [_srTabView selectedIndex];
    numberOfItems = [_srTabView numberOfItems];
    if (selectedIndex == -1) {
        return;
    }
    
    // Select previous tab
    if (selectedIndex > 0) {
        [_srTabView selectItemAtIndex:selectedIndex - 1];
    }
    else {
        [_srTabView selectItemAtIndex:numberOfItems - 1];
    }
}

- (void)nextTabAction:(id)sender
{
    // Get item number and selected index
    int numberOfItems;
    int selectedIndex;
    numberOfItems = [_srTabView numberOfItems];
    selectedIndex = [_srTabView selectedIndex];
    if (selectedIndex == -1) {
        return;
    }
    
    // Select next tab
    if (numberOfItems > selectedIndex + 1) {
        [_srTabView selectItemAtIndex:selectedIndex + 1];
    }
    else {
        [_srTabView selectItemAtIndex:0];
    }
}

- (void)newTabAction:(id)sender
{
    // Add new tab
    [self addNewTabWithLabel:NSLocalizedString(@"Untitled", @"Untitled") 
            frameName:nil 
            groupName:nil];
}

- (void)closeWindowAction:(id)sender
{
    // Close window
    [[self window] performClose:sender];
}

- (void)closeTabAction:(id)sender
{
    // Close current tab
    if ([_srTabView numberOfItems] < 2) {
        // Do nothing
        return;
    }
    
    // Get selected index
    int selectedIndex;
    selectedIndex = [_srTabView selectedIndex];
    if (selectedIndex < 0) {
        // Do nothing
        return;
    }
    
    // Remove tab
    [_srTabView removeItemAtIndex:selectedIndex];
}

- (void)closeAllRightTabsAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For tab index
        if ([representedObject isKindOfClass:[NSNumber class]]) {
            // Get number of tab items;
            int numberOfTabs;
            numberOfTabs = [_srTabView numberOfItems];
            
            // Get index
            int index = [representedObject intValue];
            if (index < 0 || index > numberOfTabs - 1) {
                return;
            }
            
            // Close all right tabs
            int i;
            for (i = numberOfTabs - 1; i > index; i--) {
                [_srTabView removeItemAtIndex:i];
            }
        }
    }
}

- (void)closeAllLeftTabsAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For tab index
        if ([representedObject isKindOfClass:[NSNumber class]]) {
            // Get number of tab items;
            int numberOfTabs;
            numberOfTabs = [_srTabView numberOfItems];
            
            // Get index
            int index = [representedObject intValue];
            if (index < 0 || index > numberOfTabs - 1) {
                return;
            }
            
            // Close all left tabs
            int i;
            for (i = index - 1; i >=0; i--) {
                [_srTabView removeItemAtIndex:i];
            }
        }
    }
}

- (void)closeAllOtherTabsAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For tab index
        if ([representedObject isKindOfClass:[NSNumber class]]) {
            // Get number of tab items;
            int numberOfTabs;
            numberOfTabs = [_srTabView numberOfItems];
            
            // Get index
            int index = [representedObject intValue];
            if (index < 0 || index > numberOfTabs - 1) {
                return;
            }
            
            // Close all right tabs
            int i;
            for (i = numberOfTabs - 1; i > index; i--) {
                [_srTabView removeItemAtIndex:i];
            }
            // Close all left tabs
            for (i = index - 1; i >=0; i--) {
                [_srTabView removeItemAtIndex:i];
            }
        }
    }
}

- (void)reloadTabAction:(id)sender
{
    // For NSMenu
    if ([sender respondsToSelector:@selector(representedObject)]) {
        // Get represented object
        id  representedObject;
        representedObject = [sender representedObject];
        if (!representedObject) {
            return;
        }
        
        // Get web view of selected tab
        int         selectedTabIndex;
        WebView*    webView;
        selectedTabIndex = [representedObject intValue];
        webView = (WebView*)[_srTabView viewAtIndex:selectedTabIndex];
        if (!webView) {
            return;
        }
        
        // Reload page
        [webView reload:self];
    }
}

- (void)reloadAllTabsAction:(id)sender
{
    // Reload all tabs
    int tabNumber;
    int i;
    tabNumber = [_srTabView numberOfItems];
    for (i = 0; i < tabNumber; i++) {
        // Get web view
        WebView*    webView;
        webView = (WebView*)[_srTabView viewAtIndex:i];
        if (!webView) {
            continue;
        }
        
        // Reload page
        [webView reload:self];
    }
}

- (void)bookmarkAllTabsAction:(id)sender
{
    // Set bookmark menu
    NSMenu* bookmarkMenu;
    bookmarkMenu = [[SRBookmarkStorage sharedInstance] bookmarkFolderMenu];
    [[self bookmarkAllPopup] removeAllItems];
    [[self bookmarkAllPopup] setMenu:bookmarkMenu];
    
    // Show add bookmark panel
    [NSApp beginSheet:[self bookmarkAllPanel] 
            modalForWindow:[self window] 
            modalDelegate:self 
            didEndSelector:@selector(_didEndBookmarkAllSheet:returnCode:contextInfo:) 
            contextInfo:NULL];
}

- (void)_didEndBookmarkAllSheet:(NSPanel*)sheet 
        returnCode:(int)returnCode 
        contextInfo:(void*)contextInfo
{
    // Close sheet
    [sheet orderOut:self];
    
    // Check return code
    if (returnCode != _OK_BUTTON_TAG) {
        return;
    }
    
    // Create bookmarks
    NSMutableArray* bookmarks;
    int             tabNum, i;
    bookmarks = [NSMutableArray array];
    tabNum = [_srTabView numberOfItems];
    for (i = 0; i < tabNum; i++) {
        // Get web view
        WebView*    webView;
        webView = (WebView*)[_srTabView viewAtIndex:i];
        if (!webView) {
            continue;
        }
        
        // Get title, URL string, and icon
        WebDataSource*  dataSource;
        NSString*       title;
        NSString*       URLString;
        dataSource = [[webView mainFrame] dataSource];
        if (!dataSource) {
            continue;
        }
        URLString = [[[dataSource request] URL] _web_userVisibleString];
        title = [dataSource pageTitle];
        if (!title) {
            title = URLString;
        }
        if (!title || !URLString) {
            continue;
        }
        
        // Create bookmark
        SRBookmark* bookmark;
        bookmark = [SRBookmark bookmarkWithTitle:title 
                URLString:URLString 
                originalBrowser:SRBrowserShiira];
        
        [bookmarks addObject:bookmark];
    }
    
    // Get parent
    SRBookmark* parent;
    parent = [[[self bookmarkAllPopup] selectedItem] representedObject];
    
    // Add bookmark
    if ([[SRBookmarkStorage sharedInstance] rootBookmark] != parent) {
        [parent addChildren:bookmarks];
    }
    else {
        NSArray*    children;
        int         i;
        children = [parent children];
        for (i = 0; i < [children count]; i++) {
            if (![[children objectAtIndex:i] isMutable]) {
                break;
            }
        }
        if (i < 0 || i > [children count]) {
            i = [children count];
        }
        
        [parent insertChildren:bookmarks atIndex:i];
    }
}

- (void)closeBookmarkAllSheetAction:(id)sender
{
    // Close sheet
    [NSApp endSheet:[self bookmarkAllPanel] returnCode:[sender tag]];
}

- (void)accordingTabToStringWidthAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // Get value
        BOOL    isAccordingTo;
        isAccordingTo = [defaults boolForKey:SRTabAccordingToStringWidth];
        
        // Set toggled value
        [defaults setBool:!isAccordingTo forKey:SRTabAccordingToStringWidth];
    }
}

- (void)openLocationAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get toolbar
    NSToolbar*  toolbar;
    toolbar = [[self window] toolbar];
    
    // When no URL combo box on toolbar
    if (![toolbar isVisible] || 
        ![self URLComboBox] || 
        [toolbar displayMode] == NSToolbarDisplayModeLabelOnly) 
    {
        if (![self URLComboBoxOnSheet]) {
            // Create URL combo box
            SRURLComboBox*  locationURLComboBox;
            locationURLComboBox = [[SRURLComboBox alloc] initWithFrame:NSMakeRect(0, 0, 0, 0)];
            [locationURLComboBox autorelease];
#if 1
            [[locationURLComboBox cell] setControlSize:NSSmallControlSize];
            [locationURLComboBox setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
#else
            if ([defaults boolForKey:SRToolbarUseSmallURLAndSearchField]) {
                [[locationURLComboBox cell] setControlSize:NSSmallControlSize];
                [locationURLComboBox setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
            }
            else {
                [[locationURLComboBox cell] setControlSize:NSRegularControlSize];
                [locationURLComboBox setFont:[NSFont systemFontOfSize:[NSFont systemFontSize]]];
            }
#endif
            [locationURLComboBox setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
            [[locationURLComboBox cell] setSendsActionOnEndEditing:NO];
            [locationURLComboBox setCompletes:YES];
            [locationURLComboBox setDelegate:self];
            [locationURLComboBox setTarget:self];
            [locationURLComboBox setAction:@selector(closeLocationPanelAction:)];
            [locationURLComboBox sizeToFit];
            
            NSRect  frame;
            frame = [[self locationURLView] frame];
            frame.origin = NSZeroPoint;
            [locationURLComboBox setFrame:frame];
            [locationURLComboBox setAutoresizingMask:NSViewWidthSizable | NSViewMinYMargin];
            
            [[self locationURLView] addSubview:locationURLComboBox];
        }
        
        // Show location panel
        [NSApp beginSheet:[self locationPanel] 
                modalForWindow:[self window] 
                modalDelegate:self 
                didEndSelector:@selector(_locationPanelDidEnd:returnCode:contextInfo:) 
                contextInfo:NULL];
        return;
    }
    
    // Make combo box first responder
    [[self window] makeFirstResponder:[self URLComboBox]];
}

- (void)closeLocationPanelAction:(id)sender
{
    // End sheet
    [NSApp endSheet:[self locationPanel] returnCode:[sender tag]];
}

- (void)_locationPanelDidEnd:(NSWindow*)sheet 
        returnCode:(int)returnCode 
        contextInfo:(void*)contextInfo
{
    // Close location panel
    [sheet orderOut:self];
    
    // For not go button
    if (returnCode != 0) {
        return;
    }
    
    // Treat as URL combo action
    [self URLComboBoxAction:[self URLComboBoxOnSheet]];
}

- (void)processComboBoxURLString:(NSString*)URLString
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Check argument
    if (!URLString || [URLString length] == 0 || [URLString isEqualToString:@"http://"]) {
        // Do not open page
        [[self window] makeFirstResponder:[self URLComboBox]];
        return;
    }
    
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Create request
    NSURLRequest*   request;
    request = [self _createRequestWithURLString:URLString];
    
    // Open request with action
    SROpenActionType    openAction;
    openAction = SROpenActionTypeFromModifierFlags([[NSApp currentEvent] modifierFlags]);
    [self openRequest:request frameName:nil groupName:nil withOpenAction:openAction];
}

- (void)URLComboBoxAction:(id)sender
{
    // Get string
    NSString*   string;
    string = [sender stringValue];
    
    // handle search keyword if one was entered
    NSDictionary* searchEngine = nil;
    NSRange whitespaceRange;
    whitespaceRange = [string rangeOfCharacterFromSet:[NSCharacterSet whitespaceCharacterSet]];
    
    if(whitespaceRange.length) {
        NSString* keyword;
        keyword = [string substringWithRange:NSMakeRange(0,whitespaceRange.location)];
        
        NSEnumerator* enumerator;
        enumerator = [[[SRSearchEnginesManager sharedInstance] searchEngines] objectEnumerator];
        
        NSDictionary* eachEngine;
        
        while((searchEngine == nil) && (eachEngine = [enumerator nextObject])) {
            if([[eachEngine objectForKey:@"keyword"] isEqualToString:keyword]) {
                searchEngine = eachEngine;
            }
        }
    }
    
    NSString*   URLString = nil;
    
    // Create URL string from search engine template
    if(searchEngine) {
        NSRange searchTermRange;
        searchTermRange = NSMakeRange(NSMaxRange(whitespaceRange),[string length] - NSMaxRange(whitespaceRange));
                                      
        NSString* searchTerm;
        searchTerm = [string substringWithRange:searchTermRange];
        
        URLString = SRCreateSearchURLStringFromSearchEngine(searchTerm, searchEngine);
    }
    else {
        // Get URL string from combo box
        URLString = [sender absoluteURLStringOfCompletedString:string];
    }
    
    // Open URL string
    [self processComboBoxURLString:URLString];
}

- (void)createNewBookmarkFolderAction:(id)sender
{
    // Pass to side bar controller
    [_sideBarController createNewBookmarkFolderAction:sender];
}

- (void)openBookmarkAction:(id)sender
{
    SRBookmark* bookmark = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            bookmark = representedObject;
        }
        // For SRBookmarkButton
        else if ([representedObject respondsToSelector:@selector(bookmark)]) {
            bookmark = [representedObject bookmark];
        }
    }
    // For SRBookmarkButton
    else if ([sender respondsToSelector:@selector(bookmark)]) {
        bookmark = [sender bookmark];
    }
    
    if (!bookmark) {
        return;
    }
    
    // Open bookmark
    if ([bookmark isFolderType]) {
        if ([bookmark isAutoTab]) {
            // Open in tabs
            [self openInTabsBookmark:bookmark];
        }
    }
    else {
        // Get open action
        SROpenActionType    openAction;
        openAction = SROpenActionTypeFromModifierFlags([[NSApp currentEvent] modifierFlags]);
        
        // Open bookmark wiht open action
        [self openBookmark:bookmark withOpenAction:openAction];
    }
}

- (void)openBookmarkInNewWindowAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmarkButton
        if ([representedObject respondsToSelector:@selector(bookmark)]) {
            // Open bookmark
            [self openInNewWindowBookmark:[representedObject bookmark]];
        }
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Open bookmark
            [self openInNewWindowBookmark:representedObject];
        }
    }
}

- (void)openBookmarkInNewBackgroundWindowAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmarkButton
        if ([representedObject respondsToSelector:@selector(bookmark)]) {
            // Open bookmark
            [self openInNewBackgroundWindowBookmark:[representedObject bookmark]];
        }
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Open bookmark
            [self openInNewBackgroundWindowBookmark:representedObject];
        }
    }
}

- (void)openBookmarkInNewTabAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmarkButton
        if ([representedObject respondsToSelector:@selector(bookmark)]) {
            // Open bookmark
            [self openInNewTabBookmark:[representedObject bookmark] select:YES];
        }
        
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Open bookmark
            [self openInNewTabBookmark:representedObject select:YES];
        }
    }
}

- (void)openBookmarkInNewBackgroundTabAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmarkButton
        if ([representedObject respondsToSelector:@selector(bookmark)]) {
            // Open bookmark
            [self openInNewTabBookmark:[representedObject bookmark] select:NO];
        }
        
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Open bookmark
            [self openInNewTabBookmark:representedObject select:NO];
        }
    }
}

- (void)openBookmarkInTabsAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmarkButton
        if ([representedObject respondsToSelector:@selector(bookmark)]) {
            // Open bookmark folder
            [self openInTabsBookmark:[representedObject bookmark]];
        }
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Open bookmark folder
            [self openInTabsBookmark:representedObject];
        }
        // For NSArray
        if ([representedObject isKindOfClass:[NSArray class]]) {
            // Open bookmarks
            [self openInTabsBookmarks:representedObject];
        }
    }
}

- (void)deleteBookmarkAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For SRBookmark
        if ([representedObject isKindOfClass:[SRBookmark class]]) {
            // Get parent
            SRBookmark* parent;
            parent = [[SRBookmarkStorage sharedInstance] parentOfBookmark:representedObject];
            
            if (parent) {
                // Remove bookmark
                [parent removeChild:representedObject];
            }
        }
        // For NSArray
        if ([representedObject isKindOfClass:[NSArray class]]) {
            [[[SRBookmarkStorage sharedInstance] rootBookmark] removeChildren:representedObject];
        }
    }
}

- (void)openBookmarkBarAtAction:(id)sender
{
    // For NSEvent
    if ([sender isKindOfClass:[NSEvent class]]) {
        // Get unicode character
        NSString*   characters;
        unichar     unicodeChar = 0;
        characters = [(NSEvent*)sender characters];
        if (characters && [characters length] > 0) {
            unicodeChar = [characters characterAtIndex:0];
        }
        
        // Get index for bookmark bar
        int index = -1;
        if (unicodeChar >= '1' && unicodeChar <= '9') {
            index = unicodeChar - '1';
        }
        else if (unicodeChar == '0') {
            index = 10;
        }
        if (index < 0) {
            return;
        }
        
        // Get bookmarks bar
        SRBookmark* bookmarksBar;
        bookmarksBar = [_bookmarksBarView bookmarksBar];
        if (!bookmarksBar) {
            return;
        }
        
        // Get bookmark
        NSArray*    children;
        SRBookmark* bookmark;
        children = [bookmarksBar children];
        if (!children || [children count] < index) {
            return;
        }
        bookmark = [children objectAtIndex:index];
        
        // Open bookmark
        if ([bookmark type] == SRBookmarkTypeHTML) {
            [self openBookmark:bookmark];
            return;
        }
        if ([bookmark isFolderType] && [bookmark isAutoTab]) {
            [self openInTabsBookmark:bookmark];
            return;
        }
    }
}

- (void)openLinkInNewTabAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            // Create request
            NSURLRequest*   request;
            request = [NSURLRequest requestWithURL:representedObject 
                    cachePolicy:NSURLRequestUseProtocolCachePolicy 
                    timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
            
            // Open request in new tab
            [self openInNewTabRequest:request frameName:nil groupName:nil select:YES];
        }
    }
}

- (void)openLinkInNewBackgroundTabAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            // Create request
            NSURLRequest*   request;
            request = [NSURLRequest requestWithURL:representedObject 
                    cachePolicy:NSURLRequestUseProtocolCachePolicy 
                    timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
            
            // Open request in new tab
            [self openInNewTabRequest:request frameName:nil groupName:nil select:NO];
        }
    }
}

- (void)openLinkInNewWindowAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            // Create request
            NSURLRequest*   request;
            request = [NSURLRequest requestWithURL:representedObject 
                    cachePolicy:NSURLRequestUseProtocolCachePolicy 
                    timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
            
            // Open request in new window
            [self openInNewWindowRequest:request frameName:nil groupName:nil];
        }
    }
}

- (void)openLinkInNewBackgroundWindowAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            // Create request
            NSURLRequest*   request;
            request = [NSURLRequest requestWithURL:representedObject 
                    cachePolicy:NSURLRequestUseProtocolCachePolicy 
                    timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
            
            // Open request in new backround window
            [self openInNewBackgroundWindowRequest:request frameName:nil groupName:nil];
        }
    }
}

- (void)openHistoryAction:(id)sender
{
    WebHistoryItem* historyItem = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            historyItem = representedObject;
        }
    }
    
    if (!historyItem) {
        return;
    }
    
    // Open hisory item
    [self openHistory:historyItem];
}

- (void)openHistoryInNewWindowAction:(id)sender
{
    WebHistoryItem* historyItem = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            historyItem = representedObject;
        }
    }
    
    if (!historyItem) {
        return;
    }
    
    // Open hisory item in new window
    [self openInNewWindowHistory:historyItem];
}

- (void)openHistoryInNewTabAction:(id)sender
{
    WebHistoryItem* historyItem = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            historyItem = representedObject;
        }
    }
    
    if (!historyItem) {
        return;
    }
    
    // Open hisory item in new tab
    [self openInNewTabHistory:historyItem];
}

- (void)openHistoryInTabsAction:(id)sender
{
    NSArray*    historyItems = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSArray
        if ([representedObject isKindOfClass:[NSArray class]]) {
            historyItems = representedObject;
        }
    }
    
    if (!historyItems) {
        return;
    }
    
    // Open hisory items in tabs
    [self openInTabsHistory:historyItems];
}

- (void)openHistoryInNewBackgroundWindowAction:(id)sender
{
    WebHistoryItem* historyItem = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            historyItem = representedObject;
        }
    }
    
    if (!historyItem) {
        return;
    }
    
    // Open hisory item in new background window
    [self openInNewBackgroundWindowHistory:historyItem];
}

- (void)openHistoryInNewBackgroundTabAction:(id)sender
{
    WebHistoryItem* historyItem = nil;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            historyItem = representedObject;
        }
    }
    
    if (!historyItem) {
        return;
    }
    
    // Open hisory item in new background tab
    [self openInNewTabHistory:historyItem select:NO];
}

- (void)deleteHistoryAction:(id)sender
{
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For WebHistoryItem
        if ([representedObject isKindOfClass:[WebHistoryItem class]]) {
            // Remove history item
            [[WebHistory optionalSharedHistory] removeItems:[NSArray arrayWithObject:representedObject]];
        }
        // For NSArray
        if ([representedObject isKindOfClass:[NSArray class]]) {
            // Remove history items
            [[WebHistory optionalSharedHistory] removeItems:representedObject];
        }
    }
}

- (void)addToBookmarkAction:(id)sender
{
    // Get current page URL, title, and icon
    NSURLRequest*       request;
    NSString*           URLString = nil;
    NSString*           title;
    
    // For NSMenuItem
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSDictionary
        if ([representedObject isKindOfClass:[NSDictionary class]]) {
            // Get URL and title
            NSURL*      linkURL;
            linkURL = [representedObject objectForKey:WebElementLinkURLKey];
            URLString = [linkURL _web_userVisibleString];
            if (!URLString) {
                return;
            }
            title = [representedObject objectForKey:WebElementLinkLabelKey];
            if (!title) {
                title = URLString;
            }
        }
    }
    // Get address from selected web view
    if (!URLString) {
        request = [[[[self selectedWebView] mainFrame] dataSource] request];
        if (!request) {
            return;
        }
        URLString = [[request URL] _web_userVisibleString];
        title = [[[[self selectedWebView] mainFrame] dataSource] pageTitle];
        if (!title) {
            title = URLString;
        }
    }
    
    // Set title to add bookmark text field
    NSText* text;
    text = [[self addBookmarkTextField] currentEditor];
    [[self addBookmarkTextField] setStringValue:title];
    [text moveToBeginningOfDocument:self];
    [text selectAll:self];
    
    // Create bookmark for adding
    // Note that this bookmark is retained
    // It will be released in _didEndAddBookmarkSheet:returnCode:contextInfo
    SRBookmark* bookmark;
    bookmark = [SRBookmark bookmarkWithTitle:title 
            URLString:URLString 
            originalBrowser:SRBrowserShiira];
    [bookmark retain];
    
    // Set bookmark menu
    NSMenu* bookmarkMenu;
    bookmarkMenu = [[SRBookmarkStorage sharedInstance] bookmarkFolderMenu];
    [[self addBookmarkPopup] removeAllItems];
    [[self addBookmarkPopup] setMenu:bookmarkMenu];
    
    // Show add bookmark panel
    [NSApp beginSheet:[self addBookmarkPanel] 
            modalForWindow:[self window] 
            modalDelegate:self 
            didEndSelector:@selector(_didEndAddBookmarkSheet:returnCode:contextInfo:) 
            contextInfo:(void*)bookmark];
}

- (void)_didEndAddBookmarkSheet:(NSPanel*)sheet 
        returnCode:(int)returnCode 
        contextInfo:(void*)contextInfo
{
    // Close sheet
    [sheet orderOut:self];
    
    if (returnCode == _OK_BUTTON_TAG) {
        // Get bookmark and parent
        SRBookmark* bookmark;
        SRBookmark* parent;
        bookmark = (SRBookmark*)contextInfo;
        parent = [[[self addBookmarkPopup] selectedItem] representedObject];
        
        // Get bookmark title
        NSString*   title;
        title = [[self addBookmarkTextField] stringValue];
        if (title && [title length] > 0) {
            [bookmark setTitle:title];
        }
        
        // Add bookmark
        SRBookmark* rootBookmark;
        rootBookmark = [[SRBookmarkStorage sharedInstance] rootBookmark];
        if (rootBookmark != parent) {
            [parent addChild:bookmark];
        }
        else {
            NSArray*    children;
            int         i;
            children = [parent children];
            for (i = 0; i < [children count]; i++) {
                if (![[children objectAtIndex:i] isMutable]) {
                    break;
                }
            }
            if (i < 0 || i > [children count]) {
                i = [children count];
            }
            
            [parent insertChild:bookmark atIndex:i];
        }
    }
    
    // Release bookmark
    [(id)contextInfo release];
}

- (void)closeAddBookmarkSheetAction:(id)sender
{
    // Close sheet
    [NSApp endSheet:[self addBookmarkPanel] returnCode:[sender tag]];
}

- (void)addToBookmarksBarAction:(id)sender
{
}

- (void)closeAddToBarSheetAction:(id)sender
{
}

- (void)showNewFolderAction:(id)sender
{
    // Show new folder dialog as modal
    int result;
    result = [NSApp runModalForWindow:[self newFolderPanel]];
    [[self newFolderPanel] orderOut:self];
    if (result != _OK_BUTTON_TAG) {
        return;
    }
    
    // Get folder name
    NSString*   folderName;
    folderName = [[self newFolderTextField] stringValue];
    if (!folderName || [folderName length] == 0) {
        return;
    }
    
    //
    // Create new bookmark folder
    //
    
    // Get bookmark popup
    NSPopUpButton*  popup = nil;
    if ([[self addBookmarkPanel] isVisible]) {
        popup = [self addBookmarkPopup];
    }
    if ([[self bookmarkAllPanel] isVisible]) {
        popup = [self bookmarkAllPopup];
    }
    if (!popup) {
        return;
    }
    
    // Get parent bookmark folder
    SRBookmark* parent = nil;
    parent = [[popup selectedItem] representedObject];
    if (!parent) {
        return;
    }
    
    // For immutable bookmark folder
    if (![parent isMutable]) {
        // Get root folder
        parent = [[SRBookmarkStorage sharedInstance] rootBookmark];
    }
    
    // Add bookmark folder
    SRBookmark* folder;
    folder = [SRBookmark folderWithTitle:folderName 
            originalBrowser:SRBrowserShiira];
    if (parent == [[SRBookmarkStorage sharedInstance] rootBookmark]) {
        // Insert it before immutable folders
        NSArray*    children;
        int         index;
        children = [parent children];
        for (index = 0; index < [children count]; index++) {
            SRBookmark* child;
            child = [children objectAtIndex:index];
            if (![child isMutable]) {
                break;
            }
        }
        
        [parent insertChild:folder atIndex:index];
    }
    else {
        [parent addChild:folder];
    }
    
    // Update bookmark popup
    NSMenu* bookmarkMenu;
    bookmarkMenu = [[SRBookmarkStorage sharedInstance] bookmarkFolderMenu];
    [popup removeAllItems];
    [popup setMenu:bookmarkMenu];
    
    // Select new one
    int index;
    index = [popup indexOfItemWithRepresentedObject:folder];
    if (index != -1) {
        [popup selectItemAtIndex:index];
    }
}

- (void)closeNewFolderPanelAction:(id)sender
{
    // Cloase new folder panel
    [NSApp stopModalWithCode:[sender tag]];
}

- (void)findBySearchEngineAction:(id)sender
{
    if (![self searchField]) {
        return;
    }
    
    // Get web view
    WebView*    webView;
    webView = [self selectedWebView];
    if (!webView) {
        return;
    }
    
    // Get information for search
    NSDictionary*   searchItem;
    NSString*       searchString;
    NSString*       engineName;
    NSString*       URLString;
    searchItem = [sender representedObject];
    searchString = [searchItem objectForKey:@"string"];
    engineName = [searchItem objectForKey:@"title"];
    URLString = [searchItem objectForKey:@"URLString"];
    
    // Add to recent item
    [[SRSearchEnginesManager sharedInstance] addRecentWord:searchString 
            engineName:engineName 
            URLString:URLString];
    
    // Search string with current modifierFlags
    SROpenActionType    openAction;
    openAction = SROpenActionTypeFromModifierFlags([[NSApp currentEvent] modifierFlags]);
    if(openAction == SROpenOptionAction){
        openAction = SROpenAction;
    }
    [self openURLString:URLString withOpenAction:openAction];
}

- (IBAction)showDownloadsAction:(id)sender
{
    // Open sidebar
    if([[_sideBarController drawer] state] != NSDrawerOpenState) {
        [[_sideBarController drawer] open];
    }
    
    // Show download tab
    if ([_sideBarController selectedTab] != SRSidebarDownloadTab) {
        [_sideBarController showTab:SRSidebarDownloadTab];
    }
}

- (IBAction)toggleDownloadsAction:(id)sender
{
    if(([[_sideBarController drawer] state] == NSDrawerOpenState) && 
       ([_sideBarController selectedTab] == SRSidebarDownloadTab))
    {
        [[_sideBarController drawer] close];
    }
    else {
        // Open sidebar
        [[_sideBarController drawer] open];
        
        // Show download tab
        [_sideBarController showTab:SRSidebarDownloadTab];
    }
}

- (void)copyImageURLToClipboardAction:(id)sender
{
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            NSPasteboard*   pb;
            pb = [NSPasteboard generalPasteboard];
            
            [pb declareTypes:[NSArray arrayWithObjects:NSStringPboardType,NSURLPboardType,nil] owner:nil];
            [representedObject writeToPasteboard:pb];
            [pb setString:[representedObject _web_userVisibleString] forType:NSStringPboardType];
        }
    }
}

- (void)openURLWithAction:(id)sender
{
    if ([sender respondsToSelector:@selector(representedObject)]) {
        id  representedObject;
        representedObject = [sender representedObject];
        
        // For NSURL
        if ([representedObject isKindOfClass:[NSURL class]]) {
            // Get bundle identifier
            NSString*   bundleId;
            bundleId = [[NSBundle bundleWithPath:[representedObject path]] bundleIdentifier];
            if (!bundleId) {
                return;
            }
            
            // Get current URL
            NSURL*  url;
            url = [[[[[self selectedWebView] mainFrame] dataSource] request] URL];
            if (!url) {
                return;
            }
            
            // Open URL
            [[NSWorkspace sharedWorkspace] 
                    openURLs:[NSArray arrayWithObject:url] 
                    withAppBundleIdentifier:bundleId 
                    options:0 
                    additionalEventParamDescriptor:nil 
                    launchIdentifiers:nil];
        }
    }
}

- (void)printPageAsPDFAction:(id)sender
{
    // Get selected web view and document view
    WebView*    webView;
    NSView*     documentView;
    webView = [self selectedWebView];
    documentView = [[[webView mainFrame] frameView] documentView];
    if (!documentView) {
        return;
    }
    
    // Get title
    NSString*   title;
    title = [[[webView mainFrame] dataSource] pageTitle];
    if (!title || [title length] == 0) {
        title = NSLocalizedString(@"Untitled", @"Untitled");
    }
    
    // Get accessory view
    SRMainDocument* document;
    NSView*         accessoryView;
    document = [self document];
    accessoryView = [document printAccessoryView];
    
    // Set properties of print panel
    WebPreferences* preferences;
    preferences = [SRPreferencesController defaultWebPreferences];
    [document setValue:[NSNumber numberWithBool:[preferences shouldPrintBackgrounds]] 
            forKey:@"printBackground"];
    
    // Show save dialog
    NSSavePanel*    savePanel;
    savePanel = [NSSavePanel savePanel];
    [savePanel setAccessoryView:accessoryView];
    [savePanel beginSheetForDirectory:nil 
            file:[title stringByAppendingPathExtension:@"pdf"] 
            modalForWindow:[self window] 
            modalDelegate:self 
            didEndSelector:@selector(didEndPrintPageAsPDFPanel:returnCode:contextInfo:) 
            contextInfo:NULL];
}

- (void)didEndPrintPageAsPDFPanel:(NSSavePanel*)sheet 
        returnCode:(int)returnCode 
        contextInfo:(void*)contextInfo
{
    if (returnCode != NSOKButton) {
        return;
    }
    
    // Get selected web view and document view
    WebView*    webView;
    NSView*     documentView;
    webView = [self selectedWebView];
    documentView = [[[webView mainFrame] frameView] documentView];
    if (!documentView) {
        return;
    }
    
    // Get file name
    NSString*   fileName;
    fileName = [sheet filename];
    if (!fileName) {
        return;
    }
    
    // Print page as a PDF
    [[documentView dataWithPDFInsideRect:[documentView frame]] 
            writeToFile:fileName atomically:YES];
}

- (void)showAllTabsAction:(id)sender
{
    // Deexpose
    if ([SRTabExposeController isExposing]) {
        [_tabExposeController deexpose];
        return;
    }
    
    // Check number of tab
    if ([_srTabView numberOfItems] < 2) {
        return;
    }
    
    // Expose
    [_tabExposeController exposeWithTabView:_srTabView];
}

- (void)toggleURLIDNAction:(id)sender
{
    // Get URL combo box
    SRURLComboBox*  comboBox;
    comboBox = [self URLComboBox];
    if (!comboBox) {
        return;
    }
    
    // Check URL string
    if ([_webViewURLString length] > 0) {
        // Create NSURL
        NSURL*  url;
        url = [NSURL _web_URLWithUserTypedString:_webViewURLString];
        
        // Use Punycode
        NSString*   URLString;
        URLString = [url _web_userVisibleString];
        if (![_webViewURLString isEqualToString:URLString]) {
            [self setValue:URLString forKey:SRMainWebViewURLString];
            return;
        }
        
        // Use ASCII
        URLString = [url absoluteString];
        if (![_webViewURLString isEqualToString:URLString]) {
            [self setValue:URLString forKey:SRMainWebViewURLString];
            return;
        }
    }
}

- (void)toggleSmallURLAndSearchFieldAction:(id)sender
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Check default
    BOOL    useSmall;
    useSmall = [defaults boolForKey:SRToolbarUseSmallURLAndSearchField];
    
    // Toggle it
    useSmall = !useSmall;
    
    NSControlSize   controlSize;
    NSFont*         font;
    if (useSmall) {
        controlSize = NSSmallControlSize;
        font = [NSFont systemFontOfSize:[NSFont smallSystemFontSize]];
    }
    else {
        controlSize = NSRegularControlSize;
        font = [NSFont systemFontOfSize:[NSFont systemFontSize]];
    }
    
    // Toggle URL combo box size
    SRURLComboBox*  URLComboBox;
    URLComboBox = [self URLComboBox];
    if (URLComboBox) {
        NSRect  frame;
        frame = [URLComboBox frame];
        
        [[URLComboBox cell] setControlSize:controlSize];
        [URLComboBox setFont:font];
        [URLComboBox sizeToFit];
        frame.size.height = [URLComboBox frame].size.height;
        [URLComboBox setFrame:frame];
        
        NSToolbarItem*  item;
        item = [[[self window] toolbar] toolbarItemWithIdentifier:SRURLIdentifier];
        [item setMinSize:NSMakeSize(SRURLMinWidth, [URLComboBox frame].size.height)];
        [item setMaxSize:NSMakeSize(SRURLMaxWidth, [URLComboBox frame].size.height)];
    }
    
    // Toggle search field size
    SRSearchField*  searchField;
    searchField = [self searchField];
    if (searchField) {
        NSRect  frame;
        frame = [searchField frame];
        
        [[searchField cell] setControlSize:controlSize];
        [searchField setFont:font];
        [searchField sizeToFit];
        frame.size.height = [searchField frame].size.height;
        [searchField setFrame:frame];
        
        NSToolbarItem*  item;
        item = [[[self window] toolbar] toolbarItemWithIdentifier:SRSearchIdentifier];
        [item setMinSize:NSMakeSize(SRSearchMinWidth, [searchField frame].size.height)];
        [item setMaxSize:NSMakeSize(SRSearchMaxWidth, [searchField frame].size.height)];
    }
    
    // Set it default
    [defaults setBool:useSmall forKey:SRToolbarUseSmallURLAndSearchField];
}

#pragma mark -
//--------------------------------------------------------------//
// Menu item validation
//--------------------------------------------------------------//

- (BOOL)validateMenuItem:(id<NSMenuItem>)menuItem
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get tag
    int tag;
    tag = [menuItem tag];
    
    // Get web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    switch (tag) {
    // File menu
    case SRNewTabTag: {
        // Check tab browsing
        return [defaults boolForKey:SRTabEnableTabbedBrowsing];
    }
    case SRCloseWindowTag: {
        // Check number of tab
        if (![defaults boolForKey:SRTabEnableTabbedBrowsing] || 
            [_srTabView numberOfItems] < 2)
        {
            [menuItem setKeyEquivalent:@"w"];
            [menuItem setKeyEquivalentModifierMask:NSCommandKeyMask];
        }
        else {
            [menuItem setKeyEquivalent:@"W"];
            [menuItem setKeyEquivalentModifierMask:NSCommandKeyMask];
        }
        
        // Always YES
        return YES;
    }
    case SRCloseTabTag: {
        // Check number of tab
        if (![defaults boolForKey:SRTabEnableTabbedBrowsing] || 
            [_srTabView numberOfItems] < 2)
        {
            [menuItem setKeyEquivalent:@""];
            return NO;
        }
        else {
            [menuItem setKeyEquivalent:@"w"];
            [menuItem setKeyEquivalentModifierMask:NSCommandKeyMask];
            return YES;
        }
    }
    
    // View menu
    case SRToolbarTag: {
        // Set menu status
        BOOL    isToolbarVisible;
        isToolbarVisible = [[[self window] toolbar] isVisible];
        [menuItem setState:isToolbarVisible ? NSOnState : NSOffState];
        
        // Always YES
        return YES;
    }
    case SRBookmarksBarTag: {
        // Set menu status
        BOOL    isBookmarksBarVisible;
        isBookmarksBarVisible = [self isBookmarksBarVisible];
        [menuItem setState:isBookmarksBarVisible ? NSOnState : NSOffState];
        
        // Always YES
        return YES;
    }
    case SRStatusBarTag: {
        // Set menu status
        BOOL    isStatusBarVisible;
        isStatusBarVisible = [self isStatusBarVisible];
        [menuItem setState:isStatusBarVisible ? NSOnState : NSOffState];
        
        // Always YES
        return YES;
    }
    case SRSidebarTag: {
        // Set menu status
        BOOL    isSidebarVisible;
        isSidebarVisible = [[_sideBarController drawer] state] == NSDrawerOpenState;
        [menuItem setState:isSidebarVisible ? NSOnState : NSOffState];
        
        // Always YES
        return YES;
    }
    case SRCustomizeToolbarTag: {
        // Always YES
        return YES;
    }
    case SRStopLoadingTag: {
        if (!webView) {
            return NO;
        }
        
        // Get data source
        WebDataSource*  provisionalDataSource;
        WebDataSource*  dataSource;
        provisionalDataSource = [[webView mainFrame] provisionalDataSource];
        dataSource = [[webView mainFrame] dataSource];
        
        if (provisionalDataSource) {
            return [provisionalDataSource isLoading];
        }
        if (dataSource) {
            if ([dataSource request]) {
                return NO;
            }
            return [dataSource isLoading];
        }
        return NO;
    }
    case SRReloadPageTag: {
        if (!webView) {
            return NO;
        }
        
        // Get data source
        WebDataSource*  provisionalDataSource;
        WebDataSource*  dataSource;
        provisionalDataSource = [[webView mainFrame] provisionalDataSource];
        dataSource = [[webView mainFrame] dataSource];
        
        if (provisionalDataSource) {
            return ![provisionalDataSource isLoading];
        }
        if (dataSource) {
            if (![dataSource request]) {
                return NO;
            }
            return ![dataSource isLoading];
        }
        return NO;
    }
    case SRBiggerTextTag: {
        if (!webView) {
            return NO;
        }
        
        return [webView canMakeTextLarger];
    }
    case SRSmallerTextTag: {
        if (!webView) {
            return NO;
        }
        
        return [webView canMakeTextSmaller];
    }
    case SRViewPageSourceTag: {
        // Get data source
        WebDataSource*  dataSource;
        dataSource = [[webView mainFrame] dataSource];
        if (!dataSource) {
            return NO;
        }
        
        return [[dataSource representation] canProvideDocumentSource];
    }
    case SROpenURLWithBrowserItemTag: {
        NSURL*  url;
        url = [[[[webView mainFrame] dataSource] request] URL];
        return url != nil;
    }
    case SRTextEncodingDefaultTag: {
        // Update submenu
        NSMenu*             textEncodingMenu;
        NSString*           encodingName;
        NSStringEncoding    encoding;
        textEncodingMenu = [menuItem menu];
        encodingName = [webView customTextEncodingName];
        if (!encodingName) {
            encoding = SRTextEncodingDefaultTag;
        }
        else {
            // Get text encoding
            encoding = [[[NSValueTransformer valueTransformerForName:SREncodingToIANATransformerName] 
                    reverseTransformedValue:encodingName] intValue];
        }
        
        // Get correspond menu item
        id<NSMenuItem>  encodingMenuItem;
        encodingMenuItem = [textEncodingMenu itemWithTag:encoding];
        
        // Set state
        NSEnumerator*   enumerator;
        id<NSMenuItem>  item;
        enumerator = [[textEncodingMenu itemArray] objectEnumerator];
        while (item = [enumerator nextObject]) {
            [item setState:item == encodingMenuItem ? NSOnState : NSOffState];
        }
        
        return YES;
    }
    
    // History menu
    case SRBackTag: {
        if (!webView) {
            return NO;
        }
        
        return [webView canGoBack];
    }
    case SRForwardTag: {
        if (!webView) {
            return NO;
        }
        
        return [webView canGoForward];
    }
    case SRHomeTag: {
        // Get home page URL
        NSUserDefaults* defaults;
        NSString*       URLString;
        defaults = [NSUserDefaults standardUserDefaults];
        URLString = [defaults stringForKey:SRGeneralHomePgae];
        return URLString && [URLString length] > 0;
    }
    case SRClearHistoryTag: {
        return YES;
    }
    
    // Bookmark menu
    case SRAddToBookmarkTag: {
        // Get selected web view
        WebView*    webView;
        webView = [self selectedWebView];
        if (![[[webView mainFrame] dataSource] request]) {
            return NO;
        }
        return YES;
    }
    case SRAddBookmarkFolderTag: {
        // Always YES
        return YES;
    }
    case SRShowBookmarkInSidebarTag: {
        // Always YES
        return YES;
    }
    
    // Window menu
    case SRPreviousTabTag: {
        // Get selected index
        int numberOfItems;
        int selectedIndex;
        numberOfItems = [_srTabView numberOfItems];
        selectedIndex = [_srTabView selectedIndex];
        if (selectedIndex == -1) {
            return NO;
        }
        
        return numberOfItems > 1;
    }
    case SRNextTabTag: {
        // Get item number and selected index
        int numberOfItems;
        int selectedIndex;
        numberOfItems = [_srTabView numberOfItems];
        selectedIndex = [_srTabView selectedIndex];
        if (selectedIndex == -1) {
            return NO;
        }
        
        return numberOfItems > 1;
    }
    case SRShowAllTabsTag: {
        return [_srTabView numberOfItems] > 1;
    }
    }
    
    return YES;
}

#pragma mark -
//--------------------------------------------------------------//
// NSMenu delegate
//--------------------------------------------------------------//

- (BOOL)menuHasKeyEquivalent:(NSMenu*)menu 
        forEvent:(NSEvent*)event 
        target:(id*)target 
        action:(SEL*)action
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get attributes
    unsigned int    modifierFlags;
    BOOL            optFlag, shiftFlag;
    NSString*       characters;
    modifierFlags = [event modifierFlags];
    optFlag = modifierFlags & NSAlternateKeyMask;
    shiftFlag = modifierFlags & NSShiftKeyMask;
    characters = [event charactersIgnoringModifiers];
    
    // For file menu
    if (menu == [SRAppDelegate fileMenu]) {
        // For cmd + 'w'
        if (!optFlag && !shiftFlag && [characters isEqualToString:@"w"]) {
            // Check number of tab
            if (![defaults boolForKey:SRTabEnableTabbedBrowsing] || 
                [_srTabView numberOfItems] < 2)
            {
                // Use close window action
                *target = nil;
                *action = @selector(closeWindowAction:);
            }
            else {
                // Use close tab action
                *target = nil;
                *action = @selector(closeTabAction:);
            }
            
            return YES;
        }
    }
    
    return NO;
}

#pragma mark -
//--------------------------------------------------------------//
// NSToolbar delegate
//--------------------------------------------------------------//

- (NSArray*)toolbarAllowedItemIdentifiers:(NSToolbar*)toolbar
{
    return _toolbarItemIdentifiers;
}

- (NSArray*)toolbarDefaultItemIdentifiers:(NSToolbar*)toolbar
{
    return _toolbarDefaultItemIdentifiers;
}

- (NSToolbarItem*)toolbar:(NSToolbar*)toolbar 
        itemForItemIdentifier:(NSString*)itemIdentifier 
        willBeInsertedIntoToolbar:(BOOL)flag
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Create toolbar item
    NSToolbarItem*  item;
    item = [[NSToolbarItem alloc] 
            initWithItemIdentifier:itemIdentifier];
    [item autorelease];
    [item setTarget:self];
    
    // Get label, image, and action
    NSString*   label = nil;
    NSImage*    image = nil;
    SEL         action = NULL;
    NSString*   paletteLabel = nil;
    
    // Go back
    if ([itemIdentifier isEqualToString:SRGoBackIdentifier]) {
        static NSString*    goBackLabel = nil;
        if (!goBackLabel) {
            goBackLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = goBackLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(goBackAction:);
    }
    // Go forward
    else if ([itemIdentifier isEqualToString:SRGoForwardIdentifier]) {
        static NSString*    goForwardLabel = nil;
        if (!goForwardLabel) {
            goForwardLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = goForwardLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(goForwardAction:);
    }
    // Reload page
    else if ([itemIdentifier isEqualToString:SRReloadPageIdentifier]) {
        static NSString*    reloadPageLabel = nil;
        if (!reloadPageLabel) {
            reloadPageLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = reloadPageLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(reloadPageAction:);
    }
    // Stop loading
    else if ([itemIdentifier isEqualToString:SRStopLoadingIdentifier]) {
        static NSString*    stopLoadingLabel = nil;
        if (!stopLoadingLabel) {
            stopLoadingLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = stopLoadingLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(stopLoadingAction:);
    }
    // Go home
    else if ([itemIdentifier isEqualToString:SRGoHomeIdentifier]) {
        static NSString*    goHomeLabel = nil;
        if (!goHomeLabel) {
            goHomeLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = goHomeLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(goHomeAction:);
    }
    // URL
    else if ([itemIdentifier isEqualToString:SRURLIdentifier]) {
        static NSString*    urlLabel = nil;
        if (!urlLabel) {
            urlLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        
        // Create URL combo box
        SRURLComboBox*  URLComboBox;
        URLComboBox = [[SRURLComboBox alloc] initWithFrame:NSMakeRect(0, 0, 0, 0)];
        [URLComboBox autorelease];
#if 1
        [[URLComboBox cell] setControlSize:NSSmallControlSize];
        [URLComboBox setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
#else
        if ([defaults boolForKey:SRToolbarUseSmallURLAndSearchField]) {
            [[URLComboBox cell] setControlSize:NSSmallControlSize];
            [URLComboBox setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
        }
        else {
            [[URLComboBox cell] setControlSize:NSRegularControlSize];
            [URLComboBox setFont:[NSFont systemFontOfSize:[NSFont systemFontSize]]];
        }
#endif
        [URLComboBox setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
        [URLComboBox sizeToFit];
        
        [item setView:URLComboBox];
        [item setMinSize:NSMakeSize(SRURLMinWidth, [URLComboBox frame].size.height)];
        [item setMaxSize:NSMakeSize(SRURLMaxWidth, [URLComboBox frame].size.height)];
        
        label = urlLabel;
    }
    // Show sidebar
    else if ([itemIdentifier isEqualToString:SRSidebarIdentifier]) {
        static NSString*    showSidebarLabel = nil;
        if (!showSidebarLabel) {
            showSidebarLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = showSidebarLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(toggleSidebarAction:);
    }
    // Bookmark page
    else if ([itemIdentifier isEqualToString:SRBookmarkPageIdentifier]) {
        static NSString*    bookmarkPageLabel = nil;
        if (!bookmarkPageLabel) {
            bookmarkPageLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = bookmarkPageLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(addToBookmarkAction:);
    }
    // New tab
    else if ([itemIdentifier isEqualToString:SRNewTabIdentifier]) {
        static NSString*    newTabLabel = nil;
        if (!newTabLabel) {
            newTabLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = newTabLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(newTabAction:);
    }
    // Close tab
    else if ([itemIdentifier isEqualToString:SRCloseTabIdentifier]) {
        static NSString*    closeTabLabel = nil;
        if (!closeTabLabel) {
            closeTabLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = closeTabLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(closeTabAction:);
    }
    // Bigger Text
    else if ([itemIdentifier isEqualToString:SRBiggerTextIdentifier]) {
        static NSString*    biggerTextLabel = nil;
        if (!biggerTextLabel) {
            biggerTextLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = biggerTextLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(biggerTextAction:);
    }
    // Smaller Text
    else if ([itemIdentifier isEqualToString:SRSmallerTextIdentifier]) {
        static NSString*    smallerTextLabel = nil;
        if (!smallerTextLabel) {
            smallerTextLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = smallerTextLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(smallerTextAction:);
    }
    // Customize
    else if ([itemIdentifier isEqualToString:SRCustomizeIdentifier]) {
        static NSString*    customizeLabel = nil;
        if (!customizeLabel) {
            customizeLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        label = customizeLabel;
        image = [_iconImageDict objectForKey:itemIdentifier];
        action = @selector(customizeToolbarAction:);
    }
    // Search
    else if ([itemIdentifier isEqualToString:SRSearchIdentifier]) {
        static NSString*    searchLabel = nil;
        if (!searchLabel) {
            searchLabel = [NSLocalizedStringFromTable(itemIdentifier, SRMainToolbarLabelTable, nil) retain];
        }
        
        SRSearchField*  searchField;
        searchField = [[SRSearchField alloc] initWithFrame:NSMakeRect(0, 0, 0, 0)];
        [searchField autorelease];
        [searchField setAction:@selector(searchAction:)];
#if 1
        [[searchField cell] setControlSize:NSSmallControlSize];
        [searchField setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
#else
        if ([defaults boolForKey:SRToolbarUseSmallURLAndSearchField]) {
            [[searchField cell] setControlSize:NSSmallControlSize];
            [searchField setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
        }
        else {
            [[searchField cell] setControlSize:NSRegularControlSize];
            [searchField setFont:[NSFont systemFontOfSize:[NSFont systemFontSize]]];
        }
#endif
        [searchField sizeToFit];
        [[searchField cell] setWraps:NO];
        [[searchField cell] setScrollable:YES];
        [[searchField cell] setSendsWholeSearchString:YES];
        [searchField setFlexible:YES];
        
        [item setView:searchField];
        int width;
        width = [[NSUserDefaults standardUserDefaults] integerForKey:SRSearchFieldWidth];
        if (width < SRSearchFieldMinWidth) {
            width = SRSearchFieldMinWidth;
        }
        if (width > SRSearchFieldMaxWidth) {
            width = SRSearchFieldMaxWidth;
        }
        [item setMinSize:NSMakeSize(width, [searchField frame].size.height)];
        [item setMaxSize:NSMakeSize(width, [searchField frame].size.height)];
        
        label = searchLabel;
    }
    
    // Set item attributes
    if (label) {
        [item setLabel:label];
    }
    if (image) {
        [item setImage:image];
    }
    if (action) {
        [item setAction:action];
    }
    if (paletteLabel) {
        [item setPaletteLabel:paletteLabel];
    }
    else {
        if (label) {
            [item setPaletteLabel:label];
        }
    }
    return item;
}

- (void)toolbarWillAddItem:(NSNotification*)notification
{
    // Get item
    NSToolbarItem*  item;
    item = [[notification userInfo] objectForKey:@"item"];
    if (!item) {
        return;
    }
    
    // URL combo box
    if ([[item itemIdentifier] isEqualToString:SRURLIdentifier]) {
        SRURLComboBox*  URLComboBox;
        URLComboBox = (SRURLComboBox*)[item view];
        
        [URLComboBox setTarget:self];
        [URLComboBox setAction:@selector(URLComboBoxAction:)];
        [[URLComboBox cell] setSendsActionOnEndEditing:NO];
        [URLComboBox setCompletes:YES];
        [URLComboBox setDelegate:self];
        [URLComboBox bind:@"value" 
                toObject:_webViewController 
                withKeyPath:@"selection.webViewURLString" 
                options:nil];
        
        return;
    }
}

//--------------------------------------------------------------//
// NSToolbarItemValidation informal protocol
//--------------------------------------------------------------//

- (BOOL)validateToolbarItem:(NSToolbarItem*)item
{
    // Get item identifier
    id  identifier;
    identifier = [item itemIdentifier];
    
    // User defaults
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get selected web view
    WebView*    webView;
    webView = [self selectedWebView];
    
    // Go Back
    if ([identifier isEqualToString:SRGoBackIdentifier]) {
        if (!webView) {
            return NO;
        }
        
        return [webView canGoBack];
    }
    // Go Forward
    if ([identifier isEqualToString:SRGoForwardIdentifier]) {
        if (!webView) {
            return NO;
        }
        return [webView canGoForward];
    }
    // Reload Page
    if ([identifier isEqualToString:SRReloadPageIdentifier]) {
        if (!webView) {
            return NO;
        }
        
        // Get data source
        WebDataSource*  provisionalDataSource;
        WebDataSource*  dataSource;
        provisionalDataSource = [[webView mainFrame] provisionalDataSource];
        dataSource = [[webView mainFrame] dataSource];
        
        if (provisionalDataSource) {
            return ![provisionalDataSource isLoading];
        }
        if (dataSource) {
            if (![dataSource request]) {
                return NO;
            }
            return ![dataSource isLoading];
        }
        return NO;
    }
    // Stop Loading
    if ([identifier isEqualToString:SRStopLoadingIdentifier]) {
        if (!webView) {
            return NO;
        }
        
        // Get data source
        WebDataSource*  provisionalDataSource;
        WebDataSource*  dataSource;
        provisionalDataSource = [[webView mainFrame] provisionalDataSource];
        dataSource = [[webView mainFrame] dataSource];
        
        if (provisionalDataSource) {
            return [provisionalDataSource isLoading];
        }
        if (dataSource) {
            if (![dataSource request]) {
                return NO;
            }
            return [dataSource isLoading];
        }
        return NO;
    }
    // Go Home
    if ([identifier isEqualToString:SRGoHomeIdentifier]) {
        // Get home page URL
        NSUserDefaults* defaults;
        NSString*       URLString;
        defaults = [NSUserDefaults standardUserDefaults];
        URLString = [defaults stringForKey:SRGeneralHomePgae];
        return URLString && [URLString length] > 0;
    }
    // URL
    if ([identifier isEqualToString:SRURLIdentifier]) {
        return YES;
    }
    // Show Sidebar
    if ([identifier isEqualToString:SRSidebarIdentifier]) {
        // Check draewr state
        int state;
        state = [[_sideBarController drawer] state];
        if (state == NSDrawerClosedState || 
            state == NSDrawerClosingState)
        {
            //[item setImage:];
        }
        else {
            //[item setImage:];
        }
        
        // Always YES
        return YES;
    }
    // Bookmark Page
    if ([identifier isEqualToString:SRBookmarkPageIdentifier]) {
        if (!webView) {
            return NO;
        }
        return [[[webView mainFrame] dataSource] request] != nil;
    }
    // New Tab
    if ([identifier isEqualToString:SRNewTabIdentifier]) {
        // Check tab browsing
        return [defaults boolForKey:SRTabEnableTabbedBrowsing];
    }
    // Close Tab
    if ([identifier isEqualToString:SRCloseTabIdentifier]) {
        // Check tab browsing
        if (![defaults boolForKey:SRTabEnableTabbedBrowsing]) {
            return NO;
        }
        
        // Check number of tab
        if ([_srTabView numberOfItems] < 2) {
            return NO;
        }
        
        return YES;
    }
    // Bigger Text
    if ([identifier isEqualToString:SRBiggerTextIdentifier]) {
        if (!webView) {
            return NO;
        }
        return [webView canMakeTextLarger];
    }
    // Smaller Text
    if ([identifier isEqualToString:SRSmallerTextIdentifier]) {
        if (!webView) {
            return NO;
        }
        return [webView canMakeTextSmaller];
    }
    // Customize
    if ([identifier isEqualToString:SRCustomizeIdentifier]) {
        // Always YES
        return YES;
    }
    // Search
    if ([identifier isEqualToString:SRSearchIdentifier]) {
        // Always YES
        return YES;
    }
    
    return NO;
}

#pragma mark -
//--------------------------------------------------------------//
// WebViewImageEx notification
//--------------------------------------------------------------//

- (void)webImageSizeChanged:(NSNotification*)notification
{
    // Get web view, data source and request
    WebView*        webView;
    WebDataSource*  dataSource;
    webView = [self selectedWebView];
    dataSource = [[webView mainFrame] dataSource];
    if (!dataSource) {
        dataSource = [[webView mainFrame] provisionalDataSource];
    }
    
    // Reset window title
    [self setValue:[dataSource pageTitle] forKey:SRMainWebViewTitle];
}

#pragma mark -
//--------------------------------------------------------------//
// SRBookmarkIconDatabase notification
//--------------------------------------------------------------//

- (void)faviconAdded:(NSNotification*)notification
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Check favicon availability
    BOOL    isFaviconUsed;
    isFaviconUsed = [defaults boolForKey:SRIconUseFavicon];
    if (isFaviconUsed) {
        // Get web view, data source and request
        WebView*        webView;
        WebDataSource*  dataSource;
        NSURLRequest*   request;
        webView = [self selectedWebView];
        dataSource = [[webView mainFrame] dataSource];
        if (!dataSource) {
            dataSource = [[webView mainFrame] provisionalDataSource];
        }
        request = [dataSource request];
        
        if (request) {
            // Get URL string
            NSString*   URLString;
            NSString*   faviconURLString;
            NSImage*    icon;
            URLString = [[request URL] absoluteString];
            faviconURLString = [[WebIconDatabase sharedIconDatabase] iconURLOfURL:URLString];
            icon = [[SRBookmarkIconDatabase sharedInstance] iconForFaviconURLString:faviconURLString];
            
            // Set icon
            if (icon) {
                [[self URLComboBox] setImage:icon];
            }
        }
    }
    
    // Update bookmark bar
    [_bookmarksBarView updateButtons];
}

#pragma mark -
//--------------------------------------------------------------//
// NSWindow delegate
//--------------------------------------------------------------//

- (void)windowDidBecomeKey:(NSNotification*)notification
{
    // Update file menu
    [[[[NSApp mainMenu] itemWithTag:SRFileTag] submenu] update];
}

- (void)windowDidBecomeMain:(NSNotification*)notification
{
    // Change first responder
    NSWindow*       window;
    WebView*        webView;
    WebDataSource*  dataSource;
    window = [self window];
    webView = [self selectedWebView];
    dataSource = [[webView mainFrame] dataSource];
    if (!dataSource) {
        [window makeFirstResponder:[self URLComboBox]];
    }

    // Invoke _showSideBar after a moment
    if (!_isResizedByJavaScript && !_isInitializedSidebarVisibility) {
        [self performSelector:@selector(_showSideBar:) withObject:self afterDelay:0.1];
        _isInitializedSidebarVisibility = YES;
    }
}

- (void)_showSideBar:(id)sender
{
    // This method will be called only once in intialize phase
    
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Check sidebar visiblity
    BOOL    isSidebarVisible;
    isSidebarVisible = [defaults boolForKey:SRMainWindowSidebarVisible];
    if (isSidebarVisible) {
        // Set sidebar visibility only for first window
        [self setSidebarVisible:YES];
    }
}

// - windowDidResignKey:

- (void)windowDidResignMain:(NSNotification*)notification
{
}

// - windowDidChangeScreen:
// - windowWillMove:

- (void)windowDidMove:(NSNotification*)notification
{
    // Check flag
    if (_isResizedByJavaScript) {
        return;
    }
    
    // Get origin
    NSWindow*   window;
    NSPoint     origin;
    window = [notification object];
    origin = [window frame].origin;
    origin.y += [window frame].size.height;
    
    // Save to defaults database
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    [defaults setObject:NSStringFromPoint(origin) 
            forKey:SRMainWindowFrameOrigin];
}

// - windowWillResize:toSize:

- (void)windowDidResize:(NSNotification*)notification
{
    // Check flag
    if (_isResizedByJavaScript) {
        return;
    }
    
    // Get size
    NSWindow*   window;
    NSSize      size;
    window = [notification object];
    size = [[window contentView] frame].size;
    
    // Save to defaults database
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    [defaults setObject:NSStringFromSize(size) 
            forKey:SRMainWindowContentSize];
}

// - windowShouldZoom:toFrame:
// - windowWillUseStandardFrame:defaultFrame:

// - windowWillMiniaturize:
// - windowDidMiniaturize:
// - windowDidDeminiaturize:
// - windowShouldClose:

- (BOOL)shouldCloseDocument
{
    return YES;
}

- (void)windowWillClose:(NSNotification*)notification
{
	// Clean up tab view
	[_srTabView removeAllItems];
	
    // Unbind from URL combo box
	[[self URLComboBox] unbind:@"value"];
    
    // Close side bar
    [[_sideBarController drawer] close];
    
    // Clear web view controller content
    [_webViewController setContent:nil];
    
    // Remove parent window of drawer
    [[_sideBarController drawer] setParentWindow:nil];
}

// - windowDidExpose:
// - windowDidUpdate:

// - windowWillBeginSheet:
// - windowDidEndSheet:
// - window:willPositionSheet:usingRect:

// - windowWillReturnFieldEditor:toObject:
// - windowWillReturnUndoManager:

#pragma mark -
//--------------------------------------------------------------//
// SRTabView delegate
//--------------------------------------------------------------//

- (void)srTabView:(SRTabView*)tabView 
        didSelectTabViewItem:(SRTabViewItem*)tabViewItem
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Check favicon availability
    BOOL    isFaviconUsed;
    isFaviconUsed = [defaults boolForKey:SRIconUseFavicon];
    
    // Get web view, data source and request
    WebView*        webView;
    WebDataSource*  dataSource;
    BOOL            isIndeterminate = NO;
    NSURLRequest*   request;
    webView = (WebView*)[tabViewItem view];
    dataSource = [[webView mainFrame] dataSource];
    if (!dataSource) {
        dataSource = [[webView mainFrame] provisionalDataSource];
        
        // If there is provisional data source, isIndeterminate is true
        isIndeterminate = dataSource != nil;
    }
    request = [dataSource request];
    
    if (!request) {
        // Set window title
        [self setValue:NSLocalizedString(@"Untitled", @"Untitled") forKey:SRMainWebViewTitle];
        // Set URL String
        [self setValue:@"" forKey:SRMainWebViewURLString];
        // Set status message
        [self setValue:@"" forKey:SRMainWebViewStatus];
        // Set loading
        [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsLoading];
        [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsIndeterminate];
        // Set icon
        [[self URLComboBox] setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
        // Set secure
        [self setValue:[NSNumber numberWithBool:NO] forKey:SRMainWebViewIsSecured];
        [_lockIconView setHidden:YES];
    }
    else {
        // Get URL string
        NSURL*      url;
        NSString*   URLString;
        url = [request URL];
        URLString = [url _web_userVisibleString];
        
        // Set window title
        [self setValue:[dataSource pageTitle] forKey:SRMainWebViewTitle];
        // Set URL String
        [self setValue:URLString forKey:SRMainWebViewURLString];
        // Set status message
        [self setValue:@"" forKey:SRMainWebViewStatus];
        // Set loading
        [self setValue:[NSNumber numberWithBool:[dataSource isLoading]] 
                forKey:SRMainWebViewIsLoading];
        [self setValue:[NSNumber numberWithBool:isIndeterminate] 
                forKey:SRMainWebViewIsIndeterminate];
        // Set icon
        if (isFaviconUsed) {
            [[self URLComboBox] setImage:
                    [[WebIconDatabase sharedIconDatabase] iconForURL:URLString 
                            withSize:NSMakeSize(16, 16) 
                            cache:YES]];
        }
        // Set secure
        [self setValue:[NSNumber numberWithBool:[[url scheme] isEqualToString:@"https"]] 
                forKey:SRMainWebViewIsSecured];
        [_lockIconView setHidden:![[url scheme] isEqualToString:@"https"]];
    }
    
    // Change first responder
    NSWindow*   window;
    window = [self window];
    if (!dataSource) {
        [window makeFirstResponder:[self URLComboBox]];
    }
    
    // Check error
    NSDictionary*   errorInfo;
    errorInfo = [tabViewItem userInfo];
    if (errorInfo) {
        // Show error dialog
        NSError*    error;
        WebFrame*   frame;
        error = [errorInfo objectForKey:@"error"];
        frame = [errorInfo objectForKey:@"frame"];
        if (error && frame) {
            [self showAlertWithError:error forFrame:frame];
        }
        
        // Clear user info
        [tabViewItem setUserInfo:nil];
    }
    
#if 0
    // Change renderers creating policy
    if (![[webView mainFrame] _shouldCreateRenderers]) {
        [[webView mainFrame] _setShouldCreateRenderers:YES];
        [[[webView mainFrame] _bridge] performSelector:@selector(addData:) withObject:[NSData data]];
        [[[webView mainFrame] _bridge] performSelector:@selector(setNeedsReapplyStyles)];
    }
    
    // Do not need to change policy for the already loaded web view?
    int i;
    for (i = 0; i < [tabView numberOfItems]; i++) {
        WebView*    hiddenWebView;
        hiddenWebView = (WebView*)[tabView viewAtIndex:i];
        if (hiddenWebView != webView && 
            [[hiddenWebView mainFrame] _shouldCreateRenderers])
        {
            [[hiddenWebView mainFrame] _setShouldCreateRenderers:NO];
        }
    }
#endif
}

- (void)srTabViewDidChangeNumberOfTabViewItems:(SRTabView*)tabView
{
    // Update menu
    [[[[NSApp mainMenu] itemWithTag:SRFileTag] submenu] update];
}

#pragma mark -
//--------------------------------------------------------------//
// NSComboBox delegate
//--------------------------------------------------------------//

- (void)comboBoxWillDismiss:(NSNotification*)notification
{
    // Get combo box
    NSComboBox* combo;
    combo = [notification object];
    
    // Select text
    [combo selectText:self];
    
    // For enter key pressing event
    NSEvent*    event;
    event = [NSApp currentEvent];
    if ([event type] == NSKeyDown) {
        // Send action only enter key is pressed
        [[combo target] performSelector:[combo action] withObject:combo];
    }
}

- (void)comboBoxWillPopUp:(NSNotification*)notification
{
    // Get combo box
    NSComboBox* combo;
    combo = [notification object];
    
    // Select first one
    if ([combo numberOfItems] > 0) {
        [combo selectItemAtIndex:0];
    }
}

//- (void)comboBoxSelectionIsChanging:(NSNotification*)notification

- (void)comboBoxSelectionDidChange:(NSNotification*)notification
{
    // Get combo box
    NSComboBox* combo;
    combo = [notification object];
    
    // Get selected index
    int selectedIndex;
    selectedIndex = [combo indexOfSelectedItem];
    
    // Select item again. It changes string value of combo box
    [combo selectItemAtIndex:selectedIndex];
}

#pragma mark -
//--------------------------------------------------------------//
// NSControl delegate
//--------------------------------------------------------------//

- (void)controlTextDidChange:(NSNotification*)notification
{
    id  control;
    control = [notification object];
    
    // For SRURLComboBox
    if ([control isKindOfClass:[SRURLComboBox class]]) {
        // Get string
        NSString*   string;
        string = [control stringValue];
        
        // Remove new line characters
        NSCharacterSet* newLineCharacterSet;
        newLineCharacterSet = [NSCharacterSet newLineCharacterSet];
        if ([string rangeOfCharacterFromSet:newLineCharacterSet 
                options:NSLiteralSearch].location != NSNotFound)
        {
            string = [string stringByTrimmingCharactersInSet:newLineCharacterSet];
            
            [control setStringValue:string];
        }
        
        // Clear image
        [control setImage:[[SRBookmarkIconDatabase sharedInstance] defaultIcon]];
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Key value observation
//--------------------------------------------------------------//

- (void)observeValueForKeyPath:(NSString*)keyPath 
        ofObject:(id)object 
        change:(NSDictionary*)change 
        context:(void *)context
{
    // Bookmark preferences
    if ([keyPath isEqualToString:SRBookmarkBookmarksBar]) {
        // Get value
        int browser;
        browser = [object integerForKey:SRBookmarkBookmarksBar];
        
        // Set bookmarks browser
        [self setBookmarksBarBrowser:browser];
        return;
    }
    
    // Tab preferences
    if ([keyPath isEqualToString:SRTabEnableTabbedBrowsing]) {
        // Get value
        BOOL    isEnabled;
        isEnabled = [object boolForKey:SRTabEnableTabbedBrowsing];
        
        // Set visible
        [self setTabVisible:isEnabled display:YES];
        
        // Update file menu
        [[[[NSApp mainMenu] itemWithTag:SRFileTag] submenu] update];
        return;
    }
    if ([keyPath isEqualToString:SRTabSelectNewTabs]) {
        // Do nothing
        return;
    }
    if ([keyPath isEqualToString:SRTabAlwaysShowTabBar]) {
        // Get value
        BOOL    isEnabled;
        isEnabled = [object boolForKey:SRTabAlwaysShowTabBar];
        
        // Set value
        [_srTabView setTabShownAlways:isEnabled];
        return;
    }
    if ([keyPath isEqualToString:SRTabAccordingToStringWidth]) {
        // Get value
        BOOL    isAccordingTo;
        isAccordingTo = [object boolForKey:SRTabAccordingToStringWidth];
        
        // Set value
        [_srTabView setAccordingToStringWidth:isAccordingTo];
        return;
    }
    
    // Securrity preference
    if ([keyPath isEqualToString:SREnableJavaScirptStatusMessage]) {
        // Get value
        BOOL    isEnabled;
        isEnabled = [object boolForKey:SREnableJavaScirptStatusMessage];
        
        // Clear status message
        if (!isEnabled) {
            [self setValue:@"" forKey:SRMainWebViewStatus];
        }
    }
    
    // Advanced preference
    if ([keyPath isEqualToString:SRUserAgent]) {
        // Get value
        int userAgent;
        userAgent = [object integerForKey:SRUserAgent];
        [self setUserAgent:userAgent];
        
        // Check SRUserAgentOtherIsUging
        BOOL    isUsing;
        isUsing = [object boolForKey:SRUserAgentOtherIsUging];
        
        if ((userAgent == SRUserAgentOther && !isUsing) || 
            (userAgent != SRUserAgentOther && isUsing))
        {
            [object setBool:!isUsing forKey:SRUserAgentOtherIsUging];
        }
        
        return;
    }
    if ([keyPath isEqualToString:SRUserAgentOtherName]) {
        // Update user agent
        [self setUserAgent:SRUserAgentOther];
        return;
    }
    
    // Icon preference
    if ([keyPath isEqualToString:SRIconName]) {
        // Get toolbar icons
        NSString*   iconName;
        iconName = [object stringForKey:SRIconName];
        [_iconImageDict release];
        _iconImageDict = [[[[NSApp delegate] iconInstaller] iconsOfName:iconName] retain];
        
        // Update toolbar icon
        NSArray*        items;
        NSEnumerator*   enumerator;
        NSToolbarItem*  item;
        items = [[[self window] toolbar] items];
        enumerator = [items objectEnumerator];
        while (item = [enumerator nextObject]) {
            NSImage*    image;
            image = [_iconImageDict objectForKey:[item itemIdentifier]];
            if (image) {
                [item setImage:image];
            }
        }
    }
    if ([keyPath isEqualToString:SRIconUseFavicon] || 
        [keyPath isEqualToString:SRIconUseFaviconBookmarkBar])
    {
        // Update bookmark bar
        [_bookmarksBarView updateButtons];
        [_bookmarksBarView setNeedsDisplay:YES];
    }
    
    // URL string
    if ([keyPath isEqualToString:SRMainWebViewURLString]) {
        // Get URL combo box
        SRURLComboBox*  comboBox;
        comboBox = [self URLComboBox];
        if (!comboBox) {
            return;
        }
        
        BOOL    showIDNButton = NO;
        
        // Check URL string
        if ([_webViewURLString length] > 0) {
            // Create NSURL and get host
            NSURL*      url;
            NSString*   host;
            url = [NSURL _web_URLWithUserTypedString:_webViewURLString];
            host = [url host];
            // For Punycode host
            if (host && [host rangeOfString:@"xn--"].location != NSNotFound) {
                showIDNButton = YES;
            }
        }
        
        // Get URL IDN button
        NSButton*   button;
        button = [comboBox buttonWithTag:SRURLButtonIDN];
        
        // Hide button
        if (!showIDNButton) {
            if (button) {
                [comboBox removeButton:button];
            }
        }
        // Show button
        else {
            if (!button) {
                button = [comboBox addButtonWithSize:NSMakeSize(16, 16)];
                [button setTag:SRURLButtonIDN];
                [button setBordered:NO];
                [button setButtonType:NSMomentaryLight];
                [button setImage:[NSImage imageNamed:@"idn"]];
                [button setTarget:self];
                [button setAction:@selector(toggleURLIDNAction:)];
            }
        }
    }
}

@end

SROpenActionType SROpenActionTypeFromModifierFlags(
        unsigned modifierFlags)
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get modifier key flag
    unsigned int    cmdFlag, optionFlag, shiftFlag;
    cmdFlag = modifierFlags & NSCommandKeyMask;
    optionFlag = modifierFlags & NSAlternateKeyMask;
    shiftFlag = modifierFlags & NSShiftKeyMask;
    
    // For option key
    if (!cmdFlag && optionFlag && !shiftFlag) {
        return SROpenOptionAction;
    }
    
    // Check tab browsing
    BOOL    enableTabbedBrowsing, selectNewTabs;
    enableTabbedBrowsing = [defaults boolForKey:SRTabEnableTabbedBrowsing];
    selectNewTabs = [defaults boolForKey:SRTabSelectNewTabs];
    
    // If tabbed browsing is enable
    if (enableTabbedBrowsing) {
        // If select new tabs
        if (selectNewTabs) {
            // Open in new tabs and select it
            if (cmdFlag && !optionFlag && !shiftFlag) {
                return SROpenInNewTabAction;
            }
            // Open in new tabs
            else if (cmdFlag && !optionFlag && shiftFlag) {
                return SROpenInNewBackgroundTabAction;
            }
            // Open in new window
            else if (cmdFlag && optionFlag && !shiftFlag) {
                return SROpenInNewWindowAction;
            }
            // Open in new window in background
            else if (cmdFlag && optionFlag && shiftFlag) {
                return SROpenInNewBackgroundWindowAction;
            }
        }
        // If not select new tabs
        else {
            // Open in new tabs
            if (cmdFlag && !optionFlag && !shiftFlag) {
                return SROpenInNewBackgroundTabAction;
            }
            // Open in new tabs and select it
            else if (cmdFlag && !optionFlag && shiftFlag) {
                return SROpenInNewTabAction;
            }
            // Open in new window in background
            else if (cmdFlag && optionFlag && !shiftFlag) {
                return SROpenInNewBackgroundWindowAction;
            }
            // Open in new window
            else if (cmdFlag && optionFlag && shiftFlag) {
                return SROpenInNewWindowAction;
            }
        }
    }
    // If tabbed browsing is disable
    else {
        // Open in new window
        if (cmdFlag && !shiftFlag) {
            return SROpenInNewWindowAction;
        }
        // Open in new window in background
        else if (cmdFlag && shiftFlag) {
            return SROpenInNewBackgroundWindowAction;
        }
    }
    
    return SROpenAction;
}
