/*
SRMainDocument.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRMainDocument.h"

NSString*   SRCSSDocumentType = @"CSS style sheet";
NSString*   SRGIFDocumentType = @"GIF image";
NSString*   SRHTMLDocumentType = @"HTML document";
NSString*   SRHTMLNoExtensionDocumentType = @"HTML document (no extension)";
NSString*   SRJavaScriptDocumentType = @"JavaScript script";
NSString*   SRJPEGDocumentType = @"JPEG image";
NSString*   SRJPEG2000DocumentType = @"JPEG 2000 image";
NSString*   SRTextDocumentType = @"Plain text document";
NSString*   SRPNGDocumentType = @"PNG image";
NSString*   SRRTFDocumentType = @"Rich Text Format (RTF) document";
NSString*   SRTIFFDocumentType = @"TIFF image";
NSString*   SRURLDocumentType = @"Web site location";
NSString*   SRWeblocDocumentType = @"Web location";
NSString*   SRICODocumentType = @"Windows icon image";
NSString*   SRXHTMLDocumentType = @"XHTML document";
NSString*   SRXMLDocumentType = @"XML document";

@implementation SRMainDocument

//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (void)_init
{
    // Load nib
    if (![NSBundle loadNibNamed:@"MainDocument" owner:self]) {
        // Fatal
        SR_FATAL(@"Could not load MainDocument.nib");
    }
    
    // Register key value observation
    [_content addObserver:self 
            forKeyPath:@"printBackground" 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize
    [self _init];
    
    return self;
}

- (id)initWithContentsOfFile:(NSString*)fileName ofType:(NSString*)docType
{
    self = [super initWithContentsOfFile:fileName ofType:docType];
    if (!self) {
        return nil;
    }
    
    // Initialize
    [self _init];
    if (![[fileName pathExtension] isEqualToString:@"webloc"]) {
        _initialFileName = [fileName retain];
    }
    
    return self;
}

- (id)initWithContentsOfURL:(NSURL*)URL ofType:(NSString*)docType
{
    self = [super initWithContentsOfURL:URL ofType:docType];
    if (!self) {
        return nil;
    }
    
    // Initialize
    [self _init];
    
    return self;
}

- (void)dealloc
{
    [_initialFileName release];
    _initialFileName = nil;
    [_initialURLString release];
    _initialURLString = nil;
    
    // Remove observers
    [_content removeObserver:self forKeyPath:@"printBackground"];
    
    [super dealloc];
}

- (void)makeWindowControllers
{
    [self makeWindowControllersWithInitialFrameName:nil groupName:nil];
}

- (void)makeWindowControllersWithInitialFrameName:(NSString*)frameName 
        groupName:(NSString*)groupName
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Create main window controller
    SRMainWindowController* mainWindowController;
    mainWindowController = [[SRMainWindowController alloc] 
            initWithWindowNibName:@"MainWindow" initialFrameName:frameName initialGroupName:groupName];
    [self addWindowController:mainWindowController];
    [mainWindowController release];
    
    // Check file name
#if 1
    if (_initialFileName) {
        // Open file
        [mainWindowController openURL:[NSURL fileURLWithPath:_initialFileName]];
    }
    else if (_initialURLString) {
        // Open URL
        [mainWindowController openURLString:_initialURLString];
    }
#else
    NSString*   fileName;
    fileName = [self fileName];
    if (fileName) {
        // Open file
        NSURLRequest*   request;
        request = [NSURLRequest requestWithURL:[NSURL fileURLWithPath:fileName] 
                cachePolicy:NSURLRequestUseProtocolCachePolicy 
                timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
        [mainWindowController openRequest:request];
    }
#endif
}

- (SRMainWindowController*)mainWindowController
{
    // Get window controllers
    NSArray*    windowControllers;
    windowControllers = [self windowControllers];
    if (!windowControllers || [windowControllers count] < 1) {
        return nil;
    }
    
    // Get first window controller
    id  windowController;
    windowController = [windowControllers objectAtIndex:0];
    if (![windowController isKindOfClass:[SRMainWindowController class]]) {
        return nil;
    }
    
    return windowController;
}

- (BOOL)readFromFile:(NSString*)fileName 
        ofType:(NSString*)docType
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    if ([[fileName pathExtension] isEqualToString:@"webloc"]) {
        // Get text and URL of webloc
        NSArray*    textAndURL;
        textAndURL = SRTextAndURLOfWeblocFile(fileName);
        if (!textAndURL || [textAndURL count] != 2) {
            return NO;
        }
        
        // Open URL
        if ([self mainWindowController]) {
            [[self mainWindowController] openURLString:[textAndURL objectAtIndex:1]];
        }
        else {
            _initialURLString = [[textAndURL objectAtIndex:1] retain];
        }
        
        return YES;
    }
    
    if ([docType isEqualToString:SRCSSDocumentType] || 
        [docType isEqualToString:SRGIFDocumentType] || 
        [docType isEqualToString:SRHTMLDocumentType] || 
        [docType isEqualToString:SRHTMLNoExtensionDocumentType] || 
        [docType isEqualToString:SRJavaScriptDocumentType] || 
        [docType isEqualToString:SRJPEGDocumentType] || 
        [docType isEqualToString:SRJPEG2000DocumentType] || 
        [docType isEqualToString:SRTextDocumentType] || 
        [docType isEqualToString:SRPNGDocumentType] || 
        [docType isEqualToString:SRRTFDocumentType] || 
        [docType isEqualToString:SRTIFFDocumentType] || 
        [docType isEqualToString:SRICODocumentType] || 
        [docType isEqualToString:SRXHTMLDocumentType] || 
        [docType isEqualToString:SRXMLDocumentType])
    {
        // Open URL request
        NSURLRequest*   request;
        request = [NSURLRequest requestWithURL:[NSURL fileURLWithPath:fileName] 
                cachePolicy:NSURLRequestUseProtocolCachePolicy 
                timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
        [[self mainWindowController] openRequest:request];
        
        return YES;
    }
    
    return NO;
}

- (BOOL)readFromURL:(NSURL*)URL
		ofType:(NSString*)docType
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    if ([docType isEqualToString:SRCSSDocumentType] || 
        [docType isEqualToString:SRGIFDocumentType] || 
        [docType isEqualToString:SRHTMLDocumentType] || 
        [docType isEqualToString:SRHTMLNoExtensionDocumentType] || 
        [docType isEqualToString:SRJavaScriptDocumentType] || 
        [docType isEqualToString:SRJPEGDocumentType] || 
        [docType isEqualToString:SRJPEG2000DocumentType] || 
        [docType isEqualToString:SRTextDocumentType] || 
        [docType isEqualToString:SRPNGDocumentType] || 
        [docType isEqualToString:SRRTFDocumentType] || 
        [docType isEqualToString:SRTIFFDocumentType] || 
        [docType isEqualToString:SRICODocumentType] || 
        [docType isEqualToString:SRXHTMLDocumentType] || 
        [docType isEqualToString:SRXMLDocumentType])
    {
        // Open URL request
        NSURLRequest*   request;
        request = [NSURLRequest requestWithURL:URL 
                cachePolicy:NSURLRequestUseProtocolCachePolicy 
                timeoutInterval:[defaults integerForKey:SRGeneralTimeout]];
        [[self mainWindowController] openRequest:request];
        
		return YES;
	}
	return NO;
}

- (BOOL)writeToFile:(NSString*)fileName 
        ofType:(NSString*)docType
{
    // Return web data source data
    WebView*    webView;
    webView = [[self mainWindowController] selectedWebView];
    if (webView) {
        // Get data
        NSData* data;
        data = [[[webView mainFrame] dataSource] data];
        
        // Write it to file
        return [data writeToFile:fileName atomically:YES];
    }
    
    return NO;
}

- (NSString*)fileName
{
    // Check data source of web view
    if ([[[[self mainWindowController] selectedWebView] mainFrame] dataSource]) {
        return [[[self mainWindowController] window] title];
    }
    
    return @"";
}

- (NSString*)displayName
{
    // Check data source of web view
    if ([[[[self mainWindowController] selectedWebView] mainFrame] dataSource]) {
        return [[[self mainWindowController] window] title];
    }
    
    return [super displayName];
}

- (NSString*)fileType
{
    // Get selected web view
    WebView*    webView;
    webView = [[self mainWindowController] selectedWebView];
    if (!webView) {
        return nil;
    }
    
    // Get file extension
    NSString*   fileName;
    NSString*   fileExtension;
    fileName = [[[[[[webView mainFrame] dataSource] request] URL] path] lastPathComponent];
    fileExtension = [fileName pathExtension];
    
    // Get file type from extension
    NSString*   fileType;
    if (fileExtension && [fileExtension length] > 0) {
        fileType = [[NSDocumentController sharedDocumentController] typeFromFileExtension:fileExtension];
        if (fileType) {
            return fileType;
        }
    }
    
    // Get MIME type
    NSString*   mimeType;
    mimeType = [[[[webView mainFrame] dataSource] response] MIMEType];
    
    // Get file type from MIME type
    if (mimeType && [mimeType length] > 0) {
        fileType = [[NSDocumentController sharedDocumentController] typeFromMIMEType:mimeType];
        if (fileType) {
            return fileType;
        }
    }
    
    // Return MIME type as file type
    return mimeType;
}

- (BOOL)isDocumentEdited
{
    // Never edited
    return NO;
}

- (void)updateChangeCount:(NSDocumentChangeType)changeType
{
    // Do nothing
}

- (BOOL)prepareSavePanel:(NSSavePanel*)savePanel
{
    // Remove accessory view
    [savePanel setAccessoryView:nil];
    
    return YES;
}

#pragma mark -
//--------------------------------------------------------------//
// Accessors
//--------------------------------------------------------------//

- (NSView*)printAccessoryView
{
    return _printView;
}

#pragma mark -
//--------------------------------------------------------------//
// Printing
//--------------------------------------------------------------//

- (void)printShowingPrintPanel:(BOOL)flag
{
    NSPrintOperation*   printOperation;
    
    // Get selected web view
    WebView*    webView;
    id          documentView;
    webView = [[self mainWindowController] selectedWebView];
    if (!webView) {
        return;
    }
    documentView = [[[webView mainFrame] frameView] documentView];
    
    // Get print info
    NSPrintInfo*    printInfo;
    printInfo = [self printInfo];
    
    [printInfo setTopMargin:15.0];
    [printInfo setBottomMargin:15.0];
    [printInfo setLeftMargin:15.0];
    [printInfo setRightMargin:15.0];
    [printInfo setHorizontallyCentered:NO];
    [printInfo setVerticallyCentered:NO];
    
    // Set properties of print panel
    WebPreferences* preferences;
    preferences = [SRPreferencesController defaultWebPreferences];
    [self setValue:[NSNumber numberWithBool:[preferences shouldPrintBackgrounds]] 
            forKey:@"printBackground"];
    
    // Create print operation
    printOperation = [NSPrintOperation printOperationWithView:documentView printInfo:printInfo];
    [printOperation setShowPanels:flag];
    [printOperation setCanSpawnSeparateThread:YES];
    [printOperation setAccessoryView:_printView];
    
    // Run print operation
    [printOperation runOperationModalForWindow:[[self mainWindowController] window] 
            delegate:self 
            didRunSelector:@selector(_printOperationDidRun:success:contextInfo:) 
            contextInfo:NULL];
}

- (void)_printOperationDidRun:(NSPrintOperation*)printOperation 
        success:(BOOL)success 
        contextInfo:(void*)contextInfo
{
}

#pragma mark -
//--------------------------------------------------------------//
// Accessors
//--------------------------------------------------------------//

- (void)setPrintBackground:(BOOL)isPrintBackground
{
    // Set web preferences
    WebPreferences* preferences;
    preferences = [SRPreferencesController defaultWebPreferences];
    [preferences setShouldPrintBackgrounds:isPrintBackground];
}

#pragma mark -
//--------------------------------------------------------------//
// AppleScript accessor methods
//--------------------------------------------------------------//

- (NSString*)source
{
    // Get selected web view
    WebView*    webView;
    webView = [[self mainWindowController] selectedWebView];
    if (!webView) {
        return @"";
    }
    
    // Get source
    NSString*   source;
    source = [[[[webView mainFrame] dataSource] representation] documentSource];
    if (!source) {
        return @"";
    }
    
    return source;
}

- (NSString*)text
{
    // Get selected web view
    WebView*    webView;
    webView = [[self mainWindowController] selectedWebView];
    if (!webView) {
        return @"";
    }
    
    // Get text
    id  documentView;
    documentView = [[[webView mainFrame] frameView] documentView];
    if (![documentView respondsToSelector:@selector(string)]) {
        return @"";
    }
    
    return [documentView string];
}

- (NSString*)URLString
{
    // Get selected web view
    WebView*    webView;
    webView = [[self mainWindowController] selectedWebView];
    if (!webView) {
        return @"";
    }
    
    // Get URL string
    NSString*   URLString;
    URLString = [[[[[webView mainFrame] dataSource] request] URL] _web_userVisibleString];
    if (!URLString) {
        return @"";
    }
    
    return URLString;
}

- (void)setURLString:(NSString*)URLString
{
    // Open URL string
    [[self mainWindowController] openURLString:URLString];
}

- (NSData*)PDF
{
    // Get selected web view and document view
    WebView*    webView;
    NSView*     documentView;
    webView = [[self mainWindowController] selectedWebView];
    documentView = [[[webView mainFrame] frameView] documentView];
    if (!documentView) {
        return nil;
    }
    
    // Get PDF data
    return [documentView dataWithPDFInsideRect:[documentView frame]];
}

#pragma mark -
//--------------------------------------------------------------//
// Menu validation
//--------------------------------------------------------------//

- (BOOL)validateMenuItem:(NSMenuItem*)menuItem
{
    // Get tag
    int tag;
    tag = [menuItem tag];
    
    switch (tag) {
    // File menu
    case SRSaveAsTag: {
        // Get selected web view
        WebView*    webView;
        webView = [[self mainWindowController] selectedWebView];
        if (!webView) {
            return NO;
        }
        
        // Get data source
        WebDataSource*  dataSource;
        dataSource = [[webView mainFrame] dataSource];
        if (!dataSource) {
            return NO;
        }
        
        return [dataSource data] != nil;
    }
    }
    
    return [super validateMenuItem:menuItem];
}

#pragma mark -
//--------------------------------------------------------------//
// Key value observation
//--------------------------------------------------------------//

- (void)observeValueForKeyPath:(NSString*)keyPath 
        ofObject:(id)object 
        change:(NSDictionary*)change 
        context:(void *)context
{
    // For print controller
    if (object == _content) {
        // Print background
        if ([keyPath isEqualToString:@"printBackground"]) {
            [self setPrintBackground:[[change objectForKey:@"new"] boolValue]];
            return;
        }
    }
}

@end

// Utility
NSArray* SRTextAndURLOfWeblocFile(
        NSString* weblocFile)
{
    OSErr       err;
    OSStatus    status;
    
    // Create FSSpec
    Boolean isDir;
    FSRef   fsRef;
    FSSpec  fsSpec;
    status = FSPathMakeRef([weblocFile fileSystemRepresentation], &fsRef, &isDir);
    if (status != noErr || isDir) {
        return nil;
    }
    err = FSGetCatalogInfo(&fsRef, kFSCatInfoNone, NULL, NULL, &fsSpec, NULL);
    if (err != noErr) {
        return nil;
    }
    
    // Open resource file
    short   fileRef;
    fileRef = FSpOpenResFile(&fsSpec, 0);
    
    short   resCount;
    Handle  handle;
    
    // Get 'TEXT' resource
    NSString*   text = nil;
    resCount = Count1Resources('TEXT');
    if (resCount == 1) {
        handle = Get1IndResource('TEXT', 1);
        if (handle) {
            // Create NSString
            text = [NSString stringWithCString:*handle length:GetHandleSize(handle)];
        }
    }
    
    // Get 'url ' resource
    NSString*   URLString = nil;
    resCount = Count1Resources('url ');
    if (resCount == 1) {
        handle = Get1IndResource('url ', 1);
        if (handle) {
            // Create NSString
            URLString = [NSString stringWithCString:*handle length:GetHandleSize(handle)];
        }
    }
    
    // Close resource file
    CloseResFile(fileRef);
    
    if (text && URLString) {
        return [NSArray arrayWithObjects:text, URLString, nil];
    }
    
    return nil;
}
