#import "SRBookmarkStorage.h"
#import "SRBookmarkButton.h"
#import "SRBookmarksBar.h"

static const int    SRBookmarksBarEndMarginWidth = 8;
static const int    SRBookmarksBarButtonMarginWidth = 4;

@implementation SRBookmarksBar

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        // Initialize instance variables
        _bookmarkButtons = [[NSMutableArray array] retain];
        _indicatorIndex = -1;
        
        // Register accept drag type
        NSArray*    types;
        types = [NSArray arrayWithObjects:SRBookmarkPboardType, nil];
        [self registerForDraggedTypes:types];
    }
    
    return self;
}

- (void)dealloc
{
    [_bookmarkButtons release];
    [super dealloc];
}

#pragma mark -
//--------------------------------------------------------------//
// Bookmark buttons
//--------------------------------------------------------------//

- (void)_updateButtons
{
    // Remove all buttons
    [[self subviews] makeObjectsPerformSelector:@selector(removeFromSuperviewWithoutNeedingDisplay)];
    
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Add buttons
    int                 currentX = SRBookmarksBarEndMarginWidth;
    NSEnumerator*       enumerator;
    SRBookmarkButton*   bookmarkButton;
    enumerator = [_bookmarkButtons objectEnumerator];
    while (bookmarkButton = [enumerator nextObject]) {
        // Get bookmark button size
        NSSize  buttonSize;
        buttonSize = [[bookmarkButton cell] cellSize];
        
        if (currentX + buttonSize.width + SRBookmarksBarEndMarginWidth < frame.size.width) {
            // Add button
            [self addSubview:bookmarkButton];
            [bookmarkButton setFrame:NSMakeRect(
                    currentX, 
                    (frame.size.height - buttonSize.height) / 2, 
                    buttonSize.width, 
                    buttonSize.height)];
            currentX += buttonSize.width + SRBookmarksBarButtonMarginWidth;
        }
    }
    
    [self setNeedsDisplay:YES];
}

- (int)_insertIndexAtPoint:(NSPoint)location
{
    int                 index = 0;
    NSEnumerator*       enumerator;
    SRBookmarkButton*   bookmarkButton;
    enumerator = [_bookmarkButtons objectEnumerator];
    while (bookmarkButton = [enumerator nextObject]) {
        // Check button is clipped or not
        if ([bookmarkButton isClipMenu]) {
            break;
        }
        
        // Get button frame
        NSRect  buttonFrame;
        buttonFrame = [bookmarkButton frame];
        
        // Make indicataion rect
        NSRect  rect;
        rect.origin = NSZeroPoint;
        rect.size.width = buttonFrame.origin.x + buttonFrame.size.width / 2;
        rect.size.height = [self frame].size.height;
        
        if (NSPointInRect(location, rect)) {
            return index;
        }
        
        index++;
    }
    
    return index;
}

- (void)_updateInsertIndicatorAtPoint:(NSPoint)location
{
    // Get insert index
    _indicatorIndex = [self _insertIndexAtPoint:location];
    
    // Update view
    [self setNeedsDisplay:YES];
}

- (SRBookmarkButton*)_createBookmarkButtonOfBookmark:(NSMutableDictionary*)bookmark
{
    // Create bookmark button
    SRBookmarkButton*   bookmarkButton;
    bookmarkButton = [[SRBookmarkButton alloc] initWithFrame:NSZeroRect];
    [bookmarkButton setBookmark:bookmark];
    [bookmarkButton setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
    [bookmarkButton setTarget:nil];
    [bookmarkButton setAction:@selector(openBookmarkOfButtonAction:)];
    
    return bookmarkButton;
}

- (void)addBookmark:(NSMutableDictionary*)bookmark
{
    // Create bookmark button
    SRBookmarkButton*   bookmarkButton;
    bookmarkButton = [self _createBookmarkButtonOfBookmark:bookmark];
    
    // Add bookmark button
    [_bookmarkButtons addObject:bookmarkButton];
    
    // Update buttons
    [self _updateButtons];
}

- (void)insertBookmark:(NSMutableDictionary*)bookmark atIndex:(int)index
{
    // Create bookmark button
    SRBookmarkButton*   bookmarkButton;
    bookmarkButton = [self _createBookmarkButtonOfBookmark:bookmark];
    
    // Insert bookmark button
    [_bookmarkButtons insertObject:bookmarkButton atIndex:index];
    
    // Update buttons
    [self _updateButtons];
}

- (void)removeBookmark:(NSMutableDictionary*)bookmark
{
}

- (void)drawRect:(NSRect)rect
{
    // Clear rect
    [[NSColor windowBackgroundColor] set];
    NSRectFill(rect);
    
    // Draw indicator
    if (_indicatorIndex != -1) {
        // Get self frame
        NSRect  frame;
        frame = [self frame];
        
        // Get bookmark button frame
        SRBookmarkButton*   bookmarkButton;
        NSRect              bookmarkFrame;
        if (_indicatorIndex < [_bookmarkButtons count]) {
            bookmarkButton = [_bookmarkButtons objectAtIndex:_indicatorIndex];
            bookmarkFrame = [bookmarkButton frame];
        }
        else {
            bookmarkFrame.origin.x = frame.size.width;
        }
        
        // Draw indicator
        [[NSColor blackColor] set];
        NSRectFill(NSMakeRect(bookmarkFrame.origin.x - 3, 7, 2, frame.size.height - 10));
        
        NSBezierPath*   path;
        path = [NSBezierPath bezierPathWithOvalInRect:NSMakeRect(bookmarkFrame.origin.x - 4, 3, 4, 4)];
        [path stroke];
    }
}

- (void)setFrame:(NSRect)frame
{
    [super setFrame:frame];
    
    [self _updateButtons];
}

#pragma mark -
//--------------------------------------------------------------//
// NSDraggingDestination protocol
//--------------------------------------------------------------//

- (NSDragOperation)draggingEntered:(id<NSDraggingInfo>)info
{
    NSPoint location;
    location = [self convertPoint:[info draggingLocation] fromView:nil];
    [self _updateInsertIndicatorAtPoint:location];
    return NSDragOperationCopy;
}

- (NSDragOperation)draggingUpdated:(id<NSDraggingInfo>)info
{
    NSPoint location;
    location = [self convertPoint:[info draggingLocation] fromView:nil];
    [self _updateInsertIndicatorAtPoint:location];
    return NSDragOperationCopy;
}

- (void)draggingEnded:(id<NSDraggingInfo>)info
{
    _indicatorIndex = -1;
    [self setNeedsDisplay:YES];
}

- (void)draggingExited:(id<NSDraggingInfo>)info
{
    _indicatorIndex = -1;
    [self setNeedsDisplay:YES];
}

- (BOOL)prepareForDragOperation:(id<NSDraggingInfo>)info
{
    return YES;
}

- (BOOL)performDragOperation:(id<NSDraggingInfo>)info
{
    // Check dragged data type
    NSPasteboard*   pboard;
    pboard = [info draggingPasteboard];
    if (![[pboard types] containsObject:SRBookmarkPboardType]) {
        // Do not accept drag
        _indicatorIndex = -1;
        [self setNeedsDisplay:YES];
        return NO;
    }
    
    // Unarchive bookmark data
    NSData*                 bookmarkData;
    NSMutableDictionary*    bookmark;
    bookmarkData = [pboard dataForType:SRBookmarkPboardType];
    if (!bookmarkData) {
        _indicatorIndex = -1;
        [self setNeedsDisplay:YES];
        return NO;
    }
    bookmark = [NSUnarchiver unarchiveObjectWithData:bookmarkData];
    
    // Insert bookmark
    if (_indicatorIndex != -1) {
        [self insertBookmark:bookmark atIndex:_indicatorIndex];
    }
    
    return YES;
}

- (void)concludeDragOperation:(id<NSDraggingInfo>)info
{
    _indicatorIndex = -1;
    [self setNeedsDisplay:YES];
}

@end
