/*
SRAnimation.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRAnimation.h"

@implementation SRAnimation 

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (id)initWithDuration:(NSTimeInterval)duration 
        animationCurve:(SRAnimationCurve)animationCurve
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initalize instance variables
    _duration = 1.0;
    _currentProgress = 0.0f;
    _frameRate = 30.0f;
    _delegate = nil;
    _timer = nil;
    _startTime = 0;
    
    _progressMarks = [[NSMutableArray array] retain];
    _nextProgressMarkIndex = 0;
    
    _animationCurve = animationCurve;
    
    return self;
}

- (void)dealloc
{
    [_progressMarks release];
    
    [super dealloc];
}

#pragma mark -
//--------------------------------------------------------------//
// Configure
//--------------------------------------------------------------//

- (id)delegate
{
    return _delegate;
}

- (void)setDelegate:(id)delegate
{
    _delegate = delegate;
}

- (NSTimeInterval)duration
{
    return _duration;
}

- (void)setDuration:(NSTimeInterval)duration
{
    _duration = duration;
}

- (float)frameRate
{
    return _frameRate;
}

- (void)setFrameRate:(float)frameRate
{
    _frameRate = frameRate;
}

#pragma mark -
//--------------------------------------------------------------//
// Controlling and monitoring
//--------------------------------------------------------------//

- (void)startAnimation
{
    // Prepare for animation
    _startTime = [NSDate timeIntervalSinceReferenceDate];
    _nextProgressMarkIndex = 0;
    
    // Schedule timer
    _timer = [[NSTimer scheduledTimerWithTimeInterval:1.0f / _frameRate 
            target:self 
            selector:@selector(_timerFired:) 
            userInfo:nil 
            repeats:YES] retain];
}

- (void)_timerFired:(id)userInfo
{
    // Calc elpased time
    NSTimeInterval  elpased;
    elpased = [NSDate timeIntervalSinceReferenceDate] - _startTime;
    
    // Get next progress mark
    float   progressMark;
    progressMark = [[_progressMarks objectAtIndex:_nextProgressMarkIndex] floatValue];
    
    if (elpased < _duration * progressMark) {
        return;
    }
    
    // Find reached progress mark
    while (_nextProgressMarkIndex < [_progressMarks count]) {
        _nextProgressMarkIndex++;
        
        float   nextProgressMark;
        nextProgressMark = [[_progressMarks objectAtIndex:_nextProgressMarkIndex] floatValue];
        if (elpased < _duration * nextProgressMark) {
            // Iinvoke delegate
            if (_delegate && [_delegate respondsToSelector:@selector(animation:didReachProgressMark:)]) {
                [_delegate animation:self didReachProgressMark:progressMark];
                return;
            }
        }
        
        progressMark = nextProgressMark;
    }
    
    // Animation end
    if (_delegate && [_delegate respondsToSelector:@selector(animationDidEnd:)]) {
        [_delegate animationDidEnd:self];
    }
    
    [_timer invalidate];
    [_timer release];
}

- (void)stopAnimation
{
}

- (BOOL)isAnimating
{
    return _timer != nil;
}

- (SRAnimationProgress)currentProgress
{
    return _currentProgress;
}

- (void)setCurrentProgress:(SRAnimationProgress)progress
{
    _currentProgress = progress;
}

- (float)currentValue
{
    switch (_animationCurve) {
    case SRAnimationLinear: return _currentProgress;
    }
    
    return _currentProgress;
}

#pragma mark -
//--------------------------------------------------------------//
// Managing progress marks
//--------------------------------------------------------------//

- (NSArray*)progressMarks
{
    return [NSArray arrayWithArray:_progressMarks];
}

- (void)setProgressMarks:(NSArray*)progressMarks
{
    [_progressMarks removeAllObjects];
    [_progressMarks addObjectsFromArray:progressMarks];
}

@end
