/*
SRSourceWindowController.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRSourceWindowController.h"

static NSMutableArray* _SRSourceWindows;

enum {
    _SRHTTPHeaderInitialRequestTag = 0, 
    _SRHTTPHeaderFinalRequestTag, 
    _SRHTTPHeaderRequestTag, 
    _SRHTTPHeaderResposneTag, 
};

@implementation SRSourceWindowController

+ (NSMutableArray*)sourceWindows
{
    if(_SRSourceWindows==nil){
        _SRSourceWindows=[[NSMutableArray alloc]initWithCapacity:8];
    }
    return _SRSourceWindows;
}

+ (SRSourceWindowController*)sourceWindowForURL:(NSURL*)URL
{
    NSMutableArray* sourceWindows=[SRSourceWindowController sourceWindows];
    int i, cnt=[sourceWindows count];
    for(i=0;i<cnt;i++){
        SRSourceWindowController*   srcCtl=[sourceWindows objectAtIndex:i];
        NSURL*  aURL=[srcCtl URL];
        if([URL isEqual:aURL]) return srcCtl;
    }
    return nil;
}

+ (SRSourceWindowController*)sourceWindowWithWebDataSource:(WebDataSource*)dataSource
{
    SRSourceWindowController*   sourceWindowController;
    NSURL*  URL=[[dataSource request]URL];
    
    //check
    if(![[dataSource representation]canProvideDocumentSource]){
        return nil;
    }
    
    sourceWindowController=[SRSourceWindowController sourceWindowForURL:URL];
    
    if (!sourceWindowController) {
        sourceWindowController = [[SRSourceWindowController alloc] 
                initWithWindowNibName:@"SourceCode"];
        if (!sourceWindowController) {
            // Fatal error
            SR_FATAL(@"Could not load SourceCode.nib");
        }
        [sourceWindowController autorelease];
        [[SRSourceWindowController sourceWindows]addObject:sourceWindowController];
        [sourceWindowController loadWindow];
    }
    [sourceWindowController setUpWithWebDataSource:dataSource];
    
    return sourceWindowController;
}

- (void)awakeFromNib
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Register key value observation
    [defaults addObserver:self 
            forKeyPath:SRSourceFontName 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRSourceFontSize 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRSourceDefaultColor 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRSourceBackgroundColor 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRSourceTagColor 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    [defaults addObserver:self 
            forKeyPath:SRSourceCommentColor 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    
    [[self window]setFrameAutosaveName:@"SRSourceWindow"];
    [[self window]setFrameUsingName:@"SRSourceWindow"];
}

- (void)dealloc
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    [defaults removeObserver:self forKeyPath:SRSourceFontName];
    [defaults removeObserver:self forKeyPath:SRSourceFontSize];
    [defaults removeObserver:self forKeyPath:SRSourceDefaultColor];
    [defaults removeObserver:self forKeyPath:SRSourceBackgroundColor];
    [defaults removeObserver:self forKeyPath:SRSourceTagColor];
    [defaults removeObserver:self forKeyPath:SRSourceCommentColor];
    
    [_dataSource release];
    
    [super dealloc];
}

- (WebDataSource*)dataSource
{
    return _dataSource;
}

- (void)setUpWithWebDataSource:(WebDataSource*)dataSource;
{
    if(_dataSource != dataSource){
        id<WebDocumentRepresentation>   representation;
        NSString*                       documentSource;

        [_dataSource release];
        _dataSource=[dataSource retain];

        if (!dataSource) {
            return;
        }

        // Get page source        
        representation = [dataSource representation];
        if (!representation) {
            return;
        }
        documentSource = [representation documentSource];
        if (!documentSource) {
            return;
        }
        [self setSource:documentSource];

        // Configure header popup
        [_headerKindPopup removeAllItems];
        
        NSURLRequest*   initialRequest;
        NSURLRequest*   request;
        initialRequest = [dataSource initialRequest];
        request = [dataSource request];
        if (request && [[request allHTTPHeaderFields] count] > 0) {
#if 1
            [_headerKindPopup addItemWithTitle:NSLocalizedString(@"Request", @"Request")];
            [[_headerKindPopup lastItem] setTag:_SRHTTPHeaderRequestTag];
#else
            if (initialRequest && [[initialRequest allHTTPHeaderFields] count] > 0) {
                [_headerKindPopup addItemWithTitle:@"Initial Request"];
                [[_headerKindPopup lastItem] setTag:_SRHTTPHeaderInitialRequestTag];
                [_headerKindPopup addItemWithTitle:@"Final Request"];
                [[_headerKindPopup lastItem] setTag:_SRHTTPHeaderFinalRequestTag];
            }
            else {
                [_headerKindPopup addItemWithTitle:NSLocalizedString(@"Request", @"Request")];
                [[_headerKindPopup lastItem] setTag:_SRHTTPHeaderRequestTag];
            }
#endif
        }
        if ([dataSource response]) {
            [_headerKindPopup addItemWithTitle:NSLocalizedString(@"Response", @"Response")];
            [[_headerKindPopup lastItem] setTag:_SRHTTPHeaderResposneTag];
        }
        
        [_HTTPHeadersTable reloadData];
        
        if([self title]) [[self window]setTitle:[self title]];
    }
}

- (NSString*)title
{
    return [_dataSource pageTitle];
}

- (NSURL*)URL
{
    return [[_dataSource request]URL];
}

- (NSTextView*)sourceTextView
{
    return _sourceTextView;
}

- (void)setSource:(NSString*)source
{
	[self setColorizedSourceWithString:source];
    [_sourceTextView setSelectedRange:NSMakeRange(0, 0)];
}

//colorize source
- (void)setColorizedSourceWithString:(NSString*)source
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
	if(1 /* setting? */){
		NSColor* txtColor=[NSUnarchiver unarchiveObjectWithData:[defaults objectForKey:SRSourceDefaultColor]];
        if (!txtColor) {
            txtColor = [NSColor blackColor];
        }
		NSColor* tagColor=[NSUnarchiver unarchiveObjectWithData:[defaults objectForKey:SRSourceTagColor]];
        if (!tagColor) {
            tagColor=[NSColor colorWithDeviceRed:0.1 green:0.1 blue:0.648 alpha:1.0];
		}
        NSColor* cmtColor=[NSUnarchiver unarchiveObjectWithData:[defaults objectForKey:SRSourceCommentColor]];
        if (!cmtColor) {
            cmtColor=[NSColor colorWithDeviceRed:0.632 green:0 blue:0 alpha:1.0];
		}
        NSColor* bakColor=[NSUnarchiver unarchiveObjectWithData:[defaults objectForKey:SRSourceBackgroundColor]];
        if (!bakColor) {
            bakColor=[NSColor whiteColor];
		}
        NSString* fontName=[defaults stringForKey:SRSourceFontName];
        if (!fontName) {
            fontName = @"Monaco";
        }
        int fontSize=[defaults integerForKey:SRSourceFontSize];
        if (fontSize < 1) {
            fontSize = 12;
        }
        NSFont * srcFont=[NSFont fontWithName:fontName size:fontSize];
        
		NSMutableAttributedString *attrString=[[[NSMutableAttributedString alloc]
		initWithString:source]autorelease];
		NSScanner*	scanner=[NSScanner scannerWithString:source];

//NS_DURING
		[attrString addAttribute:NSForegroundColorAttributeName
					 value:txtColor range:NSMakeRange(0,[attrString length])];
		
		while (![scanner isAtEnd]) {
			int tagStart, tagEnd;
			//Find next < character
			[scanner scanUpToString: @"<" intoString:nil];
			tagStart=[scanner scanLocation];		
			//<!--comment-->
			if([source length] > tagStart+6 &&
			[[source substringWithRange:NSMakeRange(tagStart+1,3)]isEqualToString:@"!--"]){
				if([scanner scanUpToString:@"-->" intoString:nil]){
					tagEnd=[scanner scanLocation]+3;
					if([source length]>tagEnd)
						[attrString addAttribute:NSForegroundColorAttributeName
						 value:cmtColor range:NSMakeRange(tagStart,tagEnd-tagStart)];
				}
			}else if([source length] > tagStart){
				if([scanner scanUpToString:@">" intoString:nil]){
					tagEnd=[scanner scanLocation]+1;
					//<tag>
					[attrString addAttribute:NSForegroundColorAttributeName
						 value:tagColor range:NSMakeRange(tagStart,tagEnd-tagStart)];
				}
			}
		}

		[attrString addAttribute:NSFontAttributeName
					 value:srcFont range:NSMakeRange(0,[attrString length])];

		[_sourceTextView setBackgroundColor:bakColor];
		[[_sourceTextView textStorage]setAttributedString:attrString];
		return;
/*NS_HANDLER
	[_sourceTextView setString:source];	
	return;
NS_ENDHANDLER*/
	}else{
		[_sourceTextView setString:source];	
	}
}

#pragma mark -
//--------------------------------------------------------------//
// action
//--------------------------------------------------------------//
- (IBAction)disclosureBtnAction:(id)sender
{
    [self _doDisclosure:[sender state]];
}

- (void)_doDisclosure:(BOOL)state
{
	NSArray* subviews = [_splitView subviews];
    if([subviews count]<2) return;
    
    NSView* topView=[subviews objectAtIndex:0];
    if(state){
        NSSize aSize=[topView frame].size;
        if(aSize.height==0){
            NSView* bottomView=[subviews objectAtIndex:1];
            NSSize  bSize=[bottomView frame].size;
            float   splitHeight=[_splitView frame].size.height;

            aSize.height=[_splitView savedFirstViewHeight];
            [topView setFrameSize:aSize];
            
            bSize.height=splitHeight-aSize.height-16;
            [bottomView setFrameSize:bSize];
            
        }
        
        // Show popup
        [_headerKindPopup setHidden:NO];
    }else{
        NSSize aSize=[topView frame].size;
        [_splitView setSavedFirstViewHeight:aSize.height];
        aSize.height=0;
        [topView setFrameSize:aSize];
        
        // Hide popup
        [_headerKindPopup setHidden:YES];
    }
	[_splitView adjustSubviews];
}

- (void)selectHeaderKindAction:(id)sender
{
    [_HTTPHeadersTable reloadData];
}

#pragma mark -
//--------------------------------------------------------------//
// NSSplitView Delegate
//--------------------------------------------------------------//

- (void)splitViewDidResizeSubviews:(NSNotification *)aNotification
{
    if([_splitView firstViewHeight]==0){
        [_disclosureBtn setState:NSOffState];
        [_headerKindPopup setHidden:YES];
    }else{
        [_disclosureBtn setState:NSOnState];
        [_headerKindPopup setHidden:NO];
    }

}

#if 0
- (float)splitView:(NSSplitView *)sender constrainMinCoordinate:(float)proposedMin ofSubviewAt:(int)offset
{
    if(offset==0 && proposedMin<32)   return 32;
    else    return proposedMin;
}
#endif

#pragma mark -
//--------------------------------------------------------------//
// NSTableDataSource
//--------------------------------------------------------------//

- (NSDictionary*)_headerFields
{
    // Get selected header kind
    int kind;
    kind = [[_headerKindPopup selectedItem] tag];
    
    switch (kind) {
    case _SRHTTPHeaderInitialRequestTag: {
        return [[_dataSource initialRequest] allHTTPHeaderFields];
    }
    case _SRHTTPHeaderFinalRequestTag:
    case _SRHTTPHeaderRequestTag: {
        return [[_dataSource request] allHTTPHeaderFields];
    }
    case _SRHTTPHeaderResposneTag: {
        return [(NSHTTPURLResponse*)[_dataSource response] allHeaderFields];
    }
    }
    
    return  nil;
}

- (int)numberOfRowsInTableView:(NSTableView*)tableView
{
    if (tableView == _HTTPHeadersTable) {
        // Get header fields
        NSDictionary*   headerFields;
        headerFields = [self _headerFields];
        
        return [headerFields count];
    }
    
    return 0;
}

- (id)tableView:(NSTableView*)tableView 
        objectValueForTableColumn:(NSTableColumn*)tableColumn 
        row:(int)rowIndex
{
    // Get identifier
    id  identifier;
    identifier = [tableColumn identifier];
    
    if (tableView == _HTTPHeadersTable) {
        // Get header fields
        NSDictionary*   headerFields;
        headerFields = [self _headerFields];
        
        // For key
        if ([identifier isEqualToString:@"key"]) {
            return [[headerFields allKeys] objectAtIndex:rowIndex];
        }
        
        // For value
        if ([identifier isEqualToString:@"value"]) {
            NSString*   key;
            key = [[headerFields allKeys] objectAtIndex:rowIndex];
            return [headerFields objectForKey:key];
        }
    }
    
    return nil;
}

#pragma mark -
//--------------------------------------------------------------//
// NSWindow delegate
//--------------------------------------------------------------//

- (void)windowDidBecomeKey:(NSNotification*)notification
{
    // Update file menu
    [[[[NSApp mainMenu] itemWithTag:SRFileTag] submenu] update];
}

- (void)windowWillClose:(NSNotification*)notification
{
    [[SRSourceWindowController sourceWindows]removeObject:self];
}

#pragma mark -
//--------------------------------------------------------------//
// Key value observation
//--------------------------------------------------------------//

- (void)observeValueForKeyPath:(NSString*)keyPath 
        ofObject:(id)object 
        change:(NSDictionary*)change 
        context:(void *)context
{
    // Bookmark preferences
    if ([keyPath isEqualToString:SRSourceFontName] || 
        [keyPath isEqualToString:SRSourceFontSize] || 
        [keyPath isEqualToString:SRSourceDefaultColor] || 
        [keyPath isEqualToString:SRSourceBackgroundColor] || 
        [keyPath isEqualToString:SRSourceTagColor] || 
        [keyPath isEqualToString:SRSourceCommentColor])
    {
        // Update text
        id<WebDocumentRepresentation>   representation;
        NSString*                       documentSource;
        representation = [_dataSource representation];
        documentSource = [representation documentSource];
        if (!documentSource) {
            return;
        }
        
        [self setColorizedSourceWithString:documentSource];
        
        return;
    }
}

@end

#pragma mark -

@implementation SRSourceSplitView

- (float)savedFirstViewHeight
{
    if(_savedFirstViewHeight<=0){
        _savedFirstViewHeight=100;  //rough
    }
    return _savedFirstViewHeight;
    
}
- (void)setSavedFirstViewHeight:(float)value
{
     _savedFirstViewHeight=value;
}

- (float)dividerThickness
{
    if([self firstViewHeight]==0){
        return 0;
    }
    return [super dividerThickness];
}

- (float)firstViewHeight
{
	NSArray* subviews = [self subviews];
    if([subviews count]>0){
        NSView* topView=[subviews objectAtIndex: 0];
        NSSize aSize=[topView frame].size;
        return aSize.height;
    }
    return -1;
}

/*
- (void)drawRect:(NSRect)aRect
{

    if([self firstViewHeight]==0){
        float x=aRect.origin.x;
        float y=aRect.origin.y+aRect.size.height-0.5;
		NSColor*	lineColor;
		if([[self window]styleMask] & NSTexturedBackgroundWindowMask){
			lineColor=[NSColor darkGrayColor];
		}else{
			lineColor=[NSColor controlShadowColor];
		}

		[lineColor set];
		[NSBezierPath strokeLineFromPoint:NSMakePoint(x, y) toPoint:NSMakePoint(x+aRect.size.width, y)];
    }
    [super drawRect:aRect];
}
*/
@end

