﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CookComputing.XmlRpc;

namespace BTS.Trac
{
    /// <summary>
    /// チケットの添付ファイル
    /// </summary>
    public class TicketAttachment
    {
        private Ticket ticket = null;

        /// <summary>
        /// ファイル名
        /// </summary>
        public string Name
        {
            get;
            private set;
        }

        /// <summary>
        /// 詳細
        /// </summary>
        public string Description
        {
            get;
            private set;
        }

        /// <summary>
        /// ファイルサイズ
        /// </summary>
        public int Size
        {
            get;
            private set;
        }

        /// <summary>
        /// 作成日時
        /// </summary>
        public DateTime CreateTime
        {
            get;
            private set;
        }

        /// <summary>
        /// 登録者
        /// </summary>
        public string Register
        {
            get;
            private set;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="ticket">チケットインスタンス</param>
        public TicketAttachment( Ticket ticket )
        {
            this.ticket = ticket;

            Name = "";
            Description = "";
            Size = 0;
            CreateTime = DateTime.Now;
            Register = "";
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="ticket">チケットインスタンス</param>
        /// <param name="attachment">チケット一覧</param>
        public TicketAttachment( Ticket ticket, object[] attachment )
        {
            this.ticket = ticket;

            Name = (string)attachment[0];
            Description = (string)attachment[1];
            Size = (int)attachment[2];
            CreateTime = (DateTime)attachment[3];
            Register = (string)attachment[4];
        }

        /// <summary>
        /// データ列を取得する
        /// </summary>
        /// <returns>バイト列</returns>
        public byte[] Get()
        {
            return Get( Name );
        }

        /// <summary>
        /// 添付ファイルを削除する
        /// </summary>
        /// <returns>処理結果</returns>
        public bool Delete()
        {
            if ( ticket.ID == 0 ) {
                throw new Exception( "チケットが指定されていません" );
            }

            return Trac.Rpc.ticketDeleteAttachment( ticket.ID, Name );
        }

        /// <summary>
        /// 添付ファイルを取得する
        /// </summary>
        /// <param name="filename">添付ファイル名</param>
        /// <returns>バイト列</returns>
        public byte[] Get( string filename )
        {
            return Trac.Rpc.ticketGetAttachment( ticket.ID, filename );
        }

        /// <summary>
        /// 添付ファイルを設定する
        /// </summary>
        /// <param name="filename">添付ファイル名</param>
        /// <param name="description">詳細</param>
        /// <param name="data">バイト列</param>
        /// <param name="replace">上書き設定</param>
        public void Put( string filename, string description, byte[] data, bool replace )
        {
            if ( ticket.ID == 0 ) {
                throw new Exception( "チケットが指定されていません" );
            }

            Trac.Rpc.ticketPutAttachment( ticket.ID, filename, description, data, replace );
        }

        /// <summary>
        /// 添付ファイルを設定する
        /// </summary>
        /// <param name="filename">添付ファイル名</param>
        /// <param name="description">詳細</param>
        /// <param name="replace">上書き設定</param>
        public void Put( string filename, string description, bool replace )
        {
            using ( FileStream fin = new FileStream( filename, FileMode.Open ) ) {
                byte[] data = new byte[fin.Length];
                fin.Read( data, 0, data.Length );

                Put( filename, description, data, replace );
            }
        }

        /// <summary>
        /// 添付ファイルを設定する
        /// </summary>
        /// <param name="filename">添付ファイル名</param>
        /// <param name="description">詳細</param>
        public void Put( string filename, string description )
        {
            Put( filename, description, false );
        }
    }
}
