// <file>
//     <copyright see="prj:///doc/copyright.txt"/>
//     <license see="prj:///doc/license.txt"/>
//     <owner name="Matthew Ward" email="mrward@users.sourceforge.net"/>
//     <version>$Revision$</version>
// </file>

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.IO;

using ICSharpCode.SharpDevelop.Internal.Templates;
using ICSharpCode.SharpDevelop.Project;

namespace ICSharpCode.WixBinding
{
	public enum WixOutputType {
		[Description("${res:ICSharpCode.WixBinding.ProjectOptions.OutputType.Installer} (.msi)")]
		package,
		[Description("${res:ICSharpCode.WixBinding.ProjectOptions.OutputType.MergeModule} (.msm)")]
		module,
		[Description("${res:ICSharpCode.WixBinding.ProjectOptions.OutputType.WixLibrary} (.wixlib)")]
		library
	}
	
	public class WixProject : MSBuildProject, IWixPropertyValueProvider
	{
		public const string DefaultTargetsFile = @"$(WixMSBuildExtensionsPath)\wix.targets";
		public const string FileNameExtension = ".wixproj";
		
		delegate bool IsFileNameMatch(string fileName);
		
		public WixProject(string fileName, string projectName)
		{
			Name = projectName;
			Language = WixLanguageBinding.LanguageName;
			SetupProject(fileName);
			IdGuid = BaseConfiguration["ProjectGuid"];
		}
		
		public WixProject(ProjectCreateInformation info)
		{
			Language = WixLanguageBinding.LanguageName;
			Create(info);
		}
		
		public override void Start(bool withDebugging)
		{
			withDebugging = false;
			switch (StartAction) {
				case StartAction.Project:
					Start(InstallerFullPath, withDebugging);
					break;
				default:
					base.Start(withDebugging);
					break;
			}
		}
		
		/// <summary>
		/// Returns the filename extension based on the project's output type.
		/// </summary>
		public static string GetInstallerExtension(string outputType)
		{
			outputType = outputType.ToLowerInvariant();
			switch (outputType) {
				case "package":
					return ".msi";
				case "module":
					return ".msm";
				case "library":
					return ".wixlib";
 				default:
					return ".msi";
			}
		}
		
		/// <summary>
		/// Adds the ability to creates Wix Library and Wix Object project items.
		/// </summary>
		public override ProjectItem CreateProjectItem(string itemType)
		{
			switch (itemType) {
				case "WixLibrary":
					return new WixLibraryProjectItem(this);
				case "CompileExtension":
					return new WixCompilerExtensionProjectItem(this);
				case "LibExtension":
					return new WixLibraryExtensionProjectItem(this);
				case "LinkExtension":
					return new WixLinkerExtensionProjectItem(this);
				default:
					return ProjectItemFactory.CreateProjectItem(this, itemType);
			}
		}
		
		/// <summary>
		/// Gets the full path to the installer file that will be generated by
		/// the Wix compiler and linker.
		/// </summary>
		public string InstallerFullPath {
			get {
				string outputPath = GetProperty("OutputPath");
				string outputType = GetProperty("OutputType");
				string outputName = GetProperty("OutputName");
				string fileName = String.Concat(outputName, GetInstallerExtension(outputType));
				return Path.Combine(Path.Combine(Directory, outputPath), fileName);
			}
		}
		
		/// <summary>
		/// Adds a set of Wix libraries (.wixlib) to the project.
		/// </summary>
		public void AddWixLibraries(string[] files)
		{
			foreach (string fileName in files) {
				AddWixLibrary(fileName);
			}
		}
				
		/// <summary>
		/// Adds a Wix library (.wixlib) to the project.
		/// </summary>
		public void AddWixLibrary(string fileName)
		{
			WixLibraryProjectItem projectItem = new WixLibraryProjectItem(this);
			projectItem.FileName = fileName;
			ProjectService.AddProjectItem(this, projectItem);
		}
		
		/// <summary>
		/// Returns the file project items that are Wix documents based on 
		/// their filename.
		/// </summary>
		public ReadOnlyCollection<FileProjectItem> WixFiles {
			get {
				return GetMatchingFiles(WixDocument.IsWixFileName);
			}
		}
		
		/// <summary>
		/// Returns the file project items that are Wix source files (.wxs).
		/// </summary>
		public ReadOnlyCollection<FileProjectItem> WixSourceFiles {
			get {
				return GetMatchingFiles(WixDocument.IsWixSourceFileName);
			}
		}
		
		/// <summary>
		/// Returns the compiler extension project items.
		/// </summary>
		public ReadOnlyCollection<WixExtensionProjectItem> WixCompilerExtensions {
			get {
				return GetExtensions(typeof(WixCompilerExtensionProjectItem));
			}
		}
		
		/// <summary>
		/// Returns the linker extension project items.
		/// </summary>
		public ReadOnlyCollection<WixExtensionProjectItem> WixLinkerExtensions {
			get {
				return GetExtensions(typeof(WixLinkerExtensionProjectItem));
			}
		}
		
		/// <summary>
		/// Returns the library extension project items.
		/// </summary>
		public ReadOnlyCollection<WixExtensionProjectItem> WixLibraryExtensions {
			get {
				return GetExtensions(typeof(WixLibraryExtensionProjectItem));
			}
		}
		
		/// <summary>
		/// Gets a preprocessor variable value with the given name.
		/// </summary>
		/// <remarks>
		/// TODO: This can be configuration specific.
		/// </remarks>
		/// <param name="name">The preprocessor variable name.</param>
		/// <returns>An empty string if the name cannot be found.</returns>
		public string GetVariable(string name)
		{
			NameValuePairCollection nameValuePairs = new NameValuePairCollection(BaseConfiguration["DefineConstants"]);
			return WixPropertyParser.Parse(nameValuePairs.GetValue(name), this);
		}
		
		/// <summary>
		/// Gets the MSBuild Property value for the given name.
		/// </summary>
		public string GetValue(string name)
		{
			string propertyValue;
			if (MSBuildEngine.MSBuildProperties.TryGetValue(name, out propertyValue)) {
				return propertyValue;
			}
			return null;
		}
		
		/// <summary>
		/// Checks whether the specified file can be compiled by the
		/// Wix project.
		/// </summary>
		/// <returns>
		/// <see langword="true"/> if the file is a WiX source file (.wxs)
		/// or a WiX include file (.wxi).</returns>
		public override bool CanCompile(string fileName)
		{
			return WixDocument.IsWixFileName(fileName);
		}
		
		/// <summary>
		/// Creates a WixProject with the default settings in its MSBuild file.
		/// </summary>
		protected override void Create(ProjectCreateInformation information)
		{
			Name = information.ProjectName;
			IdGuid = String.Concat("{", Guid.NewGuid().ToString().ToUpperInvariant(), "}");
			BaseConfiguration["OutputName"] = Name;
			BaseConfiguration["OutputType"] = "package";
			string wixToolPath = @"$(SharpDevelopBinPath)\Tools\Wix";
			BaseConfiguration["WixToolPath"] = wixToolPath;
			BaseConfiguration.SetIsGuarded("WixToolPath", true);
			BaseConfiguration["ToolPath"] = "$(WixToolPath)";
			BaseConfiguration.SetIsGuarded("ToolPath", true);
			BaseConfiguration["WixMSBuildExtensionsPath"] = wixToolPath;
			BaseConfiguration["Configuration"] = "Debug";
			BaseConfiguration.SetIsGuarded("Configuration", true);
			
			configurations["Debug|*"] = new PropertyGroup();
			configurations["Debug|*"]["BaseOutputPath"] = @"obj\";
			configurations["Debug|*"]["IntermediateOutputPath"] = @"obj\Debug\";
			configurations["Debug|*"]["OutputPath"] = @"bin\Debug\";
			
			configurations["Release|*"] = new PropertyGroup();
			configurations["Release|*"]["BaseOutputPath"] = @"obj\";
			configurations["Release|*"]["IntermediateOutputPath"] = @"obj\Release\";
			configurations["Release|*"]["OutputPath"] = @"bin\Release\";

			FileName = Path.GetFullPath(information.OutputProjectFileName);
			this.Imports.Add(new MSBuildImport(DefaultTargetsFile));
		}
		
		/// <summary>
		/// Returns a collection of FileProjectItems that match using the
		/// IsFileNameMatch delegate.
		/// </summary>
		ReadOnlyCollection<FileProjectItem> GetMatchingFiles(IsFileNameMatch match)
		{
			List<FileProjectItem> items = new List<FileProjectItem>();
			foreach (ProjectItem projectItem in Items) {
				FileProjectItem fileProjectItem = projectItem as FileProjectItem;
				if (fileProjectItem != null) {
					if (match(fileProjectItem.FileName)) {
						items.Add(fileProjectItem);
					}
				}
			}
			return new ReadOnlyCollection<FileProjectItem>(items);
		}
		
		/// <summary>
		/// Returns a collection of compiler extension items that match the specified
		/// type.
		/// </summary>
		ReadOnlyCollection<WixExtensionProjectItem> GetExtensions(Type type)
		{
			List<WixExtensionProjectItem> items = new List<WixExtensionProjectItem>();
			foreach (ProjectItem projectItem in Items) {
				WixExtensionProjectItem item = projectItem as WixExtensionProjectItem;
				if (item != null) {
					if (item.GetType() == type) {
						items.Add(item);
					}
				}
			}
			return new ReadOnlyCollection<WixExtensionProjectItem>(items);
		}
	}
}
