
/*
 * Daemon which updates the statistics database with various recorded parameters.
 *
 * Copyright (C) 2001-2005 Voluna Software. All rights reserved.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <libpq-fe.h>


PGconn *conn; /* This holds the connection handle */

void inthandler(int signum)
{
	PGresult *result;
	
	/* Ignore this signal if it occurs again while we're processing it */
	signal(signum, SIG_IGN);
	
	/* Commit work and disconnect from database */

	result = PQexec(conn, "COMMIT");

	if (PQresultStatus(result) != PGRES_COMMAND_OK)
	{
		fprintf(stderr, "COMMIT command failed! Disconnecting anyway.\n");
		PQclear(result);
		PQfinish(conn);
	}
	
	PQclear(result);
	
	PQfinish(conn);
	
	exit(0);
}

long long get_cpu_usage(long long *cpu_free, long long *cpu_total)
{
	FILE *stat;
	int percent;
	char buf[8192]; /* A line will DEFINITELY NOT BE longer than this */
	char dummy[8192];
	long long a1, a2, a3, a4;
	long long temp1, temp2;
	
	if ((stat = fopen("/proc/stat", "rb")) == NULL)
	{
		fprintf(stderr, "Unable to open file /proc/stat\n");
		return -1;
	}
	
	percent = a1 = a2 = a3 = a4 = 0;
	
	while (fgets(buf, 8192, stat))
	{
		if (strncasecmp(buf, "cpu ", 4) == 0)
		{
			sscanf(buf, "%s %Ld %Ld %Ld %Ld", dummy, &a1, &a2, &a3, &a4);
			break;
		}
	}
	
	fclose(stat);
	
	temp1 = (a1 + a2 + a3 + a4) - (*cpu_total);
	temp2 = a4 - (*cpu_free);
	
	if (temp1 <= 0)
		temp1 = 1;
	
	percent = (int) (((temp1 - temp2) / (temp1 * 1.0)) * 100);
	
	*cpu_total = a1 + a2 + a3 + a4;
	*cpu_free = a4;
	
	return percent;
}

long long get_memory_usage(void)
{
	FILE *stat;
	int percent;
	char buf[8192]; /* A line will DEFINITELY NOT BE longer than this */
	char dummy[8192];
	long long a1, a2, a3, a4, a5, a6;
	
	if ((stat = fopen("/proc/meminfo", "rb")) == NULL)
	{
		fprintf(stderr, "Unable to open file /proc/meminfo\n");
		return -1;
	}
	
	percent = a1 = a2 = a3 = a4 = a5 = a6 = 0;
	
	while (fgets(buf, 8192, stat))
	{
		if (strncasecmp(buf, "Mem: ", 5) == 0)
		{
			sscanf(buf, "%s %Ld %Ld %Ld %Ld %Ld %Ld", dummy, &a1, &a2, &a3, &a4, &a5, &a6);
			break;
		}
	}
	
	fclose(stat);
	
	if (a1 <= 0)
		a1 = 1;
	
	percent = (int) (((a2 - (a5 + a6)) / (a1 * 1.0)) * 100);
	
	return percent;
}

long long get_load_usage(long long *process_count)
{
	FILE *stat;
	char buf[8192]; /* A line will DEFINITELY NOT BE longer than this */
	char dummy[8192];
	long long a1;
	long long temp;
	
	if ((stat = fopen("/proc/stat", "rb")) == NULL)
	{
		fprintf(stderr, "Unable to open file /proc/stat\n");
		return -1;
	}
	
	temp = a1 = 0;
	
	while (fgets(buf, 8192, stat))
	{
		if (strncasecmp(buf, "processes ", 10) == 0)
		{
			sscanf(buf, "%s %Ld", dummy, &a1);
			break;
		}
	}
	
	fclose(stat);
	
	temp = a1 - (*process_count);
	
	if (temp < 0)
		temp = 0;
	
	*process_count = a1;
	
	return temp;
}

long long get_net_usage(long long *net_count)
{
	FILE *stat;

	char buf[8192]; /* A line will DEFINITELY NOT BE longer than this */
	char *bufptr;
	long long a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, a11, a12, a13, a14, a15, a16;
	long long temp1, temp2;
	int i;
	
	if ((stat = fopen("/proc/net/dev", "rb")) == NULL)
	{
		fprintf(stderr, "Unable to open file /proc/net/dev\n");
		return -1;
	}
	
	temp1 = temp2 = a1 = a2 = a3 = a4 = a5 = a6 = a7 = a8 = a9 = a10 = a11 = a12 = a13 = a14 = a15 = a16 = 0;
	
	/* Skip the first two lines */
	
	fgets(buf, 8192, stat);
	fgets(buf, 8192, stat);
	
	while (fgets(buf, 8192, stat))
	{
		i = 0;
		while (i < strlen(buf))
		{
			if (buf[i] == ':')
				break;
			i++;
		}
		
		if ((buf[i] == ':') && ((i - 3) >= 0))
		{
			if (strncmp(buf + i - 3, " lo:", 4) == 0)
				continue;
		}
		else
			continue;
	
		bufptr = buf + i + 1;
	
		sscanf(bufptr, "%Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld %Ld",
			&a1, &a2, &a3, &a4, &a5, &a6, &a7, &a8, &a9, &a10, &a11, &a12, &a13, &a14, &a15, &a16);
		
		temp2 += a1 + a9;
	}
	
	fclose(stat);
	
	temp1 = temp2 - (*net_count);
	
	if (temp1 < 0)
		temp1 = 0;
	
	*net_count = temp2;
	
	return ((int) (temp1 / 1024));
}

long long get_disk_usage(long long *disk_count)
{
	FILE *stat;
	char buf[16384]; /* A line will DEFINITELY NOT BE longer than this */
	char *bufptr, *tbuf;
	long long a1, a2, a3, a4, a5;
	long long temp1, temp2;
	int i, len;
	
	if ((stat = fopen("/proc/stat", "rb")) == NULL)
	{
		fprintf(stderr, "Unable to open file /proc/stat\n");
		return -1;
	}
	
	temp1 = temp2 = a1 = a2 = a3 = a4 = a5 = 0;
	
	while (fgets(buf, 16384, stat))
		if (strncasecmp(buf, "disk_io: ", 9) == 0)
		{
			bufptr = buf + 9;

			for (;;)
			{
				len = strlen(bufptr);
				i = 0;
				
				while (i < len)
					if (bufptr[i++] == ':')
						break;
				if (i >= len)
					break;
				
				if (bufptr[i] != '(')
				{
					fprintf(stderr, "Severe programming error in get_disk_usage()!\n");
					return -1;
				}
				
				tbuf = bufptr + i + 1;

				while (i < len)
				{
					if (bufptr[i] == ',')
						bufptr[i] = ' ';
						
					if (bufptr[i++] == ')')
						break;
				}
				
				if (i >= len)
				{
					fprintf(stderr, "Severe programming error in get_disk_usage()!\n");
					return -1;
				}
				
				bufptr[i - 1] = 0;
				bufptr += i;
				
				sscanf(tbuf, "%Ld %Ld %Ld %Ld %Ld", &a1, &a2, &a3, &a4, &a5);
				temp1 += a3 + a5;
			}
		
		}
	
	fclose(stat);
	
	temp2 = temp1 - (*disk_count);
	
	if (temp2 <= 0)
		temp2 = 0;
	
	*disk_count = temp1;
	
	return temp2;
}

int main(int argc, char *argv[])
{
	PGresult *result;
	time_t t;
	char query[8192]; /* A query will DEFINITELY NOT BE longer than this */

	unsigned int key;
	long long cpu_usage, memory_usage, load_usage, net_usage, disk_usage;
	long long cpu_free = 0, cpu_total = 0, process_count = 0, net_count = 0, disk_count = 0;
	int initialized = 0;

	/* First connect, and hold that connection */
	
	conn = PQsetdbLogin(NULL, "5432", NULL, NULL, "openfiler", "postgres", "");
	
	if (PQstatus(conn) == CONNECTION_BAD)
	{
		fprintf(stderr, "Connection to database %s failed.\n", "postgres");
		fprintf(stderr, "%s", PQerrorMessage(conn));
		return -1;
	}
	

	/* Install a signal handler */
	
	signal(SIGINT, inthandler);
	signal(SIGTERM, inthandler);
	signal(SIGKILL, inthandler);
	
	/* Caliberate to a 15 second interval */
	
	while ((t = time(NULL)) % 15);
	
	/* Now, sit in a loop forever, until stopped by
	   a SIGINT */
	
	for(;;)
	{
		/* Make *absolutely* sure we're at a 15 sec boundary */
		/* The sleep() also blocks and doesn't waste CPU */
		
		sleep(15 - ((t + 15) % 15));
		t = time(NULL);
		key = (t / 15) % 345600;

		/* Gather statistics  and record them */
		
		if (initialized == 0)
		{
			/* have to do this to initialize counters; nevermind about return value/errors */
			
			get_cpu_usage(&cpu_free, &cpu_total);
			get_load_usage(&process_count);
			get_net_usage(&net_count);
			get_disk_usage(&disk_count);
			initialized = 1;
		}

		if ((cpu_usage = get_cpu_usage(&cpu_free, &cpu_total)) < 0)
		{
			fprintf(stderr, "Error getting CPU usage information!\n");
			return -1;
		}
		
		if ((memory_usage = get_memory_usage()) < 0)
		{
			fprintf(stderr, "Error getting memory usage information!\n");
			return -1;
		}
		
		if ((load_usage = get_load_usage(&process_count)) < 0)
		{
			fprintf(stderr, "Error getting process usage information!\n");
			return -1;
		}
		
		if ((net_usage = get_net_usage(&net_count)) < 0)
		{
			fprintf(stderr, "Error getting net usage information!\n");
			return -1;
		}
		
		if ((disk_usage = get_disk_usage(&disk_count)) < 0)
		{
			fprintf(stderr, "Error getting disk usage information!\n");
			return -1;
		}
		
		result = PQexec(conn, "BEGIN");
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "BEGIN command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}
		
		PQclear(result);

		snprintf(query, 8192, "UPDATE stats_cpu SET value=%Ld WHERE time=%d;", cpu_usage, key);

		result = PQexec(conn, query);
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "UPDATE command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		snprintf(query, 8192, "UPDATE stats_memory SET value=%Ld WHERE time=%d;", memory_usage, key);

		result = PQexec(conn, query);
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "UPDATE command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		snprintf(query, 8192, "UPDATE stats_load SET value=%Ld WHERE time=%d;", load_usage, key);

		result = PQexec(conn, query);
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "UPDATE command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		snprintf(query, 8192, "UPDATE stats_net SET value=%Ld WHERE time=%d;", net_usage, key);

		result = PQexec(conn, query);
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "UPDATE command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		snprintf(query, 8192, "UPDATE stats_disk SET value=%Ld WHERE time=%d;", disk_usage, key);

		result = PQexec(conn, query);
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "UPDATE command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		result = PQexec(conn, "END");
		if (PQresultStatus(result) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "END command failed at %s", ctime(&t));
			fprintf(stderr, "%s", PQresultErrorMessage(result));
			PQclear(result);
			PQfinish(conn);
			return -1;
		}

		PQclear(result);

		/* Need to get time() again */
		t = time(NULL);
	}

	/* This should never be reached */
	
	return 0;
}
