<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

declare(strict_types=1);

namespace core_reportbuilder\local\filters;

use advanced_testcase;
use lang_string;
use core_reportbuilder\local\report\filter;

/**
 * Unit tests for date report filter
 *
 * @package     core_reportbuilder
 * @covers      \core_reportbuilder\local\filters\base
 * @covers      \core_reportbuilder\local\filters\date
 * @copyright   2021 Paul Holden <paulh@moodle.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class date_testcase extends advanced_testcase {

    /**
     * Data provider for {@see test_get_sql_filter_simple}
     *
     * @return array
     */
    public function get_sql_filter_simple_provider(): array {
        return [
            [date::DATE_ANY, true],
            [date::DATE_NOT_EMPTY, true],
            [date::DATE_EMPTY, false],
        ];
    }

    /**
     * Test getting filter SQL
     *
     * @param int $operator
     * @param bool $expectuser
     *
     * @dataProvider get_sql_filter_simple_provider
     */
    public function test_get_sql_filter_simple(int $operator, bool $expectuser): void {
        global $DB;

        $this->resetAfterTest();

        $user = $this->getDataGenerator()->create_user([
            'timecreated' => 12345,
        ]);

        $filter = new filter(
            date::class,
            'test',
            new lang_string('yes'),
            'testentity',
            'timecreated'
        );

        // Create instance of our filter, passing given operator.
        [$select, $params] = date::create($filter)->get_sql_filter([
            $filter->get_unique_identifier() . '_operator' => $operator,
        ]);

        $usernames = $DB->get_fieldset_select('user', 'username', $select, $params);
        if ($expectuser) {
            $this->assertContains($user->username, $usernames);
        } else {
            $this->assertNotContains($user->username, $usernames);
        }
    }

    /**
     * Test getting filter SQL while specifying a date range
     */
    public function test_get_sql_filter_date_range(): void {
        global $DB;

        $this->resetAfterTest();

        $userone = $this->getDataGenerator()->create_user(['timecreated' => 50]);
        $usertwo = $this->getDataGenerator()->create_user(['timecreated' => 100]);

        $filter = new filter(
            date::class,
            'test',
            new lang_string('yes'),
            'testentity',
            'timecreated'
        );

        // Create instance of our date range filter.
        [$select, $params] = date::create($filter)->get_sql_filter([
            $filter->get_unique_identifier() . '_operator' => date::DATE_RANGE,
            $filter->get_unique_identifier() . '_from' => 80,
            $filter->get_unique_identifier() . '_to' => 120,
        ]);

        // The only matching user should be our first test user.
        $usernames = $DB->get_fieldset_select('user', 'username', $select, $params);
        $this->assertEquals([$usertwo->username], $usernames);
    }
}
