/*
// $Id: //open/mondrian/src/main/mondrian/rolap/agg/Segment.java#46 $
// This software is subject to the terms of the Common Public License
// Agreement, available at the following URL:
// http://www.opensource.org/licenses/cpl.html.
// Copyright (C) 2002-2002 Kana Software, Inc.
// Copyright (C) 2002-2007 Julian Hyde and others
// All Rights Reserved.
// You must accept the terms of that agreement to use this software.
//
// jhyde, 21 March, 2002
*/
package mondrian.rolap.agg;

import mondrian.olap.*;
import mondrian.rolap.*;

import java.sql.*;
import java.util.*;
import java.io.PrintWriter;

/**
 * A <code>Segment</code> is a collection of cell values parameterized by
 * a measure, and a set of (column, value) pairs. An example of a segment is</p>
 *
 * <blockquote>
 *   <p>(Unit sales, Gender = 'F', State in {'CA','OR'}, Marital Status = <i>
 *   anything</i>)</p>
 * </blockquote>
 *
 * <p>All segments over the same set of columns belong to an Aggregation, in this
 * case</p>
 *
 * <blockquote>
 *   <p>('Sales' Star, Gender, State, Marital Status)</p>
 * </blockquote>
 *
 * <p>Note that different measures (in the same Star) occupy the same Aggregation.
 * Aggregations belong to the AggregationManager, a singleton.</p>
 * <p>Segments are pinned during the evaluation of a single MDX query. The query
 * evaluates the expressions twice. The first pass, it finds which cell values it
 * needs, pins the segments containing the ones which are already present (one
 * pin-count for each cell value used), and builds a {@link CellRequest
 * cell request} for those which are not present. It executes
 * the cell request to bring the required cell values into the cache, again,
 * pinned. Then it evalutes the query a second time, knowing that all cell values
 * are available. Finally, it releases the pins.</p>
 *
 * <p>A Segment may have a list of excluded {@link Region} objects. These are
 * caused by cache flushing. Usually a segment is a hypercube: it is defined by
 * a set of values on each of its axes. But after a cache flush request, a
 * segment may have a rectangular 'hole', and therefore not be a hypercube
 * anymore.
 *
 * <p>For example, the segment defined by {CA, OR, WA} * {F, M} is a
 * 2-dimensional hyper-rectangle with 6 cells. After flushing {CA, OR, TX} *
 * {F}, the result is 4 cells:
 *
 * <pre>
 *     F     M
 * CA  out   in
 * OR  out   in
 * WA  in    in
 * </pre>
 *
 * defined by the original segment minus the region ({CA, OR} * {F}).
 *
 * @author jhyde
 * @since 21 March, 2002
 * @version $Id: //open/mondrian/src/main/mondrian/rolap/agg/Segment.java#46 $
 */
class Segment {
    private static int nextId = 0; // generator for "id"

    final int id; // for debug
    private String desc;
    final Aggregation aggregation;
    final RolapStar.Measure measure;

    final Aggregation.Axis[] axes;
    private SegmentDataset data;
    private final CellKey cellKey; // workspace

    /** State of the segment (loading, ready, etc.). */
    private State state;

    /**
     * List of regions to ignore when reading this segment. This list is
     * populated when a region is flushed. The cells for these regions may be
     * physically in the segment, because trimming segments can be expensive,
     * but should still be ignored.
     */
    private final List<Region> excludedRegions;

    /**
     * Creates a <code>Segment</code>; it's not loaded yet.
     *
     * @param aggregation The aggregation this <code>Segment</code> belongs to
     * @param measure Measure whose values this Segment contains
     * @param axes List of axes; each is a constraint plus a list of values
     * @param excludedRegions List of regions which are not in this segment.
     */
    Segment(
        Aggregation aggregation,
        RolapStar.Measure measure,
        Aggregation.Axis[] axes,
        List<Region> excludedRegions)
    {
        this.id = nextId++;
        this.aggregation = aggregation;
        this.measure = measure;
        this.axes = axes;
        this.cellKey = CellKey.Generator.newCellKey(axes.length);
        this.state = State.Loading;
        this.excludedRegions = excludedRegions;
        for (Region region : excludedRegions) {
            assert region.getPredicates().size() == axes.length;
        }
    }

    /**
     * Sets the data, and notifies any threads which are blocked in
     * {@link #waitUntilLoaded}.
     */
    synchronized void setData(
            SegmentDataset data,
            RolapAggregationManager.PinSet pinnedSegments) {
        Util.assertTrue(this.data == null);
        Util.assertTrue(this.state == State.Loading);

        this.data = data;
        this.state = State.Ready;
        notifyAll();
    }

    /**
     * If this segment is still loading, signals that it failed to load, and
     * notifies any threads which are blocked in {@link #waitUntilLoaded}.
     */
    synchronized void setFailed() {
        switch (state) {
        case Loading:
            Util.assertTrue(this.data == null);
            this.state = State.Failed;
            notifyAll();
            break;
        case Ready:
            // The segment loaded just fine.
            break;
        default:
            throw Util.badValue(state);
        }
    }

    public boolean isReady() {
        return (state == State.Ready);
    }

    private void makeDescription(StringBuilder buf, boolean values) {
        final String sep = Util.nl + "    ";
        buf.append("Segment #");
        buf.append(id);
        buf.append(" {");
        buf.append(sep);
        buf.append("measure=");
        buf.append(measure.getAggregator().getExpression(
                        measure.getExpression().getGenericExpression()));

        RolapStar.Column[] columns = aggregation.getColumns();
        for (int i = 0; i < columns.length; i++) {
            buf.append(sep);
            buf.append(columns[i].getExpression().getGenericExpression());
            final Aggregation.Axis axis = axes[i];
            axis.getPredicate().describe(buf);
            if (values) {
                Object[] keys = axis.getKeys();
                buf.append(", values={");
                for (int j = 0; j < keys.length; j++) {
                    if (j > 0) {
                        buf.append(", ");
                    }
                    Object key = keys[j];
                    buf.append(key);
                }
                buf.append("}");
            }
        }
        if (!excludedRegions.isEmpty()) {
            buf.append(sep);
            buf.append("excluded={");
            int k = 0;
            for (Region excludedRegion : excludedRegions) {
                if (k++ > 0) {
                    buf.append(", ");
                }
                excludedRegion.describe(buf);
            }
            buf.append('}');
        }
        buf.append('}');
    }

    public String toString() {
        if (this.desc == null) {
            StringBuilder buf = new StringBuilder(64);
            makeDescription(buf, false);
            this.desc = buf.toString();
        }
        return this.desc;
    }

    /**
     * Retrieves the value at the location identified by
     * <code>keys</code>.
     *
     * <p>Returns<ul>
     *
     * <li>{@link Util#nullValue} if the cell value
     * is null (because no fact table rows met those criteria);</li>
     *
     * <li><code>null</code> if the value is not supposed to be in this segment
     * (because one or more of the keys do not pass the axis criteria);</li>
     *
     * <li>the data value otherwise</li>
     *
     * </ul></p>
     *
     * <p>Note: Must be called from a synchronized context, because uses the
     * <code>cellKey[]</code> as workspace.</p>
     */
    Object getCellValue(Object[] keys) {
        assert keys.length == axes.length;
        int missed = 0;
        for (int i = 0; i < keys.length; i++) {
            Object key = keys[i];
            int offset = axes[i].getOffset(key);
            if (offset < 0) {
                if (axes[i].getPredicate().evaluate(key)) {
                    // see whether this segment should contain this value
                    missed++;
                    continue;
                } else {
                    // this value should not appear in this segment; we
                    // should be looking in a different segment
                    return null;
                }
            }
            cellKey.setAxis(i, offset);
        }
        if (isExcluded(keys)) {
            // this value should not appear in this segment; we
            // should be looking in a different segment
            return null;
        }
        if (missed > 0) {
            // the value should be in this segment, but isn't, because one
            // or more of its keys does have any values
            return Util.nullValue;
        } else {
            Object o = data.get(cellKey);
            if (o == null) {
                o = Util.nullValue;
            }
            return o;
        }
    }

    /**
     * Returns whether the given set of key values will be in this segment
     * when it finishes loading.
     */
    boolean wouldContain(Object[] keys) {
        Util.assertTrue(keys.length == axes.length);
        for (int i = 0; i < keys.length; i++) {
            Object key = keys[i];
            if (!axes[i].getPredicate().evaluate(key)) {
                return false;
            }
        }
        return !isExcluded(keys);
    }

    /**
     * Returns whether a cell value is excluded from this segment.
     */
    private boolean isExcluded(Object[] keys) {
        for (Region excludedRegion : excludedRegions) {
            if (excludedRegion.wouldContain(keys)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Reads a segment of <code>measure</code>, where <code>columns</code> are
     * constrained to <code>values</code>.  Each entry in <code>values</code>
     * can be null, meaning don't constrain, or can have several values. For
     * example, <code>getSegment({Unit_sales}, {Region, State, Year}, {"West"},
     * {"CA", "OR", "WA"}, null})</code> returns sales in states CA, OR and WA
     * in the Western region, for all years.
     *
     * @pre segments[i].aggregation == aggregation
     */
    static void load(
        final Segment[] segments,
        final BitKey levelBitKey,
        final BitKey measureBitKey,
        final RolapAggregationManager.PinSet pinnedSegments,
        final Aggregation.Axis[] axes)
    {
        String sql = AggregationManager.instance().generateSql(
                segments,
                levelBitKey,
                measureBitKey);
        Segment segment0 = segments[0];
        RolapStar star = segment0.aggregation.getStar();
        RolapStar.Column[] columns = segment0.aggregation.getColumns();
        int arity = columns.length;

        // Workspace to build up lists of distinct values for each axis.
        SortedSet<Comparable<?>>[] axisValueSets = new SortedSet[arity];
        boolean[] axisContainsNull = new boolean[arity];
        for (int i = 0; i < axisValueSets.length; i++) {
            if (Util.PreJdk15) {
                // Work around the fact that Boolean is not Comparable until JDK
                // 1.5.
                assert !(Comparable.class.isAssignableFrom(Boolean.class));
                axisValueSets[i] =
                    new TreeSet(
                        new Comparator() {
                            public int compare(Object o1, Object o2) {
                                if (o1 instanceof Boolean) {
                                    Boolean b1 = (Boolean) o1;
                                    if (o2 instanceof Boolean) {
                                        Boolean b2 = (Boolean) o2;
                                        return (b1 == b2 ? 0 : (b1 ? 1 : -1));
                                    } else {
                                        return -1;
                                    }
                                } else {
                                    return ((Comparable) o1).compareTo(o2);
                                }
                            }
                        }
                    );
            } else {
                assert Comparable.class.isAssignableFrom(Boolean.class);
                axisValueSets[i] = new TreeSet<Comparable<?>>();
            }
        }

        // execute
        SqlStatement stmt =
            RolapUtil.executeQuery(
                star.getDataSource(), sql, "Segment.load",
                "Error while loading segment");
        final int measureCount = segments.length;
        try {
            List<Object[]> rows = new ArrayList<Object[]>();
            ResultSet resultSet = stmt.getResultSet();
            while (resultSet.next()) {
                ++stmt.rowCount;
                Object[] row = new Object[arity + measureCount];
                // get the columns
                int k = 1;
                for (int i = 0; i < arity; i++) {
                    Object o = resultSet.getObject(k++);
                    if (o == null) {
                        o = RolapUtil.sqlNullValue;
                        axisContainsNull[i] = true;
                    } else {
                        axisValueSets[i].add(Aggregation.Axis.wrap(o));
                    }
                    row[i] = o;
                }
                // get the measure
                for (int i = 0; i < measureCount; i++) {
                    Object o = resultSet.getObject(k++);
                    if (o == null) {
                        o = Util.nullValue; // convert to placeholder
                    } else if (segments[i].measure.getDatatype().isNumeric()) {
                        if (o instanceof Double) {
                            // nothing to do
                        } else if (o instanceof Number) {
                            o = ((Number) o).doubleValue();
                        } else if (o instanceof byte[]) {
                            // On MySQL 5.0 in German locale, values can come
                            // out as byte arrays. Don't know why. Bug 1594119.
                            o = Double.parseDouble(new String((byte []) o));
                        } else {
                            o = Double.parseDouble(o.toString());
                        }
                    }
                    row[arity + i] = o;
                }
                rows.add(row);
            }

            // Figure out size of dense array, and allocate it, or use a sparse
            // array if appropriate.
            boolean sparse = false;
            int n = 1;
            for (int i = 0; i < arity; i++) {
                Aggregation.Axis axis = axes[i];
                SortedSet<Comparable<?>> valueSet = axisValueSets[i];
                int size = axis.loadKeys(valueSet, axisContainsNull[i]);

                int previous = n;
                n *= size;
                if ((n < previous) || (n < size)) {
                    // Overflow has occurred.
                    n = Integer.MAX_VALUE;
                    sparse = true;
                }
            }
            sparse = sparse || useSparse((double) n, (double) rows.size());

            final SegmentDataset[] dataSets;
            final SparseSegmentDataset[] sparseDatasets;
            final DenseSegmentDataset[] denseDatasets;
            if (sparse) {
                sparseDatasets = new SparseSegmentDataset[segments.length];
                dataSets = sparseDatasets;
                denseDatasets = null;
            } else {
                denseDatasets = new DenseSegmentDataset[segments.length];
                dataSets = denseDatasets;
                sparseDatasets = null;
            }

            if (sparse) {
                for (int i = 0; i < segments.length; i++) {
                    sparseDatasets[i] = new SparseSegmentDataset(segments[i]);
                }
            } else {
                for (int i = 0; i < segments.length; i++) {
                    denseDatasets[i] = new DenseSegmentDataset(
                            segments[i], new Object[n]);
                }
            }
            // now convert the rows into a sparse array
            int[] pos = new int[arity];
            for (int i = 0, count = rows.size(); i < count; i++) {
                Object[] row = rows.get(i);
                int k = 0;
                for (int j = 0; j < arity; j++) {
                    k *= axes[j].getKeys().length;
                    Object o = row[j];
                    Aggregation.Axis axis = axes[j];
                    int offset = axis.getOffset(o);
                    pos[j] = offset;
                    k += offset;
                }
                CellKey key;
                if (sparse) {
                    key = CellKey.Generator.newCellKey(pos);
                    for (int j = 0; j < segments.length; j++) {
                        final Object o = row[arity + j];
                        sparseDatasets[j].put(key, o);
                    }
                } else {
                    for (int j = 0; j < segments.length; j++) {
                        final Object o = row[arity + j];
                        denseDatasets[j].set(k, o);
                    }
                }
            }

            for (int i = 0; i < segments.length; i++) {
                segments[i].setData(dataSets[i], pinnedSegments);
            }

        } catch (SQLException e) {
            throw stmt.handle(e);
        } finally {
            stmt.close();
            // Any segments which are still loading have failed.
            for (Segment segment : segments) {
                segment.setFailed();
            }
        }
    }

    /**
     * Decides whether to use a sparse representation for this segment, using
     * the formula described
     * {@link MondrianProperties#SparseSegmentCountThreshold here}.
     *
     * @param possibleCount Number of values in the space.
     * @param actualCount Actual number of values.
     * @return Whether to use a sparse representation.
     */
    private static boolean useSparse(
            final double possibleCount,
            final double actualCount) {
        final MondrianProperties properties = MondrianProperties.instance();
        double densityThreshold = properties.SparseSegmentDensityThreshold.get();
        if (densityThreshold < 0) {
            densityThreshold = 0;
        }
        if (densityThreshold > 1) {
            densityThreshold = 1;
        }
        int countThreshold = properties.SparseSegmentCountThreshold.get();
        if (countThreshold < 0) {
            countThreshold = 0;
        }
        boolean sparse =
            (possibleCount - countThreshold) * densityThreshold > actualCount;
        if (possibleCount < countThreshold) {
            assert !sparse :
                    "Should never use sparse if count is less " +
                    "than threshold, possibleCount=" + possibleCount +
                    ", actualCount=" + actualCount +
                    ", countThreshold=" + countThreshold +
                    ", densityThreshold=" + densityThreshold;
        }
        if (possibleCount == actualCount) {
            assert !sparse :
                    "Should never use sparse if result is 100% dense: " +
                    "possibleCount=" + possibleCount +
                    ", actualCount=" + actualCount +
                    ", countThreshold=" + countThreshold +
                    ", densityThreshold=" + densityThreshold;
        }
        return sparse;
    }

    /**
     * Blocks until this segment has finished loading; if this segment has
     * already loaded, returns immediately.
     */
    public synchronized void waitUntilLoaded() {
        if (!isReady()) {
            try {
                wait();
            } catch (InterruptedException e) {
            }
            switch (state) {
            case Ready:
                return; // excellent!
            case Failed:
                throw Util.newError("Pending segment failed to load: "
                    + toString());
            default:
                throw Util.badValue(state);
            }
        }
    }

    /**
     * Prints the state of this <code>Segment</code>, including constraints
     * and values.
     *
     * @param pw Writer
     */
    public void print(PrintWriter pw) {
        final StringBuilder buf = new StringBuilder();
        makeDescription(buf, true);
        pw.print(buf.toString());
        pw.println();
    }

    public List<Region> getExcludedRegions() {
        return excludedRegions;
    }

    /**
     * Returns the number of cells in this Segment, deducting cells in
     * excluded regions.
     *
     * <p>This method may return a value which is slightly too low, or
     * occasionally even negative. This occurs when a Segment has more than one
     * excluded region, and those regions overlap. Cells which are in both
     * regions will be counted twice.
     *
     * @return Number of cells in this Segment
     */
    public int getCellCount() {
        int cellCount = 1;
        for (Aggregation.Axis axis : axes) {
            cellCount *= axis.getKeys().length;
        }
        for (Region excludedRegion : excludedRegions) {
            cellCount -= excludedRegion.cellCount;
        }
        return cellCount;
    }

    /**
     * Creates a Segment which has the same dimensionality as this Segment and a
     * subset of the values.
     *
     * <p>If <code>bestColumn</code> is not -1, the <code>bestColumn</code>th
     * column's predicate should be replaced by <code>bestPredicate</code>.
     *
     * @param axisKeepBitSets For each axis, a bitmap of the axis values to
     *   keep; each axis must have at least one bit set
     * @param bestColumn
     * @param bestPredicate
     * @param excludedRegions List of regions to exclude from segment
     * @return Segment containing a subset of the values
     */
    Segment createSubSegment(
        BitSet[] axisKeepBitSets,
        int bestColumn,
        StarColumnPredicate bestPredicate,
        List<Segment.Region> excludedRegions) {
        assert axisKeepBitSets.length == axes.length;

        // Create a new segment with a subset of the values. If only one
        // of the axes is restricted, restrict just that axis. If more than
        // one of the axis is restricted, add a negation to the segment.
        final Aggregation.Axis[] newAxes = axes.clone();

        // For each axis, map from old position to new position.
        final Map<Integer,Integer>[] axisPosMaps = new Map[axes.length];

        int valueCount = 1;
        for (int j = 0; j < axes.length; j++) {
            Aggregation.Axis axis = axes[j];
            StarColumnPredicate newPredicate = axis.getPredicate();
            if (j == bestColumn) {
                newPredicate = bestPredicate;
            }
            final Comparable<?>[] axisKeys = axis.getKeys();
            BitSet keepBitSet = axisKeepBitSets[j];
            int firstClearBit = keepBitSet.nextClearBit(0);
            Comparable<?>[] newAxisKeys;
            if (firstClearBit >= axisKeys.length) {
                // Keep everything
                newAxisKeys = axisKeys;
                axisPosMaps[j] = null; // identity map
            } else {
                List<Object> newAxisKeyList = new ArrayList<Object>();
                Map<Integer, Integer> map
                    = axisPosMaps[j]
                    = new HashMap<Integer, Integer>();
                for (int bit = keepBitSet.nextSetBit(0);
                    bit >= 0;
                    bit = keepBitSet.nextSetBit(bit + 1)) {
                    map.put(bit, newAxisKeyList.size());
                    newAxisKeyList.add(axisKeys[bit]);
                }
                newAxisKeys =
                    newAxisKeyList.toArray(
                        new Comparable<?>[newAxisKeyList.size()]);
                assert newAxisKeys.length > 0;
            }
            final Aggregation.Axis newAxis =
                new Aggregation.Axis(newPredicate, newAxisKeys);
            newAxes[j] = newAxis;
            valueCount *= newAxisKeys.length;
        }

        // Create a new segment.
        final Segment newSegment =
            new Segment(aggregation, measure, newAxes, excludedRegions);

        // Create a dataset containing a subset of the current dataset.
        // Keep the same representation as the current dataset.
        // (We could be smarter - sometimes a subset of a sparse dataset will
        // be dense and VERY occasionally a subset of a relatively dense dataset
        // will be sparse.)
        SegmentDataset newData;
        if (data instanceof SparseSegmentDataset) {
            newData =
                new SparseSegmentDataset(
                    newSegment);
        } else {
            Object[] newValues = new Object[valueCount];
            newData =
                new DenseSegmentDataset(
                    newSegment,
                    newValues);
        }

        // If the source is sparse, it is more efficient to iterate over the
        // values we need. If it's dense, it doesn't matter too much.
        int[] pos = new int[axes.length];
        CellKey newKey = CellKey.Generator.newRefCellKey(pos);
        data:
        for (Map.Entry<CellKey, Object> entry : data) {
            CellKey key = entry.getKey();

            // Map each of the source coordinates to the target coordinate.
            // If any of the coordinates maps to null, it means that the
            // cell falls outside the subset.
            for (int i = 0; i < pos.length; i++) {
                int ordinal = key.getAxis(i);

                Map<Integer, Integer> axisPosMap = axisPosMaps[i];
                if (axisPosMap == null) {
                    pos[i] = ordinal;
                } else {
                    Integer integer = axisPosMap.get(ordinal);
                    if (integer == null) {
                        continue data;
                    }
                    pos[i] = integer;
                }
            }
            newData.put(newKey, entry.getValue());
        }
        newSegment.setData(newData, null);

        return newSegment;
    }

    /**
     * Returns this Segment's dataset, or null if the data has not yet been
     * loaded.
     */
    SegmentDataset getData() {
        return data;
    }

    /**
     * <code>State</code> enumerates the allowable values of a segment's
     * state.
     */
    private static enum State {
        Initial, Loading, Ready, Failed
    }

    /**
     * Definition of a region of values which are not in a segment.
     *
     * <p>A region is defined by a set of constraints, one for each column
     * in the segment. A constraint may be
     * {@link mondrian.rolap.agg.LiteralStarPredicate}(true), meaning that
     * the column is unconstrained.
     *
     * <p>For example,
     * <pre>
     * segment (State={CA, OR, WA}, Gender=*)
     * actual values {1997, 1998} * {CA, OR, WA} * {M, F} = 12 cells
     * excluded region (Year=*, State={CA, OR}, Gender={F})
     * excluded values {1997, 1998} * {CA, OR} * {F} = 4 cells
     *
     * Values:
     *
     *     F     M
     * CA  out   in
     * OR  out   in
     * WA  in    in
     * </pre>
     *
     * <p>Note that the resulting segment is not a hypercube: it has a 'hole'.
     * This is why regions are required.
     */
    static class Region {
        private final StarColumnPredicate[] predicates;
        private final StarPredicate[] multiColumnPredicates;
        private final int cellCount;

        Region(
            List<StarColumnPredicate> predicateList,
            List<StarPredicate> multiColumnPredicateList,
            int cellCount)
        {
            this.predicates =
                predicateList.toArray(
                    new StarColumnPredicate[predicateList.size()]);
            this.multiColumnPredicates =
                multiColumnPredicateList.toArray(
                    new StarPredicate[multiColumnPredicateList.size()]);
            this.cellCount = cellCount;
        }

        public List<StarColumnPredicate> getPredicates() {
            return Collections.unmodifiableList(Arrays.asList(predicates));
        }

        public List<StarPredicate> getMultiColumnPredicates() {
            return Collections.unmodifiableList(
                Arrays.asList(multiColumnPredicates));
        }

        public int getCellCount() {
            return cellCount;
        }

        public boolean wouldContain(Object[] keys) {
            assert keys.length == predicates.length;
            for (int i = 0; i < keys.length; i++) {
                final Object key = keys[i];
                final StarColumnPredicate predicate = predicates[i];
                if (!predicate.evaluate(key)) {
                    return false;
                }
            }
            return true;
        }

        public boolean equals(Object obj) {
            if (obj instanceof Region) {
                Region that = (Region) obj;
                return Arrays.equals(
                    this.predicates, that.predicates) &&
                    Arrays.equals(
                        this.multiColumnPredicates,
                        that.multiColumnPredicates);
            } else {
                return false;
            }
        }

        public int hashCode() {
            return Arrays.hashCode(multiColumnPredicates) ^
                Arrays.hashCode(predicates);
        }

        /**
         * Describes this Segment.
         * @param buf Buffer to write to.
         */
        public void describe(StringBuilder buf) {
            int k = 0;
            for (StarColumnPredicate predicate : predicates) {
                if (predicate instanceof LiteralStarPredicate &&
                    ((LiteralStarPredicate) predicate).getValue()) {
                    continue;
                }
                if (k++ > 0) {
                    buf.append(" AND ");
                }
                predicate.describe(buf);
            }
            for (StarPredicate predicate : multiColumnPredicates) {
                if (predicate instanceof LiteralStarPredicate &&
                    ((LiteralStarPredicate) predicate).getValue()) {
                    continue;
                }
                if (k++ > 0) {
                    buf.append(" AND ");
                }
                predicate.describe(buf);
            }
        }
    }
}

// End Segment.java
