/* sbrkw32.c -- Windows emulation of `sbrk'.
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

/* Written by Eli Zaretskii <eliz@gnu.org>.  */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>
#include <malloc.h>
#include <errno.h>

#include "xunistd.h"

/* This global can be used to get at the allocated buffer when `sbrk'
   is called with a positive argument, to avoid memory leak.  The
   value returned by `sbrk' is NOT a valid memory address, since
   Windows does not allocate memory in one contiguous region.  */
void *sbrkw32_allocated_block;

static size_t
sbrkw32_report_heap_usage (void)
{
  _HEAPINFO heap_entry;
  int status;
  size_t heap_used = 0;

  heap_entry._pentry = NULL;

  /* We could return the largest address of all the heap regions
     instead, but I believe the sum of all sizes is a more useful
     value on MS-Windows.  For starters, it is guaranteed to be
     non-decreasing when we allocate more memory.  */
  while ((status = _heapwalk (&heap_entry)) == _HEAPOK)
    heap_used += heap_entry._size;

  return heap_used;
}

void *
sbrk (ptrdiff_t delta)
{
  size_t allocated = sbrkw32_report_heap_usage ();

  if (delta < 0)
    {
      errno = ENOSYS;
      return (void *)-1;
    }
  else if (delta > 0)
    {
      sbrkw32_allocated_block = malloc (delta);
      if (!sbrkw32_allocated_block)
	errno = ENOMEM;
    }
  return (void *)allocated;
}
