/* idcache.c -- map user and group IDs, cached for speed

   Copyright (C) 1985, 1988, 1989, 1990, 1997, 1998, 2003, 2005 Free
   Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.  */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#if !defined(__MINGW32__) && !defined(_MSC_VER)
#include <pwd.h>
#include <grp.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "xalloc.h"

#ifdef __DJGPP__
static char digits[] = "0123456789";
#endif

struct userid
{
  union
    {
      uid_t u;
      gid_t g;
    } id;
  char *name;
  struct userid *next;
};

static struct userid *user_alist;

/* The members of this list have names not in the local passwd file.  */
static struct userid *nouser_alist;

/* Translate UID to a login name, with cache, or NULL if unresolved.  */

char *
getuser (uid_t uid)
{
  register struct userid *tail;
  struct passwd *pwent;

  for (tail = user_alist; tail; tail = tail->next)
    if (tail->id.u == uid)
      return tail->name;

  pwent = getpwuid (uid);
  tail = xmalloc (sizeof *tail);
  tail->id.u = uid;
  tail->name = pwent ? xstrdup (pwent->pw_name) : NULL;

  /* Add to the head of the list, so most recently used is first.  */
  tail->next = user_alist;
  user_alist = tail;
  return tail->name;
}

/* Translate USER to a UID, with cache.
   Return NULL if there is no such user.
   (We also cache which user names have no passwd entry,
   so we don't keep looking them up.)  */

uid_t *
getuidbyname (const char *user)
{
  register struct userid *tail;
  struct passwd *pwent;

  for (tail = user_alist; tail; tail = tail->next)
    /* Avoid a function call for the most common case.  */
    if (*tail->name == *user && !strcmp (tail->name, user))
      return &tail->id.u;

  for (tail = nouser_alist; tail; tail = tail->next)
    /* Avoid a function call for the most common case.  */
    if (*tail->name == *user && !strcmp (tail->name, user))
      return 0;

  pwent = getpwnam (user);
#ifdef __DJGPP__
  /* We need to pretend to be the user USER, to make
     pwd functions know about an arbitrary user name.  */
  if (!pwent && strspn (user, digits) < strlen (user))
    {
      setenv ("USER", user, 1);
      pwent = getpwnam (user);	/* now it will succeed */
    }
#endif

  tail = xmalloc (sizeof *tail);
  tail->name = xstrdup (user);

  /* Add to the head of the list, so most recently used is first.  */
  if (pwent)
    {
      tail->id.u = pwent->pw_uid;
      tail->next = user_alist;
      user_alist = tail;
      return &tail->id.u;
    }

  tail->next = nouser_alist;
  nouser_alist = tail;
  return 0;
}

/* Use the same struct as for userids.  */
static struct userid *group_alist;
static struct userid *nogroup_alist;

/* Translate GID to a group name, with cache, or NULL if unresolved.  */

char *
getgroup (gid_t gid)
{
  register struct userid *tail;
  struct group *grent;

  for (tail = group_alist; tail; tail = tail->next)
    if (tail->id.g == gid)
      return tail->name;

  grent = getgrgid (gid);
  tail = xmalloc (sizeof *tail);
  tail->id.g = gid;
  tail->name = grent ? xstrdup (grent->gr_name) : NULL;

  /* Add to the head of the list, so most recently used is first.  */
  tail->next = group_alist;
  group_alist = tail;
  return tail->name;
}

/* Translate GROUP to a GID, with cache.
   Return NULL if there is no such group.
   (We also cache which group names have no group entry,
   so we don't keep looking them up.)  */

gid_t *
getgidbyname (const char *group)
{
  register struct userid *tail;
  struct group *grent;

  for (tail = group_alist; tail; tail = tail->next)
    /* Avoid a function call for the most common case.  */
    if (*tail->name == *group && !strcmp (tail->name, group))
      return &tail->id.g;

  for (tail = nogroup_alist; tail; tail = tail->next)
    /* Avoid a function call for the most common case.  */
    if (*tail->name == *group && !strcmp (tail->name, group))
      return 0;

  grent = getgrnam (group);
#ifdef __DJGPP__
  /* We need to pretend to belong to group GROUP, to make
     grp functions know about any arbitrary group name.  */
  if (!grent && strspn (group, digits) < strlen (group))
    {
      setenv ("GROUP", group, 1);
      grent = getgrnam (group);	/* now it will succeed */
    }
#endif

  tail = xmalloc (sizeof *tail);
  tail->name = xstrdup (group);

  /* Add to the head of the list, so most recently used is first.  */
  if (grent)
    {
      tail->id.g = grent->gr_gid;
      tail->next = group_alist;
      group_alist = tail;
      return &tail->id.g;
    }

  tail->next = nogroup_alist;
  nogroup_alist = tail;
  return 0;
}

#if defined(__MINGW32__) || defined(_MSC_VER)
#include <stdlib.h>
#include <windows.h>
#include <lmcons.h>

char *
getlogin (void)
{
  static char user_name[UNLEN+1];
  DWORD  length = sizeof (user_name);

  if (GetUserName (user_name, &length))
    return user_name;
  return NULL;
}

static struct passwd pwentry = {
  NULL, 0, 0, NULL, NULL
};

static int pw_count = 0;

struct passwd *
getpwent (void)
{
  if (!pwentry.pw_name)
    pwentry.pw_name = strdup (getlogin ());
  if (stricmp (pwentry.pw_name, "administrator") != 0)
    pwentry.pw_uid = 42;
  if (!pwentry.pw_dir)
    {
      char *home = getenv ("HOME");
      char *h;

      if (!home)
	home = "/";
      pwentry.pw_dir = strdup (home);
      for (h = pwentry.pw_dir; *h; h++)
	if (*h == '\\')
	  *h = '/';
    }
  if (!pwentry.pw_shell)
    {
      char *shell = getenv ("SHELL");
      char *s;

      if (!shell)
	shell = getenv ("ComSpec");
      if (!shell)
	shell = getenv ("COMSPEC");
      if (!shell)
	shell = "C:/WINDOWS/system32/cmd.exe";
      pwentry.pw_shell = strdup (shell);
      for (s = pwentry.pw_shell; *s; s++)
	if (*s == '\\')
	  *s = '/';
    }

  if (pw_count++ == 0)
    return &pwentry;
  return NULL;
}

void
setpwent (void)
{
  pw_count = 0;
}

void
endpwent (void)
{
  pw_count = 0;
}

struct passwd *
getpwnam (const char *name)
{
  char *user = getlogin ();

  if (strcmp (user, name))
    return NULL;
  setpwent ();
  return getpwent ();
}

struct passwd *
getpwuid (uid_t uid)
{
  (void) uid;
  setpwent ();
  return getpwent ();
}

static char *grentry_members[] = {
  NULL, NULL
};

static struct group grentry = {
  "root", 42, &grentry_members[0]
};

static int gr_count = 0;

struct group *
getgrent (void)
{
  if (!grentry_members[0])
    grentry_members[0] = getlogin ();

  if (gr_count++ == 0)
    return &grentry;
  return NULL;
}

void
setgrent (void)
{
  gr_count = 0;
}

void
endgrent (void)
{
  gr_count = 0;
}

struct group *
getgrgid (gid_t gid)
{
  (void) gid;
  setgrent ();
  return getgrent ();
}

struct group *
getgrnam (const char *name)
{
  if (strcmp (name, grentry.gr_name))
    return NULL;
  setgrent ();
  return getgrent ();
}

#endif
