// Aqsis
// Copyright (c) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**
 * Copyright (C) 2006-2007  NTT DATA CORPORATION
 * 
 * Version: 1.0.0 2007/04/01
 *  
 */
package net.cellcomputing.himawari.shaderexecenv;

import static net.cellcomputing.himawari.shaderexecenv.GlobalShaderExecEnv.*;
import static net.cellcomputing.himawari.library.types.PublicFunctions.*;
import static net.cellcomputing.himawari.library.RiGlobal.*;
import static net.cellcomputing.himawari.library.EqEnvVars.*;
import static net.cellcomputing.himawari.library.EqVariableClass.*;
import static net.cellcomputing.himawari.library.EqVariableType.*;
import static net.cellcomputing.himawari.library.EqIntIndex.*;
import static net.cellcomputing.himawari.library.EqMapType.*;
import static net.cellcomputing.himawari.library.CqStats.*;
import static java.lang.Math.*;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Stack;

import net.cellcomputing.himawari.accessory.STLArray;
import net.cellcomputing.himawari.accessory.STLVector;
import net.cellcomputing.himawari.accessory.Valarray;
import net.cellcomputing.himawari.accessory.primitive.p_String;
import net.cellcomputing.himawari.accessory.primitive.p_float;
import net.cellcomputing.himawari.library.EqVariableClass;
import net.cellcomputing.himawari.library.EqVariableType;
import net.cellcomputing.himawari.library.IqAttributes;
import net.cellcomputing.himawari.library.IqLightsource;
import net.cellcomputing.himawari.library.IqShader;
import net.cellcomputing.himawari.library.IqShaderData;
import net.cellcomputing.himawari.library.IqShaderExecEnv;
import net.cellcomputing.himawari.library.IqSurface;
import net.cellcomputing.himawari.library.IqTextureMap;
import net.cellcomputing.himawari.library.IqTransform;
import net.cellcomputing.himawari.library.types.CqBitVector;
import net.cellcomputing.himawari.library.types.CqCellNoise;
import net.cellcomputing.himawari.library.types.CqColor;
import net.cellcomputing.himawari.library.types.CqMatrix;
import net.cellcomputing.himawari.library.types.CqNoise;
import net.cellcomputing.himawari.library.types.CqRandom;
import net.cellcomputing.himawari.library.types.CqSplineCubic;
import net.cellcomputing.himawari.library.types.CqVector3D;
import net.cellcomputing.himawari.library.types.CqVector4D;
import net.cellcomputing.himawari.library.CqTransform;
import net.cellcomputing.himawari.util.HimawariLogger;

/**
 * 
 * W̃VF[_sB<br>
 * W̕ϐ܂łāASIMD̋@\񋟂܂B
 * 
 * @author NTT DATA Corporation
 */
public strictfp class CqShaderExecEnv extends IqShaderExecEnv {
	
	//private STLVector<IqShaderData>	m_apVariables = new STLVector<IqShaderData>(IqShaderData.class);		///< Vector of pointers to shader variables.
	private STLArray<IqShaderData>	m_apVariables = new STLArray<IqShaderData>(IqShaderData.class);		///< Vector of pointers to shader variables.
	private static CqNoise			m_noise			= CqNoise.getInstance();	///< One off noise generator, used by all envs.
	private static CqCellNoise		m_cellnoise		= new CqCellNoise();		///< One off cell noise generator, used by all envs.
	private static CqRandom		m_random		= new CqRandom();			///< One off random number generator used by all envs.
	private static CqMatrix		m_matIdentity	= new CqMatrix();
	
	private int	m_uGridRes;							///< The resolution of the grid in u.
	private int	m_vGridRes;							///< The resolution of the grid in u.
	private int	m_GridSize;							///< The resolution of the grid.
//	private int	m_GridI;							///< The current SIMD index.
	private int	m_li;								///< Light index, used during illuminance loop.
	private int	m_Illuminate;
	private boolean		m_IlluminanceCacheValid;	///< Flag indicating whether the illuminance cache is valid.
	private IqAttributes	m_pAttributes;				///< Pointer to the associated attributes.
	private CqTransform	m_pTransform;				///< Pointer to the associated transform.
	private CqBitVector	m_CurrentState = new CqBitVector();				///< SIMD execution state bit vector accumulator.
	private CqBitVector	m_RunningState = new CqBitVector();				///< SIMD running execution state bit vector.
	private Stack<CqBitVector>	m_stkState = new Stack<CqBitVector>();			///< Stack of execution state bit vectors.
	private int			m_LocalIndex;				///< Local cached variable index to speed repeated access to the same local variable.
	private IqSurface		m_pCurrentSurface;			///< Pointer to the surface being shaded.
	
	
	//************************************************************************//
	//*** RXgN^̒`
	//************************************************************************//
	
	public CqShaderExecEnv(){
		m_LocalIndex = 0;
		m_li = 0;
		m_Illuminate = 0;
		m_pAttributes = null;
		m_apVariables.setSize( EnvVars_Last );
		int i;
		for ( i = 0; i < EnvVars_Last; i++ )
			m_apVariables.set( i, null );
//		m_apVariables.get( i ] = 0;
	}
	
	public void destruct(){
		int i;
		for ( i = 0; i < EnvVars_Last; i++ )
			m_apVariables.get(i).destruct();
		
//		if ( m_pAttributes!=null ) RELEASEREF( m_pAttributes );
		if( m_pAttributes!=null ) m_pAttributes.Release();
	}
	
	//************************************************************************//
	//*** \bh̒`
	//************************************************************************//
	
	/** Internal function to extract additional named filter parameters from an array of stack entries.
	 */
	private void GetFilterParams( int cParams, IqShaderData[] apParams, p_float _pswidth, p_float _ptwidth )
	{
		p_String strParam = new p_String();
		p_float f = new p_float();
		
		int i = 0;
		while ( cParams > 0 )
		{
			apParams[ i ].GetString( strParam, 0 );
			apParams[ i + 1 ].GetFloat( f, 0 );
			
			if ( strParam.value.compareTo( "width" ) == 0 ) _pswidth.value = _ptwidth.value = f.value;
			else if ( strParam.value.compareTo( "swidth" ) == 0 ) _pswidth.value = f.value;
			else if ( strParam.value.compareTo( "twidth" ) == 0 ) _ptwidth.value = f.value;
			
			i += 2;
			cParams -= 2;
		}
	}
	
	/** Internal function to extract additional named texture control parameters from an array of stack entries.
	 */
	private void GetTexParams( int cParams, IqShaderData[] apParams, HashMap<String, IqShaderData> map )
	{
		p_String strParam = new p_String();
		int i = 0;
		while ( cParams > 0 )
		{
			apParams[ i ].GetString( strParam, 0 );
//			map[ strParam ] = apParams[ i + 1 ];
			map.put( strParam.value, apParams[ i + 1 ] );
			i += 2;
			cParams -= 2;
		}
	}
	
	
	//************************************************************************//
	//*** ۃ\bh̎
	//************************************************************************//
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Initialise(int, int, net.cellcomputing.himawari.library.IqAttributes, net.cellcomputing.himawari.library.IqTransform, net.cellcomputing.himawari.library.IqShader, int)
	 */
	@Override
	public void Initialise(int uGridRes, int vGridRes, IqAttributes pAttr, IqTransform pTrans, IqShader pShader, int Uses) {
		
		m_uGridRes = uGridRes;
		m_vGridRes = vGridRes;
		
		m_GridSize = ( uGridRes + 1 ) * ( vGridRes + 1 );
		m_LocalIndex = 0;
		
		// Store a pointer to the attributes definition.
		if ( null != pAttr )
		{
			if( null != m_pAttributes )
				m_pAttributes.Release();
//			RELEASEREF(m_pAttributes);
			m_pAttributes = pAttr;
			m_pAttributes.AddRef();
//			ADDREF(m_pAttributes);
		}
		else
			m_pAttributes = null;
		
		// Store a pointer to the transform.
		if (pTrans!=null)
		{
			m_pTransform = (CqTransform)(pTrans);
		}
		
		m_li = 0;
		m_Illuminate = 0;
		m_IlluminanceCacheValid = false;
		
		// Initialise the state bitvectors
		m_CurrentState.SetSize( m_GridSize );
		m_RunningState.SetSize( m_GridSize );
		m_RunningState.SetAll( true );
		
		
		if ( pShader != null )
		{
			if ( USES( Uses, EnvVars_P ) && m_apVariables.get( EnvVars_P ) == null )
				m_apVariables.set( EnvVars_P, pShader.CreateVariable( new EqVariableType(type_point), new EqVariableClass(class_varying), gVariableNames[ EnvVars_P ] ) );
			if ( USES( Uses, EnvVars_Cs ) && m_apVariables.get( EnvVars_Cs ) == null )
				m_apVariables.set( EnvVars_Cs, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Cs ] ) );
			if ( USES( Uses, EnvVars_Os ) && m_apVariables.get( EnvVars_Os ) == null )
				m_apVariables.set( EnvVars_Os, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Os ] ) );
			if ( USES( Uses, EnvVars_Ng ) && m_apVariables.get( EnvVars_Ng ) == null )
				m_apVariables.set( EnvVars_Ng, pShader.CreateVariable( new EqVariableType(type_normal), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Ng ] ) );
			if ( USES( Uses, EnvVars_du ) && m_apVariables.get( EnvVars_du ) == null )
				m_apVariables.set( EnvVars_du, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_du ] ) );
			if ( USES( Uses, EnvVars_dv ) && m_apVariables.get( EnvVars_dv ) == null )
				m_apVariables.set( EnvVars_dv, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_dv ] ) );
			if ( USES( Uses, EnvVars_L ) && m_apVariables.get( EnvVars_L ) == null )
				m_apVariables.set( EnvVars_L, pShader.CreateVariable( new EqVariableType(type_vector), new EqVariableClass(class_varying), gVariableNames[ EnvVars_L ] ) );
			if ( USES( Uses, EnvVars_Cl ) && m_apVariables.get( EnvVars_Cl ) == null )
				m_apVariables.set( EnvVars_Cl, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Cl ] ) );
			if ( USES( Uses, EnvVars_Ol ) && m_apVariables.get( EnvVars_Ol ) == null )
				m_apVariables.set( EnvVars_Ol, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Ol ] ) );
			if ( USES( Uses, EnvVars_dPdu ) && m_apVariables.get( EnvVars_dPdu ) == null )
				m_apVariables.set( EnvVars_dPdu, pShader.CreateVariable( new EqVariableType(type_vector), new EqVariableClass(class_varying), gVariableNames[ EnvVars_dPdu ] ) );
			if ( USES( Uses, EnvVars_dPdv ) && m_apVariables.get( EnvVars_dPdv ) == null )
				m_apVariables.set( EnvVars_dPdv, pShader.CreateVariable( new EqVariableType(type_vector), new EqVariableClass(class_varying), gVariableNames[ EnvVars_dPdv ] ) );
			if ( USES( Uses, EnvVars_N ) && m_apVariables.get( EnvVars_N ) == null )
				m_apVariables.set( EnvVars_N, pShader.CreateVariable( new EqVariableType(type_normal), new EqVariableClass(class_varying), gVariableNames[ EnvVars_N ] ) );
			if ( USES( Uses, EnvVars_u ) && m_apVariables.get( EnvVars_u ) == null )
				m_apVariables.set( EnvVars_u, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_u ] ) );
			if ( USES( Uses, EnvVars_v ) && m_apVariables.get( EnvVars_v ) == null )
				m_apVariables.set( EnvVars_v, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_v ] ) );
			if ( USES( Uses, EnvVars_s ) && m_apVariables.get( EnvVars_s ) == null )
				m_apVariables.set( EnvVars_s, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_s ] ) );
			if ( USES( Uses, EnvVars_t ) && m_apVariables.get( EnvVars_t ) == null )
				m_apVariables.set( EnvVars_t, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_t ] ) );
			if ( USES( Uses, EnvVars_I ) && m_apVariables.get( EnvVars_I ) == null )
				m_apVariables.set( EnvVars_I, pShader.CreateVariable( new EqVariableType(type_vector), new EqVariableClass(class_varying), gVariableNames[ EnvVars_I ] ) );
			if ( USES( Uses, EnvVars_Ci ) && m_apVariables.get( EnvVars_Ci ) == null )
				m_apVariables.set( EnvVars_Ci, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Ci ] ) );
			if ( USES( Uses, EnvVars_Oi ) && m_apVariables.get( EnvVars_Oi ) == null )
				m_apVariables.set( EnvVars_Oi, pShader.CreateVariable( new EqVariableType(type_color), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Oi ] ) );
			if ( USES( Uses, EnvVars_Ps ) && m_apVariables.get( EnvVars_Ps ) == null )
				m_apVariables.set( EnvVars_Ps, pShader.CreateVariable( new EqVariableType(type_point), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Ps ] ) );
			if ( USES( Uses, EnvVars_E ) && m_apVariables.get( EnvVars_E ) == null )
				m_apVariables.set( EnvVars_E, pShader.CreateVariable( new EqVariableType(type_point), new EqVariableClass(class_uniform), gVariableNames[ EnvVars_E ] ) );
			if ( USES( Uses, EnvVars_ncomps ) && m_apVariables.get( EnvVars_ncomps ) == null )
				m_apVariables.set( EnvVars_ncomps, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_uniform), gVariableNames[ EnvVars_ncomps ] ) );
			if ( USES( Uses, EnvVars_time ) && m_apVariables.get( EnvVars_time ) == null )
				m_apVariables.set( EnvVars_time, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_uniform), gVariableNames[ EnvVars_time ] ) );
			if ( USES( Uses, EnvVars_alpha ) && m_apVariables.get( EnvVars_alpha ) == null )
				m_apVariables.set( EnvVars_alpha, pShader.CreateVariable( new EqVariableType(type_float), new EqVariableClass(class_varying), gVariableNames[ EnvVars_alpha ] ) );
			if ( USES( Uses, EnvVars_Ns ) && m_apVariables.get( EnvVars_Ns ) == null )
				m_apVariables.set( EnvVars_Ns, pShader.CreateVariable( new EqVariableType(type_normal), new EqVariableClass(class_varying), gVariableNames[ EnvVars_Ns ] ) );
		}
		
		int i;
		for ( i = 0; i < EnvVars_Last; i++ )
		{
			if ( m_apVariables.get( i )!=null && USES( Uses, i ) )
				m_apVariables.get( i ).Initialise( uGridRes, vGridRes );
		}
		
		if( USES( Uses, EnvVars_time ) )
		{
			// First try setting this to the shutter open time
			// @todo: Think about an algorithm which distributes samples in time
			
			final float[] shutter = QGetRenderContextI().GetFloatOption( "System", "Shutter" );
			if( shutter != null )
			{
				final float[] shutteroffset = QGetRenderContextI().GetFloatOption( "shutter", "offset" );
				float offset = 0;
				if( shutteroffset != null )
				{
//					offset = *shutteroffset;
					offset = shutteroffset[0];
				}
				
				// insert the open time plus shutter offset
				m_apVariables.get( EnvVars_time ).SetFloat(  shutter[ 0 ] + offset );
			}
		}
		
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#uGridRes()
	 */
	@Override
	public int uGridRes() {
		return ( m_uGridRes );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#vGridRes()
	 */
	@Override
	public int vGridRes() {
		return ( m_vGridRes );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#GridSize()
	 */
	@Override
	public int GridSize() {
		return ( m_GridSize );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#matObjectToWorld()
	 */
	@Override
	public CqMatrix matObjectToWorld() {
		return ( m_pTransform==null ? m_matIdentity : m_pTransform.matObjectToWorld(m_pTransform.Time(0)) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#pAttributes()
	 */
	@Override
	public IqAttributes pAttributes() {
		return ( m_pAttributes );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#pTransform()
	 */
	@Override
	public IqTransform pTransform() {
		return ( (IqTransform)(m_pTransform) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SetCurrentSurface(net.cellcomputing.himawari.library.IqSurface)
	 */
	@Override
	public void SetCurrentSurface(IqSurface pEnv) {
		m_pCurrentSurface = pEnv;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#GetCurrentSurface()
	 */
	@Override
	public IqSurface GetCurrentSurface() {
		return(m_pCurrentSurface);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#ValidateIlluminanceCache(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void ValidateIlluminanceCache(IqShaderData pP, IqShaderData pN, IqShader pShader) {
		
		// If this is the first call to illuminance this time round, call all lights and setup the Cl and L caches.
		if ( !m_IlluminanceCacheValid )
		{
			IqShaderData Ns = (pN != null )? pN : N();
			IqShaderData Ps = (pP != null )? pP : P();
			int li = 0;
			while ( li < m_pAttributes.cLights() )
			{
				IqLightsource lp = m_pAttributes.pLight( li );
				// Initialise the lightsource
				lp.Initialise( uGridRes(), vGridRes() );
				m_Illuminate = 0;
				// Evaluate the lightsource
				lp.Evaluate( Ps, Ns, m_pCurrentSurface );
				li++;
			}
			m_IlluminanceCacheValid = true;
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#InvalidateIlluminanceCache()
	 */
	@Override
	public void InvalidateIlluminanceCache() {
		m_IlluminanceCacheValid = false;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#CurrentState()
	 */
	@Override
	public CqBitVector CurrentState() {
		return ( m_CurrentState );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#RunningState()
	 */
	@Override
	public CqBitVector RunningState() {
		return ( m_RunningState );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#GetCurrentState()
	 */
	@Override
	public void GetCurrentState() {
		m_RunningState.assignment( m_CurrentState );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#ClearCurrentState()
	 */
	@Override
	public void ClearCurrentState() {
		m_CurrentState.SetAll( false );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#PushState()
	 */
	@Override
	public void PushState() {
		m_stkState.push( new CqBitVector( m_RunningState ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#PopState()
	 */
	@Override
	public void PopState() {
		m_RunningState = new CqBitVector( m_stkState.peek() );
		m_stkState.pop();
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#InvertRunningState()
	 */
	@Override
	public void InvertRunningState() {
		m_RunningState.Complement();
		if ( !m_stkState.empty() )
			m_RunningState.Intersect( m_stkState.peek() );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#FindStandardVar(java.lang.String)
	 */
	@Override
	public IqShaderData FindStandardVar(String pname) {
		
		int tmp = m_LocalIndex;
		long htoken = pname.hashCode();
		
		for ( ; m_LocalIndex < EnvVars_Last; m_LocalIndex++ )
		{
			if ( gVariableTokens[ m_LocalIndex ] == htoken )
				return ( m_apVariables.get( m_LocalIndex ) );
		}
		
		for ( m_LocalIndex = 0; m_LocalIndex < tmp; m_LocalIndex++ )
		{
			if ( gVariableTokens[ m_LocalIndex ] == htoken )
				return ( m_apVariables.get( m_LocalIndex ) );
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#FindStandardVarIndex(java.lang.String)
	 */
	@Override
	public int FindStandardVarIndex(String pname) {
		
		int tmp = m_LocalIndex;
		long htoken = pname.hashCode();
		for ( ; m_LocalIndex < EnvVars_Last; m_LocalIndex++ )
		{
			if ( gVariableTokens[ m_LocalIndex ] == htoken )
				return ( m_LocalIndex );
		}
		
		for ( m_LocalIndex = 0; m_LocalIndex < tmp; m_LocalIndex++ )
		{
			if ( gVariableTokens[ m_LocalIndex ] == htoken )
				return ( m_LocalIndex );
		}
		return ( -1 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#pVar(int)
	 */
	@Override
	public IqShaderData pVar(int Index) {
		return ( m_apVariables.get( Index ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#DeleteVariable(int)
	 */
	@Override
	public void DeleteVariable(int Index) {
//		delete( m_apVariables.[ Index ] );
//		m_apVariables.[ Index ] = 0;
		if( m_apVariables.get( Index ) != null )	m_apVariables.get( Index ).destruct();
		m_apVariables.set( Index, null );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Cs()
	 */
	@Override
	public IqShaderData Cs() {
		 return ( m_apVariables.get( EnvVars_Cs ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Os()
	 */
	@Override
	public IqShaderData Os() {
		return ( m_apVariables.get( EnvVars_Os ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Ng()
	 */
	@Override
	public IqShaderData Ng() {
		 return ( m_apVariables.get( EnvVars_Ng ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#du()
	 */
	@Override
	public IqShaderData du() {
		 return ( m_apVariables.get( EnvVars_du ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#dv()
	 */
	@Override
	public IqShaderData dv() {
		return ( m_apVariables.get( EnvVars_dv ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#L()
	 */
	@Override
	public IqShaderData L() {
		return ( m_apVariables.get( EnvVars_L ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Cl()
	 */
	@Override
	public IqShaderData Cl() {
		 return ( m_apVariables.get( EnvVars_Cl ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Ol()
	 */
	@Override
	public IqShaderData Ol() {
		 return ( m_apVariables.get( EnvVars_Ol ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#P()
	 */
	@Override
	public IqShaderData P() {
		return ( m_apVariables.get( EnvVars_P ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#dPdu()
	 */
	@Override
	public IqShaderData dPdu() {
		return ( m_apVariables.get( EnvVars_dPdu ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#dPdv()
	 */
	@Override
	public IqShaderData dPdv() {
		return ( m_apVariables.get( EnvVars_dPdv ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#N()
	 */
	@Override
	public IqShaderData N() {
		return ( m_apVariables.get( EnvVars_N ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#u()
	 */
	@Override
	public IqShaderData u() {
		 return ( m_apVariables.get( EnvVars_u ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#v()
	 */
	@Override
	public IqShaderData v() {
		  return ( m_apVariables.get( EnvVars_v ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#s()
	 */
	@Override
	public IqShaderData s() {
		  return ( m_apVariables.get( EnvVars_s ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#t()
	 */
	@Override
	public IqShaderData t() {
		return ( m_apVariables.get( EnvVars_t ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#I()
	 */
	@Override
	public IqShaderData I() {
		 return ( m_apVariables.get( EnvVars_I ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Ci()
	 */
	@Override
	public IqShaderData Ci() {
		 return ( m_apVariables.get( EnvVars_Ci ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Oi()
	 */
	@Override
	public IqShaderData Oi() {
		 return ( m_apVariables.get( EnvVars_Oi ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Ps()
	 */
	@Override
	public IqShaderData Ps() {
		  return ( m_apVariables.get( EnvVars_Ps ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#E()
	 */
	@Override
	public IqShaderData E() {
		return ( m_apVariables.get( EnvVars_E ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#ncomps()
	 */
	@Override
	public IqShaderData ncomps() {
		return ( m_apVariables.get( EnvVars_ncomps ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#time()
	 */
	@Override
	public IqShaderData time() {
		return ( m_apVariables.get( EnvVars_time ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#alpha()
	 */
	@Override
	public IqShaderData alpha() {
		return ( m_apVariables.get( EnvVars_alpha ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#Ns()
	 */
	@Override
	public IqShaderData Ns() {
		  return ( m_apVariables.get( EnvVars_Ns ) );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_init_illuminance()
	 */
	@Override
	public boolean SO_init_illuminance() {
		m_li = -1;
		return ( SO_advance_illuminance() );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_advance_illuminance()
	 */
	@Override
	public boolean SO_advance_illuminance() {
		m_li++;
		while ( m_li < m_pAttributes.cLights() &&
		        m_pAttributes.pLight( m_li ).pShader().fAmbient() )
		{
			m_li++;
		}
		if ( m_li < m_pAttributes.cLights() )
			return ( true );
		else
			return ( false );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_radians(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_radians(IqShaderData degrees, IqShaderData Result, IqShader pShader) {
		
		STATS_INC( SHD_so_radians );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(degrees).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_degrees = new p_float();
				(degrees).GetFloat(_aq_degrees,__iGrid);
				(Result).SetFloat((float)toRadians( _aq_degrees.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
		
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_degrees(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_degrees(IqShaderData radians, IqShaderData Result, IqShader pShader) {
		
		STATS_INC( SHD_so_degrees );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(radians).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_radians = new p_float();
				(radians).GetFloat(_aq_radians,__iGrid);
				(Result).SetFloat((float)toDegrees( _aq_radians.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
		
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_sin(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_sin(IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_sin );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				(Result).SetFloat((float)( sin( _aq_a.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_asin(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_asin(IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_asin );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				(Result).SetFloat((float)( asin( _aq_a.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cos(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cos(IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cos );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				(Result).SetFloat((float)( cos( _aq_a.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_acos(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_acos(IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_acos );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				(Result).SetFloat((float)( acos( _aq_a.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_tan(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_tan(IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_tan );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				(Result).SetFloat((float)( tan( _aq_a.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_atan(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_atan(IqShaderData yoverx, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_atan );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(yoverx).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_yoverx = new p_float();
				(yoverx).GetFloat(_aq_yoverx,__iGrid);
				(Result).SetFloat((float)( atan( _aq_yoverx.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_atan(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_atan(IqShaderData y, IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_atan );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(y).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				p_float _aq_y = new p_float();
				(y).GetFloat(_aq_y,__iGrid);
				(Result).SetFloat((float)( atan2( _aq_y.value, _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);	
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pow(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pow(IqShaderData x, IqShaderData y, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pow );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(y).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				p_float _aq_y = new p_float();
				(y).GetFloat(_aq_y,__iGrid);
				p_float yy = _aq_y;
				p_float xx = _aq_x;
				if ( xx.value < 0.0f )
					yy.value = (float)floor( yy.value );
				(Result).SetFloat((float)( pow( xx.value, yy.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_exp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_exp(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_exp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( exp( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_sqrt(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_sqrt(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_sqrt );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( sqrt( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_log(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_log(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_log );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( log( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_log(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_log(IqShaderData x, IqShaderData base, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_log );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(base).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				p_float _aq_base = new p_float();
				(base).GetFloat(_aq_base,__iGrid);
				(Result).SetFloat((float)( log( _aq_x.value ) / log( _aq_base.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_mod(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_mod(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_mod );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				p_float _aq_b = new p_float();
				(b).GetFloat(_aq_b,__iGrid);
				int n = (int)( _aq_a.value / _aq_b.value );
				float a2 = _aq_a.value - n * _aq_b.value;
				if ( a2 < 0.0f )
					a2 += _aq_b.value;
				(Result).SetFloat(a2,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_abs(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_abs(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_abs );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( abs( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_sign(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_sign(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_sign );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat(( _aq_x.value < 0.0f ) ? -1.0f : 1.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_min(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_min(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_min );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				p_float _aq_b = new p_float();
				(b).GetFloat(_aq_b,__iGrid);
				float fRes = min( _aq_a.value, _aq_b.value );
				while ( cParams-- > 0 )
				{
					p_float fn = new p_float();
					apParams[ cParams ] .GetFloat( fn, __iGrid );
					fRes = min( fRes, fn.value );
				}
				(Result).SetFloat(fRes,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_max(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_max(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_max );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				p_float _aq_b = new p_float();
				(b).GetFloat(_aq_b,__iGrid);
				float fRes = max( _aq_a.value, _aq_b.value );
				while ( cParams-- > 0 )
				{
					p_float fn = new p_float();
					apParams[ cParams ] .GetFloat( fn, __iGrid );
					fRes = max( fRes, fn.value );
				}
				(Result).SetFloat(fRes,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pmin(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_pmin(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_pmin );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_a = new CqVector3D();
				(a).GetPoint(_aq_a,__iGrid);
				CqVector3D _aq_b = new CqVector3D();
				(b).GetPoint(_aq_b,__iGrid);
				CqVector3D res = VMIN( _aq_a, _aq_b );
				while ( cParams-- > 0 )
				{
					CqVector3D pn = new CqVector3D();
					apParams[ cParams ] .GetPoint( pn, __iGrid );
					res = VMIN( res, pn );
				}
				(Result).SetPoint(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pmax(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_pmax(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_pmax );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_a = new CqVector3D();
				(a).GetPoint(_aq_a,__iGrid);
				CqVector3D _aq_b = new CqVector3D();
				(b).GetPoint(_aq_b,__iGrid);
				CqVector3D res = VMAX( _aq_a, _aq_b );
				while ( cParams-- > 0 )
				{
					CqVector3D pn = new CqVector3D();
					apParams[ cParams ] .GetPoint( pn, __iGrid );
					res = VMAX( res, pn );
				}
				(Result).SetPoint(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cmin(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_cmin(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_cmin );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_a = new CqColor();
				(a).GetColor(_aq_a,__iGrid);
				CqColor _aq_b = new CqColor();
				(b).GetColor(_aq_b,__iGrid);
				CqColor res = CMIN( _aq_a, _aq_b );
				while ( cParams-- > 0 )
				{
					CqColor cn = new CqColor();
					apParams[ cParams ] .GetColor( cn, __iGrid );
					res = CMIN( res, cn );
				}
				(Result).SetColor(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cmax(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_cmax(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_cmax );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(b).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_a = new CqColor();
				(a).GetColor(_aq_a,__iGrid);
				CqColor _aq_b = new CqColor();
				(b).GetColor(_aq_b,__iGrid);
				CqColor res = CMAX( _aq_a, _aq_b );
				while ( cParams-- > 0 )
				{
					CqColor cn = new CqColor();
					apParams[ cParams ] .GetColor( cn, __iGrid );
					res = CMAX( res, cn );
				}
				(Result).SetColor(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_clamp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_clamp(IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_clamp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_min).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_max).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_a = new p_float();
				(a).GetFloat(_aq_a,__iGrid);
				p_float _aq__min = new p_float();
				(_min).GetFloat(_aq__min,__iGrid);
				p_float _aq__max = new p_float();
				(_max).GetFloat(_aq__max,__iGrid);
				(Result).SetFloat(CLAMP( _aq_a.value, _aq__min.value, _aq__max.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pclamp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pclamp(IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pclamp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_min).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_max).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_a = new CqVector3D();
				(a).GetPoint(_aq_a,__iGrid);
				CqVector3D _aq__min = new CqVector3D();
				(_min).GetPoint(_aq__min,__iGrid);
				CqVector3D _aq__max = new CqVector3D();
				(_max).GetPoint(_aq__max,__iGrid);
				(Result).SetPoint(VCLAMP( _aq_a, _aq__min, _aq__max ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cclamp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cclamp(IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cclamp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_min).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_max).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_a = new CqColor();
				(a).GetColor(_aq_a,__iGrid);
				CqColor _aq__min = new CqColor();
				(_min).GetColor(_aq__min,__iGrid);
				CqColor _aq__max = new CqColor();
				(_max).GetColor(_aq__max,__iGrid);
				(Result).SetColor(CCLAMP( _aq_a, _aq__min, _aq__max ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_floor(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_floor(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_floor );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( floor( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ceil(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ceil(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ceil );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat((float)( ceil( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_round(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_round(IqShaderData x, IqShaderData Result, IqShader pShader) {

		boolean __fVarying=false;
		int __iGrid;
		float res;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				if ( _aq_x.value >= 0.0f )
					res = (float)( (int)( _aq_x.value + 0.5f ) );
				else
					res = (float)( (int)( ( ceil( _aq_x.value - 0.5f ) ) ) );
				(Result).SetFloat(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_step(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_step(IqShaderData _min, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_step );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(_min).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq__min = new p_float();
				(_min).GetFloat(_aq__min,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				(Result).SetFloat(( _aq_value.value < _aq__min.value ) ? 0.0f : 1.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_smoothstep(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_smoothstep(IqShaderData _min, IqShaderData _max, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_smoothstep );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_min).Class().getValue()==class_varying||__fVarying;
		__fVarying=(_max).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq__min = new p_float();
				(_min).GetFloat(_aq__min,__iGrid);
				p_float _aq__max = new p_float();
				(_max).GetFloat(_aq__max,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value < _aq__min.value )
					(Result).SetFloat(0.0f,__iGrid);
				else if ( _aq_value.value >= _aq__max.value )
					(Result).SetFloat(1.0f,__iGrid);
				else
				{
					float v = ( _aq_value.value - _aq__min.value ) / ( _aq__max.value - _aq__min.value );
					(Result).SetFloat(v * v * ( 3.0f - 2.0f * v ),__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_fspline(IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_fspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );
		spline.SetBasis("catmull-rom");

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					p_float fl = new p_float();
					apParams[ cParams - 2 ] .GetFloat( fl, __iGrid );
					(Result).SetFloat(fl.value,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					p_float ff = new p_float();
					apParams[ 1 ] .GetFloat( ff, __iGrid );
					(Result).SetFloat(ff.value,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						p_float fn = new p_float();
						apParams[ j ] .GetFloat( fn, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( fn.value, 0.0f, 0.0f, 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetFloat(res.x,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_cspline(IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_cspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					CqColor cl = new CqColor();
					apParams[ cParams - 2 ] .GetColor( cl, __iGrid );
					(Result).SetColor(cl,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					CqColor cf = new CqColor();
					apParams[ 1 ] .GetColor( cf, __iGrid );
					(Result).SetColor(cf,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						CqColor cn = new CqColor();
						apParams[ j ] .GetColor( cn, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( cn.fRed(), cn.fGreen(), cn.fBlue(), 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetColor(new CqColor( res.x, res.y, res.z ),__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_pspline(IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_pspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					CqVector3D pl = new CqVector3D();
					apParams[ cParams - 2 ] .GetPoint( pl, __iGrid );
					(Result).SetPoint(pl,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					CqVector3D pf = new CqVector3D();
					apParams[ 1 ] .GetPoint( pf, __iGrid );
					(Result).SetPoint(pf,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						CqVector3D pn = new CqVector3D();
						apParams[ j ] .GetPoint( pn, __iGrid );
						spline.valueAt( j ).assignment( pn );
					}

					CqVector3D	res = new CqVector3D( spline.Evaluate( _aq_value.value ) );
					(Result).SetPoint(res,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_sfspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_sfspline(IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_sfspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					p_float fl = new p_float();
					apParams[ cParams - 2 ] .GetFloat( fl, __iGrid );
					(Result).SetFloat(fl.value,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					p_float ff = new p_float();
					apParams[ 1 ] .GetFloat( ff, __iGrid );
					(Result).SetFloat(ff.value,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						p_float fn = new p_float();
						apParams[ j ] .GetFloat( fn, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( fn.value, 0.0f, 0.0f, 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetFloat(res.x,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_scspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_scspline(IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_scspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
//		p_float _aq_basis = new p_float();		// String̊ԈႢ̉\lāAStringɕύX@nttdata
//		(basis).GetFloat(_aq_basis,__iGrid);
//		spline.SetBasis( _aq_basis.toString() );
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					CqColor cl = new CqColor();
					apParams[ cParams - 2 ] .GetColor( cl, __iGrid );
					(Result).SetColor(cl,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					CqColor cf = new CqColor();
					apParams[ 1 ] .GetColor( cf, __iGrid );
					(Result).SetColor(cf,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						CqColor cn = new CqColor();
						apParams[ j ] .GetColor( cn, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( cn.fRed(), cn.fGreen(), cn.fBlue(), 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetColor(new CqColor( res.x, res.y, res.z ),__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_spspline(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_spspline(IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_spspline );

		boolean __fVarying=false;
		int __iGrid;

		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		int v;
		for ( v = 0; v < cParams; v++ )
		{
			__fVarying=(( apParams[ v ] )).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				if ( _aq_value.value >= 1.0f )
				{
					CqVector3D pl = new CqVector3D();
					apParams[ cParams - 2 ] .GetPoint( pl, __iGrid );
					(Result).SetPoint(pl,__iGrid);
				}
				else if ( _aq_value.value <= 0.0f )
				{
					CqVector3D pf = new CqVector3D();
					apParams[ 1 ] .GetPoint( pf, __iGrid );
					(Result).SetPoint(pf,__iGrid);
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						CqVector3D pn = new CqVector3D();
						apParams[ j ] .GetPoint( pn, __iGrid );
						spline.valueAt( j ).assignment( pn );
					}

					CqVector3D	res = new CqVector3D( spline.Evaluate( _aq_value.value ) );
					(Result).SetPoint(res,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fDu(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fDu(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fDu );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetFloat(SO_DuType_TqFloat( p, __iGrid, this, Deffloat ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fDv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fDv(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fDv );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetFloat(SO_DvType_TqFloat( p, __iGrid, this, Deffloat ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fDeriv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fDeriv(IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fDeriv );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(den).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetFloat(SO_DerivType_TqFloat( p, den, __iGrid, this ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cDu(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cDu(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cDu );

		CqColor Defcol = new CqColor( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetColor(SO_DuType_CqColor( p, __iGrid, this, Defcol ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cDv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cDv(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cDv );

		CqColor Defcol = new CqColor( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetColor(SO_DvType_CqColor( p, __iGrid, this, Defcol ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cDeriv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cDeriv(IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cDeriv );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(den).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetColor(SO_DerivType_CqColor( p, den, __iGrid, this ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pDu(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pDu(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pDu );

		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(SO_DuType_CqVector3D( p, __iGrid, this, Defvec ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pDv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pDv(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pDv );

		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(SO_DvType_CqVector3D( p, __iGrid, this, Defvec ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pDeriv(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pDeriv(IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pDeriv );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(den).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(SO_DerivType_CqVector3D( p, den, __iGrid, this ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_frandom(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_frandom(IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_frandom );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetFloat(m_random.RandomFloat(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);	
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_crandom(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_crandom(IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_crandom );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				float a, b, c;
	                        a = m_random.RandomFloat();
	                        b = m_random.RandomFloat();
	                        c = m_random.RandomFloat();

				(Result).SetColor(new CqColor(a,b,c),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_prandom(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_prandom(IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_prandom );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				float a, b, c;
	                        a = m_random.RandomFloat();
	                        b = m_random.RandomFloat();
	                        c = m_random.RandomFloat();

				(Result).SetPoint(new CqVector3D(a,b,c),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetFloat(( m_noise.FGNoise1( _aq_v.value ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetFloat(( m_noise.FGNoise2( _aq_u.value, _aq_v.value ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetFloat(( m_noise.FGNoise3( _aq_p ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fnoise4(IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader) {

		//TODO: Do proper 4D noise.
		STATS_INC( SHD_so_fnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				(Result).SetFloat(( m_noise.FGNoise3( _aq_p ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetColor(( m_noise.CGNoise1( _aq_v.value ).add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetColor(( m_noise.CGNoise2( _aq_u.value, _aq_v.value ).add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetColor(( m_noise.CGNoise3( _aq_p ) .add( 1 ) ) .div( 2.0f ), __iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cnoise4(IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader) {
		
		// TODO: Do proper 4D noise.
		STATS_INC( SHD_so_cnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				(Result).SetColor(( m_noise.CGNoise3( _aq_p ) .add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetPoint(( m_noise.PGNoise1( _aq_v.value ) .add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetPoint(( m_noise.PGNoise2( _aq_u.value, _aq_v.value ) .add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetPoint(( m_noise.PGNoise3( _aq_p ) .add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pnoise4(IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader) {

		// TODO: Do proper 4D noise.
		STATS_INC( SHD_so_pnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				(Result).SetPoint(( m_noise.PGNoise3( _aq_p ) .add( 1 ) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_setcomp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_setcomp(IqShaderData p, IqShaderData index, IqShaderData v, IqShader pShader) {

		STATS_INC( SHD_so_setcomp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(index).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_p = new CqColor();
				(p).GetColor(_aq_p,__iGrid);
				p_float _aq_index = new p_float();
				(index).GetFloat(_aq_index,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				_aq_p.setValueAt( (int)( _aq_index.value ), _aq_v.value );
				(p).SetColor(_aq_p,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_setxcomp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_setxcomp(IqShaderData p, IqShaderData v, IqShader pShader) {

		STATS_INC( SHD_so_setxcomp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				_aq_p.x = ( _aq_v.value );
				(p).SetPoint(_aq_p,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_setycomp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_setycomp(IqShaderData p, IqShaderData v, IqShader pShader) {

		STATS_INC( SHD_so_setycomp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				_aq_p.y = ( _aq_v.value );
				(p).SetPoint(_aq_p,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_setzcomp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_setzcomp(IqShaderData p, IqShaderData v, IqShader pShader) {

		STATS_INC( SHD_so_setzcomp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				_aq_p.z = ( _aq_v.value );
				(p).SetPoint(_aq_p,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_length(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_length(IqShaderData V, IqShaderData Result, IqShader pShader) {
		
		STATS_INC( SHD_so_length );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(V).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_V = new CqVector3D();
				(V).GetVector(_aq_V,__iGrid);
				(Result).SetFloat(_aq_V.Magnitude(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_distance(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_distance(IqShaderData P1, IqShaderData P2, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_distance );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(P1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(P2).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_P1 = new CqVector3D();
				(P1).GetPoint(_aq_P1,__iGrid);
				CqVector3D _aq_P2 = new CqVector3D();
				(P2).GetPoint(_aq_P2,__iGrid);
				(Result).SetFloat(( _aq_P1.sub(_aq_P2) ).Magnitude(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_area(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_area(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_area );

		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		CqVector3D	vecR;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				if ( m_pAttributes!=null )
				{
					p_float fdu = new p_float(), fdv = new p_float();
					du() .GetFloat( fdu, __iGrid );
					dv() .GetFloat( fdv, __iGrid );
					vecR = ( SO_DuType_CqVector3D( p, __iGrid, this, Defvec ).mul(fdu.value) ) .mod(
					       ( SO_DvType_CqVector3D( p, __iGrid, this, Defvec ).mul(fdv.value) ) );
					(Result).SetFloat(vecR.Magnitude(),__iGrid);
				}

			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_normalize(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_normalize(IqShaderData V, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_normalize );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(V).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_V = new CqVector3D();
				(V).GetVector(_aq_V,__iGrid);
				_aq_V.Unit();
				(Result).SetVector(_aq_V,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_faceforward(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
//	public void SO_faceforward(IqShaderData N, IqShaderData I, IqShaderData Result, IqShader pShader) {
//
//		STATS_INC( SHD_so_faceforward );
//
//		boolean __fVarying=false;
//		int __iGrid;
//
//		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
//		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
//		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;
//
//		__iGrid = 0;
//		CqBitVector RS = RunningState();
//		do
//		{
//			if(!__fVarying || RS.Value( __iGrid ) )
//			{
//				CqVector3D _aq_N = new CqVector3D();
//				(N).GetNormal(_aq_N,__iGrid);
//				CqVector3D _aq_I = new CqVector3D();
//				(I).GetVector(_aq_I,__iGrid);
//				CqVector3D Nref = new CqVector3D();
//				Ng() .GetNormal( Nref, __iGrid );
////				float s = ( ( ( -_aq_I ) * Nref ) < 0.0f ) ? -1.0f : 1.0f;
//				float s = ( ( ( _aq_I.negative() ) .mul(Nref) ) < 0.0f ) ? -1.0f : 1.0f;
//				(Result).SetNormal(_aq_N .mul( s ),__iGrid);
//			}
//		}
//		while( ( ++__iGrid < GridSize() ) && __fVarying);
//	}
	public void SO_faceforward(IqShaderData N, IqShaderData I, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_faceforward );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		
		
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);

				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				
				CqVector3D Nref = new CqVector3D();
				Ng() .GetNormal( Nref, __iGrid );
//				float s = ( ( ( -_aq_I ) * Nref ) < 0.0f ) ? -1.0f : 1.0f;
				float s = ( ( ( _aq_I.negative() ) .mul(Nref) ) < 0.0f ) ? -1.0f : 1.0f;
				(Result).SetNormal(_aq_N .mul( s ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_faceforward2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_faceforward2(IqShaderData N, IqShaderData I, IqShaderData Nref, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_faceforward2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Nref).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				CqVector3D _aq_Nref = new CqVector3D();
				(Nref).GetNormal(_aq_Nref,__iGrid);
//				float s = ( ( ( -_aq_I ) * _aq_Nref ) < 0.0f ) ? -1.0f : 1.0f;
				float s = ( ( ( _aq_I.negative() ) .mul(_aq_Nref) ) < 0.0f ) ? -1.0f : 1.0f;
				(Result).SetNormal(_aq_N .mul( s ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_reflect(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_reflect(IqShaderData I, IqShaderData N, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_reflect );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				float idn = 2.0f * ( _aq_I .mul(_aq_N) );
				CqVector3D res = _aq_I .sub( ( _aq_N.mul(idn) ) );
				(Result).SetVector(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_refract(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_refract(IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_refract );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(eta).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				p_float _aq_eta = new p_float();
				(eta).GetFloat(_aq_eta,__iGrid);
				float IdotN = _aq_I .mul( _aq_N );
				float feta = _aq_eta.value;
				float k = 1 - feta * feta * ( 1 - IdotN * IdotN );
				(Result).SetVector(( k < 0.0f ) ? new CqVector3D( 0, 0, 0 ) : new CqVector3D( _aq_I.mul(feta) .sub( _aq_N.mul( feta * IdotN + (float)sqrt( k ) ) ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fresnel(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fresnel(IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShader pShader) {

		STATS_INC( SHD_so_fresnel );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(eta).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Kr).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Kt).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				p_float _aq_eta = new p_float();
				(eta).GetFloat(_aq_eta,__iGrid);
				p_float _aq_Kr = new p_float();
				(Kr).GetFloat(_aq_Kr,__iGrid);
				p_float _aq_Kt = new p_float();
				(Kt).GetFloat(_aq_Kt,__iGrid);
				float cos_theta = _aq_I.negative().mul( _aq_N );
				float fuvA = ((1.0f / _aq_eta.value)*(1.0f / _aq_eta.value)) - ( 1.0f - ((cos_theta)*(cos_theta)) );
				float fuvB = abs( fuvA );
				float fu2 = ( fuvA + fuvB ) / 2;
				float fv2 = ( -fuvA + fuvB ) / 2;
				float fv2sqrt = ( fv2 == 0.0f ) ? 0.0f : (float)sqrt( abs( fv2 ) );
				float fu2sqrt = ( fu2 == 0.0f ) ? 0.0f : (float)sqrt( abs( fu2 ) );
				float fperp2 = ( ((cos_theta - fu2sqrt)*(cos_theta - fu2sqrt)) + fv2 ) / ( ((cos_theta + fu2sqrt)*(cos_theta + fu2sqrt)) + fv2 );
				float feta = _aq_eta.value;
				float fpara2 = ( ((((1.0f / feta)*(1.0f / feta)) * cos_theta - fu2sqrt)*(((1.0f / feta)*(1.0f / feta)) * cos_theta - fu2sqrt)) + ((-fv2sqrt)*(-fv2sqrt)) ) /
				                 ( ((((1.0f / feta)*(1.0f / feta)) * cos_theta + fu2sqrt)*(((1.0f / feta)*(1.0f / feta)) * cos_theta + fu2sqrt)) + ((fv2sqrt)*(fv2sqrt)) );

				float __Kr = 0.5f * ( fperp2 + fpara2 );
				(Kr).SetFloat(__Kr,__iGrid);
				(Kt).SetFloat(1.0f - __Kr,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fresnel(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fresnel(IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShaderData R, IqShaderData T, IqShader pShader) {

		STATS_INC( SHD_so_fresnel );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(I).Class().getValue()==class_varying||__fVarying;
		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(eta).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Kr).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Kt).Class().getValue()==class_varying||__fVarying;
		__fVarying=(R).Class().getValue()==class_varying||__fVarying;
		__fVarying=(T).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_I = new CqVector3D();
				(I).GetVector(_aq_I,__iGrid);
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				p_float _aq_eta = new p_float();
				(eta).GetFloat(_aq_eta,__iGrid);
				p_float _aq_Kr = new p_float();
				(Kr).GetFloat(_aq_Kr,__iGrid);
				p_float _aq_Kt = new p_float();
				(Kt).GetFloat(_aq_Kt,__iGrid);
				CqVector3D _aq_R = new CqVector3D();
				(R).GetVector(_aq_R,__iGrid);
				CqVector3D _aq_T = new CqVector3D();
				(T).GetVector(_aq_T,__iGrid);
				float cos_theta = _aq_I.negative().mul( _aq_N );
				float fuvA = ((1.0f / _aq_eta.value)*(1.0f / _aq_eta.value)) - ( 1.0f - ((cos_theta)*(cos_theta)) );
				float fuvB = abs( fuvA );
				float fu2 = ( fuvA + fuvB ) / 2;
				float fv2 = ( -fuvA + fuvB ) / 2;
				float feta = _aq_eta.value;
				float fv2sqrt = ( fv2 == 0.0f ) ? 0.0f : (float)sqrt( abs( fv2 ) );
				float fu2sqrt = ( fu2 == 0.0f ) ? 0.0f : (float)sqrt( abs( fu2 ) );
				float fperp2 = ( ((cos_theta - fu2sqrt)*(cos_theta - fu2sqrt)) + fv2 ) / ( ((cos_theta + fu2sqrt)*(cos_theta + fu2sqrt)) + fv2 );
				float fpara2 = ( ((((1.0f / feta)*(1.0f / feta)) * cos_theta - fu2sqrt)*(((1.0f / feta)*(1.0f / feta)) * cos_theta - fu2sqrt)) + ((-fv2sqrt)*(-fv2sqrt)) ) /
				                 ( ((((1.0f / feta)*(1.0f / feta)) * cos_theta + fu2sqrt)*(((1.0f / feta)*(1.0f / feta)) * cos_theta + fu2sqrt)) + ((fv2sqrt)*(fv2sqrt)) );
				float __Kr = 0.5f * ( fperp2 + fpara2 );
				(Kr).SetFloat(__Kr,__iGrid);
				(Kt).SetFloat(1.0f - __Kr,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		SO_reflect( I, N, R );
		SO_refract( I, N, eta, T );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_transform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_transform(IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_transform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_fromspace = new p_String();
			(fromspace).GetString(_aq_fromspace,__iGrid);
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matSpaceToSpace( _aq_fromspace.value, _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );


			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetPoint(_aq_p,__iGrid);
					(Result).SetPoint(mat .multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetPoint(_aq_p,__iGrid);
					(Result).SetPoint(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_transform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_transform(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_transform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matSpaceToSpace( "current", _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );


			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetPoint(_aq_p,__iGrid);
					(Result).SetPoint(mat.multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetPoint(_aq_p,__iGrid);
					(Result).SetPoint(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_transformm(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_transformm(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_transform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_tospace = new CqMatrix();
				(tospace).GetMatrix(_aq_tospace,__iGrid);
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetPoint(_aq_tospace.multiply( _aq_p ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_vtransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_vtransform(IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_vtransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_fromspace = new p_String();
			(fromspace).GetString(_aq_fromspace,__iGrid);
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matVSpaceToSpace( _aq_fromspace.value, _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );


			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetVector(_aq_p,__iGrid);
					(Result).SetVector(mat.multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetVector(_aq_p,__iGrid);
					(Result).SetVector(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_vtransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_vtransform(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_vtransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matVSpaceToSpace( "current", _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );


			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetVector(_aq_p,__iGrid);
					(Result).SetVector(mat.multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetVector(_aq_p,__iGrid);
					(Result).SetVector(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_vtransformm(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_vtransformm(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_vtransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_tospace = new CqMatrix();
				(tospace).GetMatrix(_aq_tospace,__iGrid);
				CqVector3D _aq_p = new CqVector3D();
				(p).GetVector(_aq_p,__iGrid);
				(Result).SetVector(_aq_tospace.multiply( _aq_p ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ntransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ntransform(IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ntransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_fromspace = new p_String();
			(fromspace).GetString(_aq_fromspace,__iGrid);
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matNSpaceToSpace( _aq_fromspace.value, _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );
			__iGrid = 0;

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetNormal(_aq_p,__iGrid);
					(Result).SetNormal(mat.multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetNormal(_aq_p,__iGrid);
					(Result).SetNormal(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ntransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ntransform(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ntransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		if ( null != QGetRenderContextI() )
		{
			__iGrid = 0;
			p_String _aq_tospace = new p_String();
			(tospace).GetString(_aq_tospace,__iGrid);
			final CqMatrix mat = QGetRenderContextI() .matNSpaceToSpace( "current", _aq_tospace.value, pShader.matCurrent(), matObjectToWorld(), QGetRenderContextI().Time() );
			__iGrid = 0;

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetNormal(_aq_p,__iGrid);
					(Result).SetNormal(mat.multiply( _aq_p ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_p = new CqVector3D();
					(p).GetNormal(_aq_p,__iGrid);
					(Result).SetNormal(_aq_p,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}	
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ntransformm(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ntransformm(IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ntransform );

		boolean __fVarying=false;
		int __iGrid;

		assert( pShader != null );

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_tospace = new CqMatrix();
				(tospace).GetMatrix(_aq_tospace,__iGrid);
				CqVector3D _aq_p = new CqVector3D();
				(p).GetNormal(_aq_p,__iGrid);
				(Result).SetNormal(_aq_tospace.multiply( _aq_p ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_depth(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_depth(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_depth );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				float d = _aq_p.z;
				d = ( d - QGetRenderContextI() .GetFloatOption( "System", "Clipping" ) [ 0 ] ) /
				    ( QGetRenderContextI() .GetFloatOption( "System", "Clipping" ) [ 1 ] - QGetRenderContextI() .GetFloatOption( "System", "Clipping" ) [ 0 ] );
				(Result).SetFloat(d,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_calculatenormal(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_calculatenormal(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_calculatenormal );

//		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		// Find out if the orientation is inverted.
		boolean CSO = pTransform().GetHandedness(QGetRenderContextI().Time());
		boolean O = false;
		if( pAttributes()!=null )
			O = pAttributes() .GetIntegerAttribute( "System", "Orientation" ) [ 0 ] != 0;
		float neg = 1;
		if ( O != CSO )
			neg = -1;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				//CqVector3D	dPdu = SO_DuType<CqVector3D>( p, __iGrid, this, Defvec );
				//CqVector3D	dPdv = SO_DuType_CqVector3D( p, __iGrid, this, Defvec );
				//CqVector3D	N = dPdu % dPdv;

				CqVector3D Ret = new CqVector3D(), Ret2 = new CqVector3D();
				int uRes = uGridRes();
				int GridX = __iGrid % ( uRes + 1 );

				CqVector3D v1 = new CqVector3D(), v2 = new CqVector3D();
				if ( GridX < uRes )
				{
					p.GetValue( v1, __iGrid + 1 );
					p.GetValue( v2, __iGrid );
					Ret = ( v1 .sub(v2) );
				}
				else
				{
					p.GetValue( v1, __iGrid );
					p.GetValue( v2, __iGrid - 1 );
					Ret = ( v1 .sub(v2) );
				}
				int vRes = vGridRes();
				int GridY = ( __iGrid / ( uRes + 1 ) );

				if ( GridY < vRes )
				{
					p.GetValue( v1, __iGrid + uRes + 1 );
					p.GetValue( v2, __iGrid );
					Ret2 = ( v1 .sub(v2) );
				}
				else
				{
					p.GetValue( v1, __iGrid );
					p.GetValue( v2, __iGrid - ( uRes + 1 ) );
					Ret2 = ( v1 .sub(v2) );
				}

				CqVector3D N = Ret .mod( Ret2 );
				N.Unit();
				N.assignMul( neg );
				(Result).SetNormal(N,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cmix(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cmix(IqShaderData color0, IqShaderData color1, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cmix );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(color0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(color1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_color0 = new CqColor();
				(color0).GetColor(_aq_color0,__iGrid);
				CqColor _aq_color1 = new CqColor();
				(color1).GetColor(_aq_color1,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				CqColor c = ( _aq_color0.mul( 1.0f - _aq_value.value ).add( _aq_color1.mul(_aq_value.value) ) );
				(Result).SetColor(c,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fmix(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fmix(IqShaderData f0, IqShaderData f1, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fmix );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(f0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(f1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_f0 = new p_float();
				(f0).GetFloat(_aq_f0,__iGrid);
				p_float _aq_f1 = new p_float();
				(f1).GetFloat(_aq_f1,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				float f = ( ( 1.0f - _aq_value.value ) * _aq_f0.value + _aq_value.value * _aq_f1.value );
				(Result).SetFloat(f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pmix(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pmix(IqShaderData p0, IqShaderData p1, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pmix );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(p1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p0 = new CqVector3D();
				(p0).GetPoint(_aq_p0,__iGrid);
				CqVector3D _aq_p1 = new CqVector3D();
				(p1).GetPoint(_aq_p1,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				CqVector3D p = ( _aq_p0.mul( 1.0f - _aq_value.value ) .add( _aq_p1.mul(_aq_value.value) ) );
				(Result).SetPoint(p,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_vmix(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_vmix(IqShaderData v0, IqShaderData v1, IqShaderData value, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_vmix );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_v0 = new CqVector3D();
				(v0).GetVector(_aq_v0,__iGrid);
				CqVector3D _aq_v1 = new CqVector3D();
				(v1).GetVector(_aq_v1,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				CqVector3D v = ( _aq_v0.mul( 1.0f - _aq_value.value ) .add( _aq_v1.mul( _aq_value.value ) ) );
				(Result).SetVector(v,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_nmix(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_nmix(IqShaderData n0, IqShaderData n1, IqShaderData value, IqShaderData Result, IqShader pShader) {

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(n0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(n1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(value).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_n0 = new CqVector3D();
				(n0).GetNormal(_aq_n0,__iGrid);
				CqVector3D _aq_n1 = new CqVector3D(9);
				(n1).GetNormal(_aq_n1,__iGrid);
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);
				CqVector3D n = ( _aq_n0.mul( 1.0f - _aq_value.value ) .add( _aq_n1.mul(_aq_value.value) ) );
				(Result).SetNormal(n,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ambient(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ambient(IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ambient );

		boolean __fVarying=false;
		int __iGrid;

		// Use the lightsource stack on the current surface
		if ( m_pAttributes != null )
		{
			// If this is the first call to illuminance this time round, call all lights and setup the Cl and L caches.
			if ( !m_IlluminanceCacheValid )
			{
				ValidateIlluminanceCache( null, null, pShader );
			}

			Result.SetColor( gColBlack );

			for ( int light_index = 0; light_index < m_pAttributes .cLights(); light_index++ )
			{
				__fVarying = true;

				IqLightsource lp = m_pAttributes .pLight( light_index );
				if ( lp.pShader() .fAmbient() )
				{
					__iGrid = 0;
					CqBitVector RS = RunningState();
					do
					{
						if(!__fVarying || RS.Value( __iGrid ) )
						{
							// Now Combine the color of all ambient lightsources.
							CqColor _aq_Result = new CqColor();
							(Result).GetColor(_aq_Result,__iGrid);
							CqColor colCl = new CqColor();
							if ( null != lp.Cl() )
								lp.Cl() .GetColor( colCl, __iGrid );
							(Result).SetColor(_aq_Result .add( colCl ),__iGrid);

						}
					}
					while( ( ++__iGrid < GridSize() ) && __fVarying);
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_diffuse(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
//	public void SO_diffuse(IqShaderData N, IqShaderData Result, IqShader pShader) {
//
//		STATS_INC( SHD_so_diffuse );
//
//		boolean __fVarying=false;
//		int __iGrid;
//
//		// If the illuminance cache is already OK, then we don't need to bother filling in the illuminance parameters.
//		if ( !m_IlluminanceCacheValid )
//		{
//			ValidateIlluminanceCache( null, N, pShader );
//		}
//
//		IqShaderData pDefAngle = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_uniform) );
//		if ( null == pDefAngle )
//			return ;
//
//		pDefAngle.SetFloat( (float)PI/2.0f );
//
//		Result.SetColor( gColBlack );
//
//		__fVarying = true;
//		IqShaderData __nondiffuse = null;
//		__nondiffuse = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_varying) );
//
//		// SO_init_illuminance returns TRUE if there are any non ambient ligthsources available.
//		if ( SO_init_illuminance() )
//		{
//			IqShader pLightsource = null;
//			do
//			{
//				// Get the "__nondiffuse" setting from the current lightsource, if specified.
//				p_float	__nondiffuse_val = new p_float();
//				if ( m_li < m_pAttributes .cLights() )
//					pLightsource = m_pAttributes .pLight( m_li ) .pShader();
//				if ( pLightsource!=null )
//				{
//					pLightsource.GetValue( "__nondiffuse", __nondiffuse );
//					/// \note: This is OK here, outside the BEGIN_VARYING_SECTION as, varying in terms of lightsources
//					/// is not valid.
//					if( null != __nondiffuse )
//					{
//						__nondiffuse.GetFloat( __nondiffuse_val, 0 );
//						if( __nondiffuse_val.value != 0.0f )
//							continue;
//					}
//				}
//
//				// SO_illuminance sets the current state to whether the lightsource illuminates the points or not.
//				SO_illuminance( null, null, N, pDefAngle, null );
//
//				PushState();
//				GetCurrentState();
//
//				__iGrid = 0;
//				CqBitVector RS = RunningState();
//				do
//				{
//					if(!__fVarying || RS.Value( __iGrid ) )
//					{
//						// Get the light vector and color from the lightsource.
//						CqVector3D Ln = new CqVector3D();
//						L() .GetVector( Ln, __iGrid );
//						Ln.Unit();
//
//						// Combine the light color into the result
//						CqColor _aq_Result = new CqColor();
//						(Result).GetColor(_aq_Result,__iGrid);
//						CqVector3D _aq_N = new CqVector3D();
//						(N).GetNormal(_aq_N,__iGrid);
//						CqColor colCl = new CqColor();
//						Cl() .GetColor( colCl, __iGrid );
//						(Result).SetColor(_aq_Result .add( colCl .mul( Ln.mul(_aq_N) ) ),__iGrid);
//
//					}
//				}
//				while( ( ++__iGrid < GridSize() ) && __fVarying);
//				PopState();
//				// SO_advance_illuminance returns TRUE if there are any more non ambient lightsources.
//			}
//			while ( SO_advance_illuminance() );
//		}
//		pShader.DeleteTemporaryStorage( __nondiffuse );
//		pShader.DeleteTemporaryStorage( pDefAngle );
//	}
	public void SO_diffuse(IqShaderData N, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_diffuse );

		boolean __fVarying=false;
		int __iGrid;

		// If the illuminance cache is already OK, then we don't need to bother filling in the illuminance parameters.
		if ( !m_IlluminanceCacheValid )
		{
			ValidateIlluminanceCache( null, N, pShader );
		}

		IqShaderData pDefAngle = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_uniform) );
		if ( null == pDefAngle )
			return ;

		pDefAngle.SetFloat( (float)PI/2.0f );

		Result.SetColor( gColBlack );

		__fVarying = true;
		IqShaderData __nondiffuse = null;
		__nondiffuse = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_varying) );

		// SO_init_illuminance returns TRUE if there are any non ambient ligthsources available.
		if ( SO_init_illuminance() )
		{
			CqVector3D Ln = new CqVector3D();
			CqColor _aq_Result = new CqColor();
			CqVector3D _aq_N = new CqVector3D();
			CqColor colCl = new CqColor();
			
			IqShader pLightsource = null;
			do
			{
				// Get the "__nondiffuse" setting from the current lightsource, if specified.
				p_float	__nondiffuse_val = new p_float();
				if ( m_li < m_pAttributes .cLights() )
					pLightsource = m_pAttributes .pLight( m_li ) .pShader();
				if ( pLightsource!=null )
				{
					pLightsource.GetValue( "__nondiffuse", __nondiffuse );
					/// \note: This is OK here, outside the BEGIN_VARYING_SECTION as, varying in terms of lightsources
					/// is not valid.
					if( null != __nondiffuse )
					{
						__nondiffuse.GetFloat( __nondiffuse_val, 0 );
						if( __nondiffuse_val.value != 0.0f )
							continue;
					}
				}

				// SO_illuminance sets the current state to whether the lightsource illuminates the points or not.
				SO_illuminance( null, null, N, pDefAngle, null );

				PushState();
				GetCurrentState();

				__iGrid = 0;
				CqBitVector RS = RunningState();
				do
				{
					if(!__fVarying || RS.Value( __iGrid ) )
					{
						// Get the light vector and color from the lightsource.
						
						L() .GetVector( Ln, __iGrid );
						Ln.Unit();

						// Combine the light color into the result
						(Result).GetColor(_aq_Result,__iGrid);
						(N).GetNormal(_aq_N,__iGrid);
						Cl() .GetColor( colCl, __iGrid );
						(Result).SetColor(_aq_Result .add( colCl .mul( Ln.mul(_aq_N) ) ),__iGrid);

					}
				}
				while( ( ++__iGrid < GridSize() ) && __fVarying);
				PopState();
				// SO_advance_illuminance returns TRUE if there are any more non ambient lightsources.
			}
			while ( SO_advance_illuminance() );
		}
		pShader.DeleteTemporaryStorage( __nondiffuse );
		pShader.DeleteTemporaryStorage( pDefAngle );
	}
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_specular(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_specular(IqShaderData N, IqShaderData V, IqShaderData roughness, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_specular );

		boolean __fVarying=false;
		int __iGrid;

		// If the illuminance cache is already OK, then we don't need to bother filling in the illuminance parameters.
		if ( !m_IlluminanceCacheValid )
		{
			ValidateIlluminanceCache( null, N, pShader );
		}

		IqShaderData pDefAngle = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_uniform) );
		if ( null == pDefAngle )
			return ;

		pDefAngle.SetFloat( (float)PI/2.0f );

		Result.SetColor( gColBlack );
		__fVarying = true;

		IqShaderData __nonspecular = null;
		__nonspecular = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_varying) );

		// SO_init_illuminance returns TRUE if there are any non ambient ligthsources available.
		if ( SO_init_illuminance() )
		{
			IqShader pLightsource = null;
			do
			{
				// Get the "__nonspecular" setting from the current lightsource, if specified.
				p_float	__nonspecular_val = new p_float();
				if ( m_li < m_pAttributes .cLights() )
					pLightsource = m_pAttributes .pLight( m_li ) .pShader();
				if ( pLightsource!=null )
				{
					pLightsource.GetValue( "__nonspecular", __nonspecular );
					/// \note: This is OK here, outside the BEGIN_VARYING_SECTION as, varying in terms of lightsources
					/// is not valid.
					if( null != __nonspecular )
					{
						__nonspecular.GetFloat( __nonspecular_val, 0 );
						if( __nonspecular_val.value != 0.0f )
							continue;
					}
				}

				// SO_illuminance sets the current state to whether the lightsource illuminates the points or not.
				SO_illuminance( null, null, N, pDefAngle, null );

				PushState();
				GetCurrentState();
				__iGrid = 0;
				CqBitVector RS = RunningState();
				do
				{
					if(!__fVarying || RS.Value( __iGrid ) )
					{

						CqVector3D _aq_V = new CqVector3D();
						(V).GetVector(_aq_V,__iGrid);
						// Get the ligth vector and color from the lightsource
						CqVector3D Ln = new CqVector3D();
						L() .GetVector( Ln, __iGrid );
						Ln.Unit();
						CqVector3D	H = Ln .add( _aq_V );
						H.Unit();

						// Combine the color into the result.
						/// \note The (roughness/8) term emulates the BMRT behaviour for prmanspecular.
						CqColor _aq_Result = new CqColor();
						(Result).GetColor(_aq_Result,__iGrid);
						CqVector3D _aq_N = new CqVector3D();
						(N).GetNormal(_aq_N,__iGrid);
						p_float _aq_roughness = new p_float();
						(roughness).GetFloat(_aq_roughness,__iGrid);
						CqColor colCl = new CqColor();
						Cl() .GetColor( colCl, __iGrid );
						(Result).SetColor(_aq_Result .add( colCl .mul( (float)pow( max( 0.0f, _aq_N.mul(H) ), 1.0f / ( _aq_roughness.value / 8.0f ) ) ) ),__iGrid);

					}
				}
				while( ( ++__iGrid < GridSize() ) && __fVarying);
				PopState();
				// SO_advance_illuminance returns TRUE if there are any more non ambient lightsources.
			}
			while ( SO_advance_illuminance() );
		}
		pShader.DeleteTemporaryStorage( __nonspecular );
		pShader.DeleteTemporaryStorage( pDefAngle );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_phong(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_phong(IqShaderData N, IqShaderData V, IqShaderData size, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_phong );

		boolean __fVarying=false;
		int __iGrid;

		IqShaderData pnV = pShader .CreateTemporaryStorage( new EqVariableType(type_vector), new EqVariableClass(class_varying) );
		IqShaderData pnN = pShader .CreateTemporaryStorage( new EqVariableType(type_normal), new EqVariableClass(class_varying) );
		IqShaderData pR  = pShader .CreateTemporaryStorage( new EqVariableType(type_vector), new EqVariableClass(class_varying) );

		/// note: Not happy about this, the shader should take care of this at construction time,
		/// but at the moment, it can't guarantee the validity of the m_u/vGridRes data members.
		pnV.Initialise( uGridRes(), vGridRes() );
		pnN.Initialise( uGridRes(), vGridRes() );
		pR.Initialise( uGridRes(), vGridRes() );

		SO_normalize( V, pnV );
		SO_normalize( N, pnN );

		__fVarying = true;
		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D vecnV = new CqVector3D();
				pnV.GetVector( vecnV, __iGrid );
				pnV.SetVector( vecnV.negative(), __iGrid );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		SO_reflect( pnV, pnN, pR );

		pShader.DeleteTemporaryStorage( pnV );
		pShader.DeleteTemporaryStorage( pnN );

		// If the illuminance cache is already OK, then we don't need to bother filling in the illuminance parameters.
		if ( !m_IlluminanceCacheValid )
		{
			ValidateIlluminanceCache( null, N, pShader );
		}

		IqShaderData pDefAngle = pShader.CreateTemporaryStorage( new EqVariableType(type_float), new EqVariableClass(class_uniform) );
		if ( null == pDefAngle )
			return ;

		pDefAngle.SetFloat( (float)PI/2.0f );

		// Initialise the return value
		Result.SetColor( gColBlack );

		// SO_init_illuminance returns TRUE if there are any non ambient ligthsources available.
		if ( SO_init_illuminance() )
		{
			do
			{
				// SO_illuminance sets the current state to whether the lightsource illuminates the points or not.
				SO_illuminance( null, null, N, pDefAngle, null );

				PushState();
				GetCurrentState();

				__iGrid = 0;
//				CqBitVector RS = RunningState();
				RS = RunningState();
				do
				{
					if(!__fVarying || RS.Value( __iGrid ) )
					{

						// Get the light vector and color from the loght source.
						CqVector3D Ln = new CqVector3D();
						L() .GetVector( Ln, __iGrid );
						Ln.Unit();

						// Now combine the color into the result.
						CqColor _aq_Result = new CqColor();
						(Result).GetColor(_aq_Result,__iGrid);
						CqVector3D vecR = new CqVector3D();
						pR.GetVector( vecR, __iGrid );
						p_float _aq_size = new p_float();
						(size).GetFloat(_aq_size,__iGrid);
						CqColor colCl = new CqColor();
						Cl() .GetColor( colCl, __iGrid );
						(Result).SetColor(_aq_Result.add( colCl.mul( (float)pow( max( 0.0f, vecR.mul(Ln) ), _aq_size.value ) ) ),__iGrid);

					}
				}
				while( ( ++__iGrid < GridSize() ) && __fVarying);

				PopState();
				// SO_advance_illuminance returns TRUE if there are any more non ambient lightsources.
			}
			while ( SO_advance_illuminance() );
		}
		pShader.DeleteTemporaryStorage( pDefAngle );
		pShader.DeleteTemporaryStorage( pR );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_trace(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_trace(IqShaderData P, IqShaderData R, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_trace );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(P).Class().getValue()==class_varying||__fVarying;
		__fVarying=(R).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetColor(new CqColor( 0, 0, 0 ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ftexture1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ftexture1(IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ftexture1 );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ).GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					float swidth = 0.0f, twidth = 0.0f;
					if ( fdu.value != 0.0f && fdv.value != 0.0f )
					{
						float dsdu = SO_DuType_TqFloat( s(), __iGrid, this, Deffloat );
						swidth = abs( dsdu * fdu.value );
						float dtdu = SO_DuType_TqFloat( t(), __iGrid, this, Deffloat );
						twidth = abs( dtdu * fdu.value );

						float dsdv = SO_DvType_TqFloat( s(), __iGrid, this, Deffloat );
						swidth += abs( dsdv * fdv.value );
						float dtdv = SO_DvType_TqFloat( t(), __iGrid, this, Deffloat );
						twidth += abs( dtdv * fdv.value );
					}
					else
					{
						swidth = (float)( 1.0 / pTMap.XRes() );
						twidth = (float)( 1.0 / pTMap.YRes() );
					}

					// Sample the texture.
					p_float fs = new p_float(), ft = new p_float();
					s() .GetFloat( fs, __iGrid );
					t() .GetFloat( ft, __iGrid );
					pTMap.SampleMap( fs.value, ft.value, swidth, twidth, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan >= val.size() )
						(Result).SetFloat(fill.value,__iGrid);
					else
						(Result).SetFloat(val.valueAt( (int)( fchan ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ftexture2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ftexture2(IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ftexture2 );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					float swidth = 0.0f, twidth = 0.0f;
					if ( fdu.value != 0.0f && fdv.value != 0.0f )
					{
						float dsdu = SO_DuType_TqFloat( s, __iGrid, this, Deffloat );
						swidth = abs( dsdu * fdu.value );
						float dtdu = SO_DuType_TqFloat( t, __iGrid, this, Deffloat );
						twidth = abs( dtdu * fdu.value );

						float dsdv = SO_DvType_TqFloat( s, __iGrid, this, Deffloat );
						swidth += abs( dsdv * fdv.value );
						float dtdv = SO_DvType_TqFloat( t, __iGrid, this, Deffloat );
						twidth += abs( dtdv * fdv.value );
					}
					else
					{
						swidth = (float)( 1.0 / pTMap.XRes() );
						twidth = (float)( 1.0 / pTMap.YRes() );
					}

					// Sample the texture.
					p_float _aq_s = new p_float();
					(s).GetFloat(_aq_s,__iGrid);
					p_float _aq_t = new p_float();
					(t).GetFloat(_aq_t,__iGrid);
					pTMap.SampleMap( _aq_s.value, _aq_t.value, swidth, twidth, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan >= val.size() )
						(Result).SetFloat(fill.value,__iGrid);
					else
						(Result).SetFloat(val.valueAt( (int)( fchan ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ftexture3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ftexture3(IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ftexture3 );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{

					// Sample the texture.
					p_float _aq_s1 = new p_float();
					(s1).GetFloat(_aq_s1,__iGrid);
					p_float _aq_t1 = new p_float();
					(t1).GetFloat(_aq_t1,__iGrid);
					p_float _aq_s2 = new p_float();
					(s2).GetFloat(_aq_s2,__iGrid);
					p_float _aq_t2 = new p_float();
					(t2).GetFloat(_aq_t2,__iGrid);
					p_float _aq_s3 = new p_float();
					(s3).GetFloat(_aq_s3,__iGrid);
					p_float _aq_t3 = new p_float();
					(t3).GetFloat(_aq_t3,__iGrid);
					p_float _aq_s4 = new p_float();
					(s4).GetFloat(_aq_s4,__iGrid);
					p_float _aq_t4 = new p_float();
					(t4).GetFloat(_aq_t4,__iGrid);
					pTMap.SampleMap( _aq_s1.value, _aq_t1.value, _aq_s2.value, _aq_t2.value, _aq_s3.value, _aq_t3.value, _aq_s4.value, _aq_t4.value, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan >= val.size() )
						(Result).SetFloat(fill.value,__iGrid);
					else
						(Result).SetFloat(val.valueAt( (int)( fchan ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ctexture1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ctexture1(IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ctexture1 );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					float swidth = 0.0f, twidth = 0.0f;
					if ( fdu.value != 0.0f && fdv.value != 0.0f )
					{
						float dsdu = SO_DuType_TqFloat( s(), __iGrid, this, Deffloat );
						swidth = abs( dsdu * fdu.value );
						float dsdv = SO_DvType_TqFloat( s(), __iGrid, this, Deffloat );
						swidth += abs( dsdv * fdv.value );

						float dtdu = SO_DuType_TqFloat( t(), __iGrid, this, Deffloat );
						twidth = abs( dtdu * fdu.value );
						float dtdv = SO_DvType_TqFloat( t(), __iGrid, this, Deffloat );
						twidth += abs( dtdv * fdv.value );
					}
					else
					{
						swidth = (float)( 1.0 / pTMap.XRes() );
						twidth = (float)( 1.0 / pTMap.YRes() );
					}

					// Sample the texture.
					p_float fs = new p_float(), ft = new p_float();
					s() .GetFloat( fs, __iGrid );
					t() .GetFloat( ft, __iGrid );
					pTMap.SampleMap( fs.value, ft.value, swidth, twidth, val );

					// Grab the appropriate channel.
					
					float fchan = _aq_channel.value;
					CqColor colResult = new CqColor();
					colResult.SetfRed( (fchan >= val.size())? fill.value : (val.valueAt( (int)( fchan ) ) )  );
					colResult.SetfGreen( ((fchan + 1) >= val.size())? fill.value : val.valueAt( (int)( fchan+1 )  ) );
					colResult.SetfBlue( ((fchan + 2) >= val.size())? fill.value : val.valueAt( (int)( fchan+2 ) )  );

					(Result).SetColor(colResult,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetColor(new CqColor( 0, 0, 0 ),__iGrid);	// Default, no color
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ctexture2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ctexture2(IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ctexture2 );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					float swidth = 0.0f, twidth = 0.0f;
					if ( fdu.value != 0.0f && fdv.value != 0.0f )
					{
						float dsdu = SO_DuType_TqFloat( s, __iGrid, this, Deffloat );
						swidth = abs( dsdu * fdu.value );
						float dsdv = SO_DvType_TqFloat( s, __iGrid, this, Deffloat );
						swidth += abs( dsdv * fdv.value );

						float dtdu = SO_DuType_TqFloat( t, __iGrid, this, Deffloat );
						twidth = abs( dtdu * fdu.value );
						float dtdv = SO_DvType_TqFloat( t, __iGrid, this, Deffloat );
						twidth += abs( dtdv * fdv.value );
					}
					else
					{
						swidth = (float)( 1.0 / pTMap.XRes() );
						twidth = (float)( 1.0 / pTMap.YRes() );
					}

					// Sample the texture.
					p_float _aq_s = new p_float();
					(s).GetFloat(_aq_s,__iGrid);
					p_float _aq_t = new p_float();
					(t).GetFloat(_aq_t,__iGrid);
					pTMap.SampleMap( _aq_s.value, _aq_t.value, swidth, twidth, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					CqColor colResult = new CqColor();
					colResult.SetfRed( (fchan >= val.size())? fill.value : val.valueAt( (int)( fchan ) ) );
					colResult.SetfGreen( ((fchan + 1) >= val.size())? fill.value : val.valueAt( (int)( fchan+1 ) ) );
					colResult.SetfBlue( ((fchan + 2) >= val.size())? fill.value : val.valueAt( (int)( fchan+2 ) ) );

					(Result).SetColor(colResult,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetColor(new CqColor( 0, 0, 0 ),__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ctexture3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_ctexture3(IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_ctexture3 );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetTextureMap( _aq_name.value );



		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					// Sample the texture.
					p_float _aq_s1 = new p_float();
					(s1).GetFloat(_aq_s1,__iGrid);
					p_float _aq_t1 = new p_float();
					(t1).GetFloat(_aq_t1,__iGrid);
					p_float _aq_s2 = new p_float();
					(s2).GetFloat(_aq_s2,__iGrid);
					p_float _aq_t2 = new p_float();
					(t2).GetFloat(_aq_t2,__iGrid);
					p_float _aq_s3 = new p_float();
					(s3).GetFloat(_aq_s3,__iGrid);
					p_float _aq_t3 = new p_float();
					(t3).GetFloat(_aq_t3,__iGrid);
					p_float _aq_s4 = new p_float();
					(s4).GetFloat(_aq_s4,__iGrid);
					p_float _aq_t4 = new p_float();
					(t4).GetFloat(_aq_t4,__iGrid);
					pTMap.SampleMap( _aq_s1.value, _aq_t1.value, _aq_s2.value, _aq_t2.value, _aq_s3.value, _aq_t3.value, _aq_s4.value, _aq_t4.value, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					CqColor colResult = new CqColor();
					colResult.SetfRed( (fchan >= val.size())? fill.value : val.valueAt( (int)( fchan ) ) );
					colResult.SetfGreen( ((fchan + 1) >= val.size())? fill.value : val.valueAt( (int)( fchan+1 ) ) );
					colResult.SetfBlue( ((fchan + 2) >= val.size())? fill.value : val.valueAt( (int)( fchan+2 ) ) );

					(Result).SetColor(colResult,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetColor(new CqColor( 0, 0, 0 ),__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fenvironment2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_fenvironment2(IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_fenvironment2 );

		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetEnvironmentMap( _aq_name.value );

		// Try with LatLong map file
		if ( pTMap == null )
		{
			pTMap = QGetRenderContextI() .GetLatLongMap( _aq_name.value );
		}
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}


		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			pTMap.PrepareSampleOptions( paramMap );
			Valarray val = new Valarray();

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D swidth = new CqVector3D(0.0f), twidth = new CqVector3D(0.0f);
					if ( fdu.value != 0.0f )
					{
						CqVector3D dRdu = SO_DuType_CqVector3D( R, __iGrid, this, Defvec );
						swidth = dRdu.mul( fdu.value );
					}
					if ( fdv.value != 0.0f )
					{
//						CqVector3D dRdv = SO_DuType_CqVector3D( R, __iGrid, this, Defvec );
						CqVector3D dRdv = SO_DvType_CqVector3D( R, __iGrid, this, Defvec );
						twidth = dRdv.mul( fdv.value );
					}
					else
					{
						swidth = new CqVector3D( (float)(1.0 / pTMap.XRes()) );
						twidth = new CqVector3D( (float)(1.0 / pTMap.YRes()) );
					}

					// Sample the texture.
					CqVector3D _aq_R = new CqVector3D();
					(R).GetVector(_aq_R,__iGrid);
					pTMap.SampleMap( _aq_R, swidth, twidth, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan >= val.size() )
						(Result).SetFloat(fill.value,__iGrid);
					else
						(Result).SetFloat(val.valueAt( (int)( fchan ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fenvironment3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_fenvironment3(IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_fenvironment3 );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetEnvironmentMap( _aq_name.value );
		// Try with LatLong map file
		if ( pTMap == null )
		{
			pTMap = QGetRenderContextI() .GetLatLongMap( _aq_name.value );
		}


		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					// Sample the texture.
					CqVector3D _aq_R1 = new CqVector3D();
					(R1).GetVector(_aq_R1,__iGrid);
					CqVector3D _aq_R2 = new CqVector3D();
					(R2).GetVector(_aq_R2,__iGrid);
					CqVector3D _aq_R3 = new CqVector3D(9);
					(R3).GetVector(_aq_R3,__iGrid);
					CqVector3D _aq_R4 = new CqVector3D();
					(R4).GetVector(_aq_R4,__iGrid);
					pTMap.SampleMap( _aq_R1, _aq_R2, _aq_R3, _aq_R4, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan >= val.size() )
						(Result).SetFloat(fill.value,__iGrid);
					else
						(Result).SetFloat(val.valueAt( (int)( fchan ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cenvironment2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_cenvironment2(IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_cenvironment2 );

		CqVector3D Defvec = new CqVector3D( 0.0f, 0.0f, 0.0f );
		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetEnvironmentMap( _aq_name.value );
		// Try with LatLong map file
		if ( pTMap == null )
		{
			pTMap = QGetRenderContextI() .GetLatLongMap( _aq_name.value );
		}
		p_float fdu = new p_float(0.0f), fdv = new p_float(0.0f);
		if ( m_pAttributes!=null )
		{
			du() .GetFloat( fdu );
			dv() .GetFloat( fdv );
		}


		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D swidth = new CqVector3D(0.0f), twidth = new CqVector3D(0.0f);
					if ( fdu.value != 0.0f )
					{
						CqVector3D dRdu = SO_DuType_CqVector3D( R, __iGrid, this, Defvec );
						swidth = dRdu.mul( fdu.value );
					}
					if ( fdv.value != 0.0f )
					{
//						CqVector3D dRdv = SO_DuType_CqVector3D( R, __iGrid, this, Defvec ); 2006/02/07 nttdata C
						CqVector3D dRdv = SO_DvType_CqVector3D( R, __iGrid, this, Defvec );
						twidth = dRdv.mul( fdv.value );
					}
					else
					{
						swidth = new CqVector3D( (float)(1.0 / pTMap.XRes()) );
						twidth = new CqVector3D( (float)(1.0 / pTMap.YRes()) );
					}

					// Sample the texture.
					CqVector3D _aq_R = new CqVector3D();
					(R).GetVector(_aq_R,__iGrid);
					pTMap.SampleMap( _aq_R, swidth, twidth, val );


					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan + 2 >= val.size() )
						(Result).SetColor(new CqColor( fill.value, fill.value, fill.value ),__iGrid);
					else
						(Result).SetColor(new CqColor( val.valueAt( (int)( fchan ) ), val.valueAt( (int)( fchan ) + 1 ), val.valueAt( (int)( fchan ) + 2 ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetColor(new CqColor( 0.0f, 0.0f, 0.0f ),__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cenvironment3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_cenvironment3(IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_cenvironment3 );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		p_float fill = new p_float(0.0f);
		if ( paramMap.containsKey( "fill" ) )
			paramMap.get( "fill" ) .GetFloat( fill );

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pTMap = QGetRenderContextI() .GetEnvironmentMap( _aq_name.value );
		// Try with LatLong map file
		if ( pTMap == null )
		{
			pTMap = QGetRenderContextI() .GetLatLongMap( _aq_name.value );
		}
		__iGrid = 0;

		__fVarying = true;
		if ( pTMap != null && pTMap.IsValid() )
		{
			Valarray val = new Valarray();
			pTMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					// Sample the texture.
					CqVector3D _aq_R1 = new CqVector3D();
					(R1).GetVector(_aq_R1,__iGrid);
					CqVector3D _aq_R2 = new CqVector3D();
					(R2).GetVector(_aq_R2,__iGrid);
					CqVector3D _aq_R3 = new CqVector3D();
					(R3).GetVector(_aq_R3,__iGrid);
					CqVector3D _aq_R4 = new CqVector3D();
					(R4).GetVector(_aq_R4,__iGrid);
					pTMap.SampleMap( _aq_R1, _aq_R2, _aq_R3, _aq_R4, val );

					// Grab the appropriate channel.
					float fchan = _aq_channel.value;
					if ( fchan + 2 >= val.size() )
						(Result).SetColor(new CqColor( fill.value, fill.value, fill.value ),__iGrid);
					else
						(Result).SetColor(new CqColor( val.valueAt( (int)( fchan ) ), val.valueAt( (int)( fchan ) + 1 ), val.valueAt( (int)( fchan ) + 2 ) ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetColor(new CqColor( 0.0f, 0.0f, 0.0f ),__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bump1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bump1(IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		boolean __fVarying=false;
		int __iGrid;

		__fVarying = true;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(new CqVector3D( 0, 0, 0 ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bump2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bump2(IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		boolean __fVarying=false;
		int __iGrid;

		__fVarying = true;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(new CqVector3D( 0, 0, 0 ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bump3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bump3(IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		boolean __fVarying=false;
		int __iGrid;

		__fVarying = true;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetPoint(new CqVector3D( 0, 0, 0 ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_shadow(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_shadow(IqShaderData name, IqShaderData channel, IqShaderData P, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_shadow );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pMap = QGetRenderContextI() .GetShadowMap( _aq_name.value );


		__fVarying = true;
		if ( pMap != null && pMap.IsValid() )
		{
			Valarray fv = new Valarray();
			pMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D swidth = new CqVector3D(0.0f), twidth = new CqVector3D(0.0f);
					
					CqVector3D _aq_P = new CqVector3D();
					(P).GetPoint(_aq_P,__iGrid);

					pMap.SampleMap( _aq_P, swidth, twidth, fv, 0 );
					(Result).SetFloat(fv.valueAt( 0 ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_shadow1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_shadow1(IqShaderData name, IqShaderData channel, IqShaderData P1, IqShaderData P2, IqShaderData P3, IqShaderData P4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_shadow1 );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		p_float _aq_channel = new p_float();
		(channel).GetFloat(_aq_channel,__iGrid);
		IqTextureMap pMap = QGetRenderContextI() .GetShadowMap( _aq_name.value );


		__fVarying = true;
		if ( pMap != null && pMap.IsValid() )
		{
			Valarray fv = new Valarray();
			pMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					CqVector3D _aq_P1 = new CqVector3D();
					(P1).GetPoint(_aq_P1,__iGrid);
					CqVector3D _aq_P2 = new CqVector3D();
					(P2).GetPoint(_aq_P2,__iGrid);
					CqVector3D _aq_P3 = new CqVector3D();
					(P3).GetPoint(_aq_P3,__iGrid);
					CqVector3D _aq_P4 = new CqVector3D();
					(P4).GetPoint(_aq_P4,__iGrid);
					pMap.SampleMap( _aq_P1, _aq_P2, _aq_P3, _aq_P4, fv, 0 );
					(Result).SetFloat(fv.valueAt( 0 ),__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_illuminance(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_illuminance(IqShaderData Category, IqShaderData P, IqShader pShader) {

		STATS_INC( SHD_so_illuminance );

		SO_illuminance( Category, P, null, null );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_illuminance(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
//	public void SO_illuminance(IqShaderData Category, IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShader pShader) {
//
//		STATS_INC( SHD_so_illuminance );
//
//		boolean __fVarying=false;
//		int __iGrid;
//
//		__iGrid = 0;
//		p_String cat = new p_String( "" );
//		if ( null != Category )
//			Category.GetString( cat );
//
//
//		__fVarying = true;
//
//		// Fill in the lightsource information, and transfer the results to the shader variables,
//		if ( m_pAttributes != null )
//		{
//			IqLightsource lp = m_pAttributes .pLight( m_li );
//
//			if ( null != Axis )
//				__fVarying=(Axis).Class().getValue()==class_varying||__fVarying;
//			if ( null != Angle )
//				__fVarying=(Angle).Class().getValue()==class_varying||__fVarying;
//
//			boolean exec = true;
//
//			if( cat.value.length()!=0 )
//			{
//
//				boolean exclude = false;
//				p_String lightcategories = new p_String();
//				String catname;
//
//
////				if( cat.find( "-" ) == 0 )
//				if( cat.value.startsWith("-") )
//				{
//					exclude = true;
//					catname = cat.value.substring( 1, cat.value.length() );
//				}
//				else
//				{
//					catname = cat.value;
//				}
//
//				IqShaderData pcats = lp.pShader().FindArgument("__category");
//				if( pcats!=null )
//				{
//					pcats.GetString( lightcategories );
//
//					exec = false;
//					// While no matching category has been found...
//					int tokenpos = 0, tokenend;
//					while( true )
//					{
//						tokenend = lightcategories.value.indexOf(",", tokenpos);
//						if( tokenend == -1 )
//							break;
//						String token = lightcategories.value.substring( tokenpos, tokenend );
//						if( catname.compareTo( token ) == 0 )
//						{
//							if( !exclude )
//							{
//								exec = true;
//								break;
//							}
//						}
////						else
//							tokenpos = tokenend+1;
//					}
//				}
//			}
//
//			if( exec )
//			{
//				__iGrid = 0;
//				CqBitVector RS = RunningState();
//				do
//				{
//					if(!__fVarying || RS.Value( __iGrid ) )
//					{
//
//						CqVector3D Ln = new CqVector3D();
//						lp.L() .GetVector( Ln, __iGrid );
////						Ln = -Ln;
//						Ln.negate();
//
//						// Store them locally on the surface.
//						L() .SetVector( Ln, __iGrid );
//						CqColor colCl = new CqColor();
//						lp.Cl() .GetColor( colCl, __iGrid );
//						Cl() .SetColor( colCl, __iGrid );
//
//						// Check if its within the cone.
//						Ln.Unit();
//						CqVector3D vecAxis = new CqVector3D( 0, 1, 0 );
//						if ( null != Axis )
//							Axis.GetVector( vecAxis, __iGrid );
//						p_float fAngle = new p_float((float)PI);
//						if ( null != Angle )
//							Angle.GetFloat( fAngle, __iGrid );
//
//						float cosangle = Ln .mul(vecAxis);
//						cosangle = CLAMP( cosangle, -1, 1 );
//						if ( acos( cosangle ) > fAngle.value )
//							m_CurrentState.SetValue( __iGrid, false );
//						else
//							m_CurrentState.SetValue( __iGrid, true );
//					}
//				}
//				while( ( ++__iGrid < GridSize() ) && __fVarying);
//			}
//		}
//	}
	public void SO_illuminance(IqShaderData Category, IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShader pShader) {

		STATS_INC( SHD_so_illuminance );

		boolean __fVarying=false;
		int __iGrid;

		__iGrid = 0;
		p_String cat = new p_String( "" );
		if ( null != Category )
			Category.GetString( cat );


		__fVarying = true;

		// Fill in the lightsource information, and transfer the results to the shader variables,
		if ( m_pAttributes != null )
		{
			IqLightsource lp = m_pAttributes .pLight( m_li );

			if ( null != Axis )
				__fVarying=(Axis).Class().getValue()==class_varying||__fVarying;
			if ( null != Angle )
				__fVarying=(Angle).Class().getValue()==class_varying||__fVarying;

			boolean exec = true;

			if( cat.value.length()!=0 )
			{

				boolean exclude = false;
				p_String lightcategories = new p_String();
				String catname;


//				if( cat.find( "-" ) == 0 )
				if( cat.value.startsWith("-") )
				{
					exclude = true;
					catname = cat.value.substring( 1, cat.value.length() );
				}
				else
				{
					catname = cat.value;
				}

				IqShaderData pcats = lp.pShader().FindArgument("__category");
				if( pcats!=null )
				{
					pcats.GetString( lightcategories );

					exec = false;
					// While no matching category has been found...
					int tokenpos = 0, tokenend;
					while( true )
					{
						tokenend = lightcategories.value.indexOf(",", tokenpos);
						if( tokenend == -1 )
							break;
						String token = lightcategories.value.substring( tokenpos, tokenend );
						if( catname.compareTo( token ) == 0 )
						{
							if( !exclude )
							{
								exec = true;
								break;
							}
						}
//						else
							tokenpos = tokenend+1;
					}
				}
			}

			if( exec )
			{
				CqVector3D Ln = new CqVector3D();
				CqColor colCl = new CqColor();
				CqVector3D vecAxis = new CqVector3D( 0, 1, 0 );
				p_float fAngle = new p_float((float)PI);
				
				
				__iGrid = 0;
				CqBitVector RS = RunningState();
				do
				{
					if(!__fVarying || RS.Value( __iGrid ) )
					{

						
						lp.L() .GetVector( Ln, __iGrid );
//						Ln = -Ln;
//						Ln.negate();
						Ln.x = -Ln.x;
						Ln.y = -Ln.y;
						Ln.z = -Ln.z;
						

						// Store them locally on the surface.
						L() .SetVector( Ln, __iGrid );
						
						lp.Cl() .GetColor( colCl, __iGrid );
						Cl() .SetColor( colCl, __iGrid );

						// Check if its within the cone.
						Ln.Unit();
						
						if ( null != Axis )
							Axis.GetVector( vecAxis, __iGrid );
						
						if ( null != Angle )
							Angle.GetFloat( fAngle, __iGrid );

						float cosangle = Ln .mul(vecAxis);
						cosangle = CLAMP( cosangle, -1, 1 );
						if ( acos( cosangle ) > fAngle.value )
							m_CurrentState.SetValue( __iGrid, false );
						else
							m_CurrentState.SetValue( __iGrid, true );
					}
				}
				while( ( ++__iGrid < GridSize() ) && __fVarying);
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_illuminate(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_illuminate(IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShader pShader) {

		STATS_INC( SHD_so_illuminate );

		boolean __fVarying=false;
		int __iGrid;

		boolean res = true;
		if ( m_Illuminate > 0 )
			res = false;

		__fVarying = true;
		if ( res )
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					// Get the point being lit and set the ligth vector.
					CqVector3D _aq_P = new CqVector3D();
					(P).GetPoint(_aq_P,__iGrid);
					CqVector3D vecPs = new CqVector3D();
					Ps() .GetPoint( vecPs, __iGrid );
					L() .SetVector( vecPs.sub( _aq_P ), __iGrid );

					// Check if its within the cone.
					CqVector3D Ln = new CqVector3D();
					L() .GetVector( Ln, __iGrid );
					Ln.Unit();

					CqVector3D vecAxis = new CqVector3D( 0.0f, 1.0f, 0.0f );
					if ( null != Axis )
						Axis.GetVector( vecAxis, __iGrid );
					p_float fAngle = new p_float((float)PI);
					if ( null != Angle )
						Angle.GetFloat( fAngle, __iGrid );
					float cosangle = Ln .mul( vecAxis );
					cosangle = CLAMP( cosangle, -1, 1 );
					if ( acos( cosangle ) > fAngle.value )
					{
						// Make sure we set the light color to zero in the areas that won't be lit.
						Cl() .SetColor( new CqColor( 0, 0, 0 ), __iGrid );
						m_CurrentState.SetValue( __iGrid, false );
					}
					else
						m_CurrentState.SetValue( __iGrid, true );
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}

		m_Illuminate++;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_illuminate(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_illuminate(IqShaderData P, IqShader pShader) {

		STATS_INC( SHD_so_illuminate );

		SO_illuminate( P, null, null, pShader );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_solar(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_solar(IqShaderData Axis, IqShaderData Angle, IqShader pShader) {

		// TODO: Check light cone, and exclude points outside.
		STATS_INC( SHD_so_solar );

		boolean __fVarying=false;
		int __iGrid;

		boolean res = true;
		if ( m_Illuminate > 0 )
			res = false;

		__fVarying = true;
		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				if ( res )
				{
					CqVector3D vecAxis = new CqVector3D( 0.0f, 1.0f, 0.0f );
					if ( null != Axis )
						Axis.GetVector( vecAxis, __iGrid );
					L() .SetVector( vecAxis, __iGrid );
					m_CurrentState.SetValue( __iGrid, true );
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		m_Illuminate++;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_solar(net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_solar(IqShader pShader) {

		STATS_INC( SHD_so_solar );

		SO_solar( null, null, pShader );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_printf(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_printf(IqShaderData str, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_printf );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		__fVarying=(str).Class().getValue()==class_varying||__fVarying;
		int ii;
		for ( ii = 0; ii < cParams; ii++ )
		{
			__fVarying=(apParams[ ii ]).Class().getValue()==class_varying||__fVarying;
		}

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_String _aq_str = new p_String("");
				(str).GetString(_aq_str,__iGrid);
				String strA = SO_sprintf( _aq_str.value, cParams, apParams, __iGrid );
				QGetRenderContextI() .PrintString( strA );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_format(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_format(IqShaderData str, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_format );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(str).Class().getValue()==class_varying||__fVarying;
		int ii;
		for ( ii = 0; ii < cParams; ii++ )
		{
			__fVarying=(apParams[ ii ]).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_String _aq_str = new p_String("");
				(str).GetString(_aq_str,__iGrid);
				String strA = SO_sprintf( _aq_str.value, cParams, apParams, __iGrid );
				(Result).SetString(strA,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_concat(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_concat(IqShaderData stra, IqShaderData strb, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_concat );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(stra).Class().getValue()==class_varying||__fVarying;
		__fVarying=(strb).Class().getValue()==class_varying||__fVarying;
		int ii;
		for ( ii = 0; ii < cParams; ii++ )
		{
			__fVarying=(apParams[ ii ]).Class().getValue()==class_varying||__fVarying;
		}
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_String _aq_stra = new p_String();
				(stra).GetString(_aq_stra,__iGrid);
				String strRes = _aq_stra.value;
				p_String _aq_strb = new p_String();
				(strb).GetString(_aq_strb,__iGrid);
				strRes += _aq_strb.value;
				for ( ii = 0; ii < cParams; ii++ )
				{
					p_String sn = new p_String();
					apParams[ ii ] .GetString( sn, __iGrid );
					strRes += sn.value;
				}
				(Result).SetString(strRes,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_atmosphere(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_atmosphere(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_atmosphere );

//		boolean __fVarying=false;
		int __iGrid;

		IqShader pAtmosphere = null;

		if ( null != m_pAttributes && (m_pAttributes.pshadAtmosphere(QGetRenderContextI().Time())!=null) )
			pAtmosphere = m_pAttributes .pshadAtmosphere(QGetRenderContextI().Time());

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		if ( pAtmosphere!=null )
			Result.SetValue( pAtmosphere.GetValue( _aq_name.value, pV ) ? 1.0f : 0.0f, 0 );
		else
			Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_displacement(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_displacement(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_displacement );

//		boolean __fVarying=false;
		int __iGrid;

		IqShader pDisplacement = null;

		if ( null != m_pAttributes && (m_pAttributes .pshadDisplacement(QGetRenderContextI().Time())!=null) )
			pDisplacement = m_pAttributes .pshadDisplacement(QGetRenderContextI().Time());

		__iGrid = 0;
		p_String _aq_name = new p_String("");
		(name).GetString(_aq_name,__iGrid);
		if ( pDisplacement!=null )
			Result.SetValue( pDisplacement.GetValue( _aq_name.value, pV ) ? 1.0f : 0.0f, 0 );
		else
			Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_lightsource(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_lightsource(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_lightsource );

//		boolean __fVarying=false;
		int __iGrid;

		// This should only be called within an Illuminance construct, so m_li should be valid.
		IqShader pLightsource = null;

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		if ( m_li < m_pAttributes .cLights() )
			pLightsource = m_pAttributes .pLight( m_li ) .pShader();
		if ( pLightsource!=null )
			Result.SetValue( pLightsource.GetValue( _aq_name.value, pV ) ? 1.0f : 0.0f, 0 );
		else
			Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_surface(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_surface(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_surface );

//		boolean __fVarying=false;
		int __iGrid;

		IqShader pSurface = null;

		if ( GetCurrentSurface()!=null &&
		        null != GetCurrentSurface().pAttributes() &&
		        GetCurrentSurface().pAttributes() .pshadSurface(QGetRenderContextI().Time())!=null )
			pSurface = GetCurrentSurface().pAttributes() .pshadSurface(QGetRenderContextI().Time());

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		if ( pSurface!=null )
			Result.SetValue( pSurface.GetValue( _aq_name.value, pV ) ? 1.0f : 0.0f, 0 );
		else
			Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_attribute(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_attribute(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_attribute );

//		boolean __fVarying=false;
		int __iGrid;

		//Find out if it is a specific attribute request
		__iGrid = 0;
		p_String p_aq_name = new p_String();
		(name).GetString(p_aq_name,__iGrid);
		String _aq_name = p_aq_name.value;
		float Ret = 0.0f;

		if ( _aq_name.compareTo( "ShadingRate" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float )
			{
				pV.SetFloat( m_pAttributes .GetFloatAttribute( "System", "ShadingRate" ) [ 0 ] );
				Ret = 1.0f;
			}
		}
		else if ( _aq_name.compareTo( "Sides" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float )
			{
				pV.SetFloat( m_pAttributes .GetIntegerAttribute( "System", "Sides" ) [ 0 ] );
				Ret = 1.0f;
			}
		}
		else if ( _aq_name.compareTo( "Matte" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float )
			{
				pV.SetFloat( m_pAttributes .GetIntegerAttribute( "System", "Matte" ) [ 0 ] );
				Ret = 1.0f;
			}
		}
		else
		{
			int iColon = _aq_name.indexOf( ':' );
			if ( iColon >= 0 )
			{
//				CqString strParam = _aq_name.substr( iColon + 1, _aq_name.size() - iColon - 1 );
				String strParam = _aq_name.substring( iColon + 1, _aq_name.length() );
				_aq_name = _aq_name.substring( 0, iColon );
				//const CqParameter* pParam = m_pAttributes .pParameter( STRING( name ), strParam );

				//TODO:attributefloatłȂꍇŏ̃p^[ɂ͂Ă܂AClassCastExceptionԁBCłlBnttdata
				
				Ret = 1.0f;
				if ( null != pAttributes() .GetFloatAttribute( _aq_name, strParam ) )
					pV.SetFloat( pAttributes() .GetFloatAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetIntegerAttribute( _aq_name, strParam ) )
					pV.SetFloat( pAttributes() .GetIntegerAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetStringAttribute( _aq_name, strParam ) )
					pV.SetString( pAttributes() .GetStringAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetPointAttribute( _aq_name, strParam ) )
					pV.SetPoint( pAttributes() .GetPointAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetVectorAttribute( _aq_name, strParam ) )
					pV.SetVector( pAttributes() .GetVectorAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetNormalAttribute( _aq_name, strParam ) )
					pV.SetNormal( pAttributes() .GetNormalAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetColorAttribute( _aq_name, strParam ) )
					pV.SetColor( pAttributes() .GetColorAttribute( _aq_name, strParam ) [ 0 ] );
				else if ( null != pAttributes() .GetMatrixAttribute( _aq_name, strParam ) )
					pV.SetMatrix( pAttributes() .GetMatrixAttribute( _aq_name, strParam ) [ 0 ] );
				else
					Ret = 0.0f;
			}
		}
		Result.SetValue( Ret, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_option(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_option(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_option );

//		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		__iGrid = 0;
		//Find out if it is a specific option request
		p_String p_aq_name = new p_String();
		(name).GetString(p_aq_name,__iGrid);
		String _aq_name = p_aq_name.value;
		float Ret = 0.0f;

		if ( _aq_name.compareTo( "Format" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 3 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( QGetRenderContextI() .GetIntegerOption( "System", "Resolution" ) [ 0 ] ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( QGetRenderContextI() .GetIntegerOption( "System", "Resolution" ) [ 1 ] ) );
					pV.ArrayEntry( 2 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "PixelAspectRatio" ) [ 0 ] ) );//XXX: [0]ł͂Ȃ̂H nttdata
					Ret = 1.0f;
				}
			}
		}
		else if ( _aq_name.compareTo( "CropWindow" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 4 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "CropWindow" ) [ 0 ] ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "CropWindow" ) [ 1 ] ) );
					pV.ArrayEntry( 2 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "CropWindow" ) [ 2 ] ) );
					pV.ArrayEntry( 3 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "CropWindow" ) [ 3 ] ) );
					Ret = 1.0f;
				}
			}
		}
		else if ( _aq_name.compareTo( "FrameAspectRatio" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float )
			{
				pV.SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "FrameAspectRatio" ) [ 0 ] ) );
				Ret = 1.0f;
			}
		}
		else if ( _aq_name.compareTo( "DepthOfField" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 3 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "DepthOfField" ) [ 0 ] ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "DepthOfField" ) [ 1 ] ) );
					pV.ArrayEntry( 2 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "DepthOfField" ) [ 2 ] ) );
					Ret = 1.0f;
				}
			}
		}
		else if ( _aq_name.compareTo( "Shutter" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 2 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "Shutter" ) [ 0 ] ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "Shutter" ) [ 1 ] ) );
					Ret = 1.0f;
				}
			}
		}
		else if ( _aq_name.compareTo( "Clipping" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 2 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "Clipping" ) [ 0 ] ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( QGetRenderContextI() .GetFloatOption( "System", "Clipping" ) [ 1 ] ) );
					Ret = 1.0f;
				}
			}
		}
		else
		{
			String strName = _aq_name;
			int iColon = strName.indexOf( ':' );
			if ( iColon >= 0 )
			{
//				String strParam = strName.substring( iColon + 1, strName.length() - iColon - 1 );
				String strParam = strName.substring( iColon + 1, strName.length() );
				strName = strName.substring( 0, iColon );
				//const CqParameter* pParam = m_pAttributes .pParameter( strName, strParam );

				//TODO:optionfloatłȂꍇŏ̃p^[ɂ͂Ă܂AClassCastExceptionԁBCłlBnttdata

				Ret = 1.0f;
				if ( null != QGetRenderContextI() .GetStringOption( strName, strParam ) )
					pV.SetString( QGetRenderContextI() .GetStringOption( strName, strParam ) [ 0 ] );
				else if ( null != QGetRenderContextI() .GetIntegerOption( strName, strParam ) )
					pV.SetFloat( QGetRenderContextI() .GetIntegerOption( strName, strParam ) [ 0 ] );

				else if ( null != QGetRenderContextI() .GetPointOption( strName, strParam ) )
					pV.SetPoint( QGetRenderContextI() .GetPointOption( strName, strParam ) [ 0 ] );

				else if ( null != QGetRenderContextI() .GetColorOption( strName, strParam ) )
					pV.SetColor( QGetRenderContextI() .GetColorOption( strName, strParam ) [ 0 ] );
				else if ( null != QGetRenderContextI() .GetFloatOption( strName, strParam ) )
					pV.SetFloat( QGetRenderContextI() .GetFloatOption( strName, strParam ) [ 0 ] );
				/* did not deal with Vector, Normal and Matrix yet */
				else
					Ret = 0.0f;
			}
		}

		Result.SetValue( Ret, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_rendererinfo(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_rendererinfo(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_rendererinfo );

//		boolean __fVarying=false;
		int __iGrid;

		__iGrid = 0;
		p_String p_aq_name = new p_String();
		(name).GetString(p_aq_name,__iGrid);
		String _aq_name = p_aq_name.value;
		float Ret = 0.0f;

		if ( _aq_name.compareTo( "renderer" ) == 0 )
		{
			if ( pV.Type().getValue() == type_string )
			{
				pV.SetString( STRNAME );
				Ret = 1.0f;
			}
		}
		else if ( _aq_name.compareTo( "version" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{
				if ( pV.ArrayLength() >= 4 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( VERMAJOR ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( VERMINOR ) );
					pV.ArrayEntry( 2 ) .SetFloat( (float)( BUILD ) );
					pV.ArrayEntry( 3 ) .SetFloat( 0.0f );
					Ret = 1.0f;
				}
			}
		}
		else if ( _aq_name.compareTo( "versionstring" ) == 0 )
		{
			if ( pV.Type().getValue() == type_string )
			{
				pV.SetString( VERSION_STR );
				Ret = 1.0f;
			}
		}
		Result.SetValue( Ret, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_incident(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_incident(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_incident );

//		boolean __fVarying=false;
//		int __iGrid;
//
//		__iGrid = 0;
		Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_opposite(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_opposite(IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_opposite );

//		boolean __fVarying=false;
//		int __iGrid;
//
//		__iGrid = 0;
		Result.SetValue( 0.0f, 0 );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fcellnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fcellnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fcellnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetFloat(m_cellnoise.FCellNoise1( _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fcellnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fcellnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fcellnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetFloat(m_cellnoise.FCellNoise2( _aq_u.value, _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fcellnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fcellnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fcellnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetFloat(m_cellnoise.FCellNoise3( _aq_p ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fcellnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fcellnoise4(IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader) {
		
		STATS_INC( SHD_so_fcellnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetFloat(m_cellnoise.FCellNoise4( _aq_p, _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ccellnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ccellnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ccellnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetColor(new CqColor( m_cellnoise.PCellNoise1( _aq_v.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ccellnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ccellnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ccellnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetColor(new CqColor( m_cellnoise.PCellNoise2( _aq_u.value, _aq_v.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ccellnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ccellnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ccellnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetColor(new CqColor( m_cellnoise.PCellNoise3( _aq_p ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ccellnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ccellnoise4(IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ccellnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetColor(new CqColor( m_cellnoise.PCellNoise4( _aq_p, _aq_v.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pcellnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pcellnoise1(IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pcellnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetPoint(m_cellnoise.PCellNoise1( _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pcellnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pcellnoise2(IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pcellnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetPoint(m_cellnoise.PCellNoise2( _aq_u.value, _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pcellnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pcellnoise3(IqShaderData p, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pcellnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				(Result).SetPoint(m_cellnoise.PCellNoise3( _aq_p ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_pcellnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_pcellnoise4(IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_pcellnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				(Result).SetPoint(m_cellnoise.PCellNoise4( _aq_p, _aq_v.value ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	static int cnt = 0;
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fpnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fpnoise1(IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fpnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(period).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_period = new p_float();
				(period).GetFloat(_aq_period,__iGrid);
				(Result).SetFloat(( m_noise.FGNoise1( ( _aq_v.value % _aq_period.value ) ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fpnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fpnoise2(IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fpnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(uperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(vperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_uperiod = new p_float();
				(uperiod).GetFloat(_aq_uperiod,__iGrid);
				p_float _aq_vperiod = new p_float();
				(vperiod).GetFloat(_aq_vperiod,__iGrid);

	                        float a, b;
	                        a = ( _aq_u.value % _aq_uperiod.value ); 
	                        b = ( _aq_v.value % _aq_vperiod.value);

				(Result).SetFloat(( m_noise.FGNoise2( a, b ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fpnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fpnoise3(IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fpnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);

				float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x ); 
				b = ( _aq_p.y % _aq_pperiod.y ); 
				c = ( _aq_p.z % _aq_pperiod.z );

				(Result).SetFloat(( m_noise.FGNoise3( new CqVector3D( a, b, c )) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fpnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fpnoise4(IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fpnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(tperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);
				p_float _aq_tperiod = new p_float();
				(tperiod).GetFloat(_aq_tperiod,__iGrid);


				float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x ); 
				b = ( _aq_p.y % _aq_pperiod.y ); 
				c = ( _aq_p.z % _aq_pperiod.z );
	            
				(Result).SetFloat(( m_noise.FGNoise3( new CqVector3D( a, b, c ) ) + 1 ) / 2.0f,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cpnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cpnoise1(IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cpnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(period).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_period = new p_float();
				(period).GetFloat(_aq_period,__iGrid);
				(Result).SetColor(( m_noise.CGNoise1( ( _aq_v.value % _aq_period.value ) ) .add(1) .div( 2.0f ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cpnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cpnoise2(IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cpnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(uperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(vperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_uperiod = new p_float();
				(uperiod).GetFloat(_aq_uperiod,__iGrid);
				p_float _aq_vperiod = new p_float();
				(vperiod).GetFloat(_aq_vperiod,__iGrid);

	                        float a, b;
				a = ( _aq_u.value % _aq_uperiod.value );
				b = ( _aq_v.value % _aq_vperiod.value );

				(Result).SetColor(( m_noise.CGNoise2( a, b ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cpnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cpnoise3(IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cpnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);

	                        float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x);
				b = ( _aq_p.y % _aq_pperiod.y);
				c = ( _aq_p.z % _aq_pperiod.z);

				(Result).SetColor(( m_noise.CGNoise3( new CqVector3D( a, b, c ) ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_cpnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_cpnoise4(IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_cpnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(tperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);
				p_float _aq_tperiod = new p_float();
				(tperiod).GetFloat(_aq_tperiod,__iGrid);

				float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x);
				b = ( _aq_p.y % _aq_pperiod.y);
				c = ( _aq_p.z % _aq_pperiod.z);

				(Result).SetColor(( m_noise.CGNoise3( new CqVector3D( a, b, c ) ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ppnoise1(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ppnoise1(IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ppnoise1 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(period).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_period = new p_float();
				(period).GetFloat(_aq_period,__iGrid);
				(Result).SetPoint(( m_noise.PGNoise1( ( _aq_v.value % _aq_period.value ) ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);	
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ppnoise2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ppnoise2(IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ppnoise2 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(u).Class().getValue()==class_varying||__fVarying;
		__fVarying=(uperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;
		__fVarying=(vperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_u = new p_float();
				(u).GetFloat(_aq_u,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				p_float _aq_uperiod = new p_float();
				(uperiod).GetFloat(_aq_uperiod,__iGrid);
				p_float _aq_vperiod = new p_float();
				(vperiod).GetFloat(_aq_vperiod,__iGrid);

	                        float a, b;
	                        a = ( _aq_u.value % _aq_uperiod.value ); 
	                        b = ( _aq_v.value % _aq_vperiod.value );

				(Result).SetPoint(( m_noise.PGNoise2( a, b ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ppnoise3(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ppnoise3(IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ppnoise3 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);

				float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x);
				b = ( _aq_p.y % _aq_pperiod.y);
				c = ( _aq_p.z % _aq_pperiod.z);
				
				(Result).SetPoint(( m_noise.PGNoise3( new CqVector3D( a,b,c ) ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ppnoise4(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ppnoise4(IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ppnoise4 );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(p).Class().getValue()==class_varying||__fVarying;
		__fVarying=(pperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;
		__fVarying=(tperiod).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_p = new CqVector3D();
				(p).GetPoint(_aq_p,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_pperiod = new CqVector3D();
				(pperiod).GetPoint(_aq_pperiod,__iGrid);
				p_float _aq_tperiod = new p_float();
				(tperiod).GetFloat(_aq_tperiod,__iGrid);


				float a, b, c;
				a = ( _aq_p.x % _aq_pperiod.x);
				b = ( _aq_p.y % _aq_pperiod.y);
				c = ( _aq_p.z % _aq_pperiod.z);

				(Result).SetPoint(( m_noise.PGNoise3( new CqVector3D( a,b,c ) ) .add(1) ) .div( 2.0f ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ctransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ctransform(IqShaderData fromspace, IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ctransform );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(c).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		String strfromspace = "rgb";
		if ( null != fromspace ){
			p_String p_strfromspace = new p_String(strfromspace);
			fromspace.GetString( p_strfromspace );
			strfromspace = p_strfromspace.value;
		}
		String _aq_tospace = "";
		p_String p_aq_tospace = new p_String(_aq_tospace);
		(tospace).GetString(p_aq_tospace,__iGrid);
		_aq_tospace = p_aq_tospace.value;
		

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqColor _aq_c = new CqColor();
				(c).GetColor(_aq_c,__iGrid);
				CqColor res = new CqColor( new CqVector3D(_aq_c) );
				if ( strfromspace.compareTo( "hsv" ) == 0 )
					res = _aq_c.hsvtorgb();
				else if ( strfromspace.compareTo( "hsl" ) == 0 )
					res = _aq_c.hsltorgb();
				else if ( strfromspace.compareTo( "XYZ" ) == 0 )
					res = _aq_c.XYZtorgb();
				else if ( strfromspace.compareTo( "xyY" ) == 0 )
					res = _aq_c.xyYtorgb();
				else if ( strfromspace.compareTo( "YIQ" ) == 0 )
					res = _aq_c.YIQtorgb();

				if ( _aq_tospace.compareTo( "hsv" ) == 0 )
					res = _aq_c.rgbtohsv();
				else if ( _aq_tospace.compareTo( "hsl" ) == 0 )
					res = _aq_c.rgbtohsl();
				else if ( _aq_tospace.compareTo( "XYZ" ) == 0 )
					res = _aq_c.rgbtoXYZ();
				else if ( _aq_tospace.compareTo( "xyY" ) == 0 )
					res = _aq_c.rgbtoxyY();
				else if ( _aq_tospace.compareTo( "YIQ" ) == 0 )
					res = _aq_c.rgbtoYIQ();

				(Result).SetColor(res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ctransform(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ctransform(IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ctransform );

		assert( pShader != null );
		SO_ctransform( null, tospace, c, Result, pShader );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_ptlined(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_ptlined(IqShaderData P0, IqShaderData P1, IqShaderData Q, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_ptlined );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(P0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(P1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Q).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_P0 = new CqVector3D();
				(P0).GetPoint(_aq_P0,__iGrid);
				CqVector3D _aq_P1 = new CqVector3D();
				(P1).GetPoint(_aq_P1,__iGrid);
				CqVector3D _aq_Q = new CqVector3D();
				(Q).GetPoint(_aq_Q,__iGrid);
				CqVector3D kDiff = _aq_Q.sub( _aq_P0 );
				CqVector3D vecDir = _aq_P1.sub( _aq_P0 );
				float fT = kDiff .mul( vecDir );

				if ( fT <= 0.0f )
					fT = 0.0f;
				else
				{
					float fSqrLen = vecDir.Magnitude2();
					if ( fT >= fSqrLen )
					{
						fT = 1.0f;
						kDiff .assignSub( vecDir );
					}
					else
					{
						fT /= fSqrLen;
						kDiff .assignSub( vecDir.mul(fT) );
					}
				}
				(Result).SetFloat(kDiff.Magnitude(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_inversesqrt(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_inversesqrt(IqShaderData x, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_inversesqrt );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(x).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_x = new p_float();
				(x).GetFloat(_aq_x,__iGrid);
				(Result).SetFloat(1.0f / (float)( sqrt( _aq_x.value ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_match(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_match(IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader) {

		// TODO: Do this properly.
		STATS_INC( SHD_so_match );

//		boolean __fVarying=false;
		int __iGrid;

		__iGrid = 0;
		float r = 0.0f;
		p_String _aq_a = new p_String();
		(a).GetString(_aq_a,__iGrid);
		p_String _aq_b = new p_String();
		(b).GetString(_aq_b,__iGrid);
		if ( _aq_a.value.length() == 0 )
			r = 0.0f;
		else if ( _aq_b.value.length() == 0 )
			r = 0.0f;
		else
		{
			// MJO> Only check the occurrence of string b in string a
			// It doesn't support the regular expression yet
//			r = ( float ) ( strstr( _aq_b, _aq_a ) != 0 );
			r = ( _aq_b.value.indexOf( _aq_a.value ) != -1 )? 1.0f : 0.0f;
		}

		(Result).SetFloat(r,__iGrid);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_rotate(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_rotate(IqShaderData Q, IqShaderData angle, IqShaderData P0, IqShaderData P1, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_rotate );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Q).Class().getValue()==class_varying||__fVarying;
		__fVarying=(angle).Class().getValue()==class_varying||__fVarying;
		__fVarying=(P0).Class().getValue()==class_varying||__fVarying;
		__fVarying=(P1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_angle = new p_float();
				(angle).GetFloat(_aq_angle,__iGrid);
				CqVector3D _aq_Q = new CqVector3D();
				(Q).GetVector(_aq_Q,__iGrid);
				CqVector3D _aq_P0 = new CqVector3D();
				(P0).GetPoint(_aq_P0,__iGrid);
				CqVector3D _aq_P1 = new CqVector3D();
				(P1).GetPoint(_aq_P1,__iGrid);
				CqMatrix matR = new CqMatrix( _aq_angle.value, _aq_P1 .sub( _aq_P0 ) );

				CqVector3D	Res = new CqVector3D( _aq_Q );
				Res = matR .multiply( Res );

				(Result).SetPoint(Res,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_filterstep(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_filterstep(IqShaderData edge, IqShaderData s1, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_filterstep );

		float Deffloat = 0.0f;
		boolean __fVarying=false;
		int __iGrid;

		p_float _pswidth=new p_float(1.0f),_ptwidth=new p_float(1.0f);
		GetFilterParams(cParams, apParams, _pswidth,_ptwidth);

		__fVarying=(edge).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_float fdu = new p_float(), fdv = new p_float();
		du() .GetFloat( fdu );
		dv() .GetFloat( fdv );

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s1 = new p_float();
				(s1).GetFloat(_aq_s1,__iGrid);
				p_float _aq_edge = new p_float();
				(edge).GetFloat(_aq_edge,__iGrid);
				float dsdu = SO_DuType_TqFloat( s1, __iGrid, this, Deffloat );
				float dsdv = SO_DvType_TqFloat( s1, __iGrid, this, Deffloat );

				float uwidth = abs( dsdu * fdu.value );
				float vwidth = abs( dsdv * fdv.value );

				float w = uwidth + vwidth;
				w *= _pswidth.value;

				(Result).SetFloat(CLAMP( ( _aq_s1.value + w / 2.0f - _aq_edge.value ) / w, 0, 1 ),__iGrid);

				//	p_float res  = RiCatmullRomFilter( FLOAT( s1 ) - FLOAT( edge ), 0, w, 0);
				//	SETFLOAT( Result, res );

				//	std::cout << res << std::endl;
				//	p_float res = 1.0f - CLAMP( ( FLOAT( s1 ) + w / 2.0f - FLOAT( edge ) ) / w, 0, 1 );
				//	if( res > 0.0f )
				//		std::cout << "Aqsis angle/dangle: " << FLOAT(s1) << ", edge: " << FLOAT(edge) << ", dsdu: " << dsdu << ", dsdv: " << dsdv << ", w: " << w << ", res: " << res << std::endl;
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_filterstep2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_filterstep2(IqShaderData edge, IqShaderData s1, IqShaderData s2, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_filterstep2 );

		p_float _pswidth=new p_float(1.0f),_ptwidth=new p_float(1.0f);
		GetFilterParams(cParams, apParams, _pswidth,_ptwidth);

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(edge).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s1).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s2).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_edge = new p_float();
				(edge).GetFloat(_aq_edge,__iGrid);
				p_float _aq_s1 = new p_float();
				(s1).GetFloat(_aq_s1,__iGrid);
				p_float _aq_s2 = new p_float();
				(s2).GetFloat(_aq_s2,__iGrid);
				float w = _aq_s2.value - _aq_s1.value;
				w *= _pswidth.value;
				(Result).SetFloat(CLAMP( ( _aq_s1.value + w / 2.0f - _aq_edge.value ) / w, 0, 1 ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_specularbrdf(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_specularbrdf(IqShaderData L, IqShaderData N, IqShaderData V, IqShaderData rough, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_specularbrdf );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(L).Class().getValue()==class_varying||__fVarying;
		__fVarying=(N).Class().getValue()==class_varying||__fVarying;
		__fVarying=(V).Class().getValue()==class_varying||__fVarying;
		__fVarying=(rough).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_L = new CqVector3D();
				(L).GetVector(_aq_L,__iGrid);
				CqVector3D _aq_V = new CqVector3D();
				(V).GetVector(_aq_V,__iGrid);
				_aq_L.Unit();

				CqVector3D	H = _aq_L .add( _aq_V );
				H.Unit();
				/// \note The (roughness/8) term emulates the BMRT behaviour for prmanspecular.
				CqVector3D _aq_N = new CqVector3D();
				(N).GetNormal(_aq_N,__iGrid);
				p_float _aq_rough = new p_float();
				(rough).GetFloat(_aq_rough,__iGrid);
				CqColor colCl = new CqColor();
				Cl() .GetColor( colCl, __iGrid );
				(Result).SetColor(colCl.mul( (float)pow( max( 0.0f, _aq_N .mul(H) ), 1.0f / ( _aq_rough.value / 8.0f ) ) ),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_setmcomp(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_setmcomp(IqShaderData M, IqShaderData r, IqShaderData c, IqShaderData v, IqShader pShader) {

		STATS_INC( SHD_so_setmcomp );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(M).Class().getValue()==class_varying||__fVarying;
		__fVarying=(r).Class().getValue()==class_varying||__fVarying;
		__fVarying=(c).Class().getValue()==class_varying||__fVarying;
		__fVarying=(v).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_M = new CqMatrix();
				(M).GetMatrix(_aq_M,__iGrid);
				p_float _aq_r = new p_float();
				(r).GetFloat(_aq_r,__iGrid);
				p_float _aq_c = new p_float();
				(c).GetFloat(_aq_c,__iGrid);
				p_float _aq_v = new p_float();
				(v).GetFloat(_aq_v,__iGrid);
				_aq_M.m_aaElement[ (int)( _aq_r.value )][ (int)( _aq_c.value )] = _aq_v.value ;
				_aq_M.SetfIdentity( false );
				M.SetValue( _aq_M, __iGrid );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_determinant(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_determinant(IqShaderData M, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_determinant );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(M).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_M = new CqMatrix();
				(M).GetMatrix(_aq_M,__iGrid);
				(Result).SetFloat(_aq_M.Determinant(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_mtranslate(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_mtranslate(IqShaderData M, IqShaderData V, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_mtranslate );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(M).Class().getValue()==class_varying||__fVarying;
		__fVarying=(V).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_M = new CqMatrix();
				(M).GetMatrix(_aq_M,__iGrid);
				CqVector3D _aq_V = new CqVector3D();
				(V).GetVector(_aq_V,__iGrid);
				_aq_M.Translate( _aq_V );
				(Result).SetMatrix(_aq_M,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_mrotate(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_mrotate(IqShaderData M, IqShaderData angle, IqShaderData axis, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_mrotate );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(M).Class().getValue()==class_varying||__fVarying;
		__fVarying=(angle).Class().getValue()==class_varying||__fVarying;
		__fVarying=(axis).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqMatrix _aq_M = new CqMatrix();
				(M).GetMatrix(_aq_M,__iGrid);
				p_float _aq_angle = new p_float();
				(angle).GetFloat(_aq_angle,__iGrid);
				CqVector3D _aq_axis = new CqVector3D();
				(axis).GetVector(_aq_axis,__iGrid);
				_aq_M.Rotate( _aq_angle.value, _aq_axis );
				(Result).SetMatrix(_aq_M,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_mscale(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_mscale(IqShaderData M, IqShaderData S, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_mscale );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(M).Class().getValue()==class_varying||__fVarying;
		__fVarying=(S).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				CqVector3D _aq_S = new CqVector3D();
				(S).GetPoint(_aq_S,__iGrid);
				CqMatrix _aq_M = new CqMatrix();
				(M).GetMatrix(_aq_M,__iGrid);
				_aq_M.Scale( _aq_S.x, _aq_S.y, _aq_S.z );
				(Result).SetMatrix(_aq_M,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_fsplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_fsplinea(IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_fsplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_float );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

				p_float fTemp = new p_float();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetFloat( fTemp, __iGrid );
					Result.SetFloat( fTemp.value, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetFloat( fTemp, __iGrid );
					Result.SetFloat( fTemp.value, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetFloat( fTemp, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( fTemp.value, 0.0f, 0.0f, 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetFloat(res.x,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_csplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_csplinea(IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_csplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_color );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );
		CqColor colTemp = new CqColor();

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

//				CqColor cTemp = new CqColor();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetColor( colTemp, __iGrid );
					Result.SetColor( colTemp, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetColor( colTemp, __iGrid );
					Result.SetColor( colTemp, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetColor( colTemp, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( colTemp.fRed(), colTemp.fGreen(), colTemp.fBlue(), 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetColor( new CqColor( res.x, res.y, res.z ),__iGrid );
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_psplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_psplinea(IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_psplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_point );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );
//		CqVector3D vecTemp = new CqVector3D();

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

				CqVector3D vecTemp = new CqVector3D();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetPoint( vecTemp, __iGrid );
					Result.SetPoint( vecTemp, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetPoint( vecTemp, __iGrid );
					Result.SetPoint( vecTemp, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetPoint( vecTemp, __iGrid );
						spline.valueAt( j ).assignment( vecTemp );
					}

					CqVector3D	res = new CqVector3D( spline.Evaluate( _aq_value.value ) );
					(Result).SetPoint(res,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_sfsplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_sfsplinea(IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_sfsplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_float );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

				p_float fTemp = new p_float();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetFloat( fTemp, __iGrid );
					Result.SetFloat( fTemp.value, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetFloat( fTemp, __iGrid );
					Result.SetFloat( fTemp.value, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetFloat( fTemp, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( fTemp.value, 0.0f, 0.0f, 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetFloat(res.x,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_scsplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_scsplinea(IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_scsplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_color );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );
//		CqColor colTemp;

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

				CqColor colTemp = new CqColor();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetColor( colTemp, __iGrid );
					Result.SetColor( colTemp, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetColor( colTemp, __iGrid );
					Result.SetColor( colTemp, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetColor( colTemp, __iGrid );
						spline.valueAt( j ).assignment( new CqVector4D( colTemp.fRed(), colTemp.fGreen(), colTemp.fBlue(), 1.0f ) );
					}

					CqVector4D	res = spline.Evaluate( _aq_value.value );
					(Result).SetColor(new CqColor( res.x, res.y, res.z ),__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_spsplinea(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_spsplinea(IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_spsplinea );

		boolean __fVarying=false;
		int __iGrid;

		assert( a.ArrayLength() > 0 );
		assert( a.Type().getValue() == type_point );

		int	cParams = a.ArrayLength();
		CqSplineCubic spline = new CqSplineCubic( cParams );
//		CqVector3D vecTemp = new CqVector3D();

		__fVarying=(value).Class().getValue()==class_varying||__fVarying;
		__fVarying=(a).Class().getValue()==class_varying||__fVarying;
		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_basis = new p_String();
		(basis).GetString(_aq_basis,__iGrid);
		spline.SetBasis( _aq_basis.value );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_value = new p_float();
				(value).GetFloat(_aq_value,__iGrid);

				CqVector3D vecTemp = new CqVector3D();
				if ( _aq_value.value >= 1.0f )
				{
					a.ArrayEntry( cParams - 2 ) .GetPoint( vecTemp, __iGrid );
					Result.SetPoint( vecTemp, __iGrid );
				}
				else if ( _aq_value.value <= 0.0f )
				{
					a.ArrayEntry( 1 ) .GetPoint( vecTemp, __iGrid );
					Result.SetPoint( vecTemp, __iGrid );
				}
				else
				{
					int j;
					for ( j = 0; j < cParams; j++ )
					{
						a.ArrayEntry( j ) .GetPoint( vecTemp, __iGrid );
						spline.valueAt( j ).assignment( vecTemp );
					}

					CqVector3D	res = new CqVector3D( spline.Evaluate( _aq_value.value ) );
					(Result).SetPoint(res,__iGrid);
				}
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_shadername(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_shadername(IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_shadername );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				(Result).SetString(pShader.strName(),__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_shadername2(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_shadername2(IqShaderData shader, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_shadername2 );

		boolean __fVarying=false;
		int __iGrid;

		String strName = ( "" );
//		String strShader = "";
		IqShader pSurface = null;
		IqShader pDisplacement = null;
		IqShader pAtmosphere = null;
		if( m_pAttributes!=null )
		{
			pSurface = m_pAttributes .pshadSurface(QGetRenderContextI().Time());
			pDisplacement = m_pAttributes .pshadDisplacement(QGetRenderContextI().Time());
			pAtmosphere = m_pAttributes .pshadAtmosphere(QGetRenderContextI().Time());
		}

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				strName = "";
				p_String _aq_shader = new p_String();
				(shader).GetString(_aq_shader,__iGrid);
				if ( _aq_shader.value.compareTo( "surface" ) == 0 && pSurface!=null )
					strName = pSurface.strName();
				else if ( _aq_shader.value.compareTo( "displacement" ) == 0 && pDisplacement!=null )
					strName = pDisplacement.strName();
				else if ( _aq_shader.value.compareTo( "atmosphere" ) == 0 && pAtmosphere!=null )
					strName = pAtmosphere.strName();
				(Result).SetString(strName,__iGrid);
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_textureinfo(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader)
	 */
	@Override
	public void SO_textureinfo(IqShaderData name, IqShaderData dataname, IqShaderData pV, IqShaderData Result, IqShader pShader) {

		STATS_INC( SHD_so_textureinfo );

//		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		float Ret = 0.0f;
		IqTextureMap pMap = null;
		IqTextureMap pSMap = null;
//		IqTextureMap pLMap = null;
		IqTextureMap pEMap = null;
		IqTextureMap pTMap = null;

		__iGrid = 0;
		String _aq_name = "";
		p_String p_aq_name = new p_String(_aq_name);
		(name).GetString(p_aq_name,__iGrid);
		_aq_name = p_aq_name.value;
		
		String _aq_dataname = "";
		p_String p_aq_dataname = new p_String(_aq_dataname);
		(dataname).GetString(p_aq_dataname,__iGrid);
		_aq_dataname = p_aq_dataname.value;

//		if ( pMap==null && strstr( _aq_name, ".tif" ) )
		if ( pMap==null && _aq_name.indexOf( ".tif" )!=-1 )
		{
			pTMap = QGetRenderContextI() .GetTextureMap( _aq_name );
			if ( pTMap!=null && ( pTMap.Type().getValue() == MapType_Texture ) )
			{
				pMap = pTMap;
			}
			else if ( pTMap!=null )
//				delete pTMap;
				pTMap = null;
		}
		if ( pMap==null )
		{
			pSMap = QGetRenderContextI() .GetShadowMap( _aq_name );
			if ( pSMap!=null && ( pSMap.Type().getValue() == MapType_Shadow ) )
			{
				pMap = pSMap;
			}
			else if ( pSMap!=null )
//				delete pSMap;
				pSMap = null;
		}

		if ( pMap==null )
		{
			pEMap = QGetRenderContextI() .GetEnvironmentMap( _aq_name );
			if ( pEMap!=null && ( pEMap.Type().getValue() == MapType_Environment ) )
			{
				pMap = pEMap;
			}
			else if ( pEMap!=null )
//				delete pEMap;
				pEMap = null;
		}

		if ( pMap==null )
		{
			pTMap = QGetRenderContextI() .GetTextureMap( _aq_name );
			if ( pTMap!=null && ( pTMap.Type().getValue() == MapType_Texture ) )
			{
				pMap = pTMap;
			}
			else if ( pTMap!=null )
//				delete pTMap;
				pTMap = null;
		}


		if ( pMap == null )
			return ;

		if ( _aq_dataname.compareTo( "resolution" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float &&
			        pV.ArrayLength() > 0 )
			{

				if ( pV.ArrayLength() == 2 )
				{
					pV.ArrayEntry( 0 ) .SetFloat( (float)( pMap.XRes() ) );
					pV.ArrayEntry( 1 ) .SetFloat( (float)( pMap.YRes() ) );
					Ret = 1.0f;

				}
			}
		}
		if ( _aq_dataname.compareTo( "type" ) == 0 )
		{
			if ( pV.Type().getValue() == type_string )
			{
				if ( pMap.Type().getValue() == MapType_Texture )
				{
					pV.SetString( "texture" );
					Ret = 1.0f;

				}
				if ( pMap.Type().getValue() == MapType_Bump )
				{
					pV.SetString( "bump" );
					Ret = 1.0f;

				}

				if ( pMap.Type().getValue() == MapType_Shadow )
				{
					pV.SetString( "shadow" );
					Ret = 1.0f;

				}
				if ( pMap.Type().getValue() == MapType_Environment )
				{
					pV.SetString( "environment" );
					Ret = 1.0f;

				}
				if ( pMap.Type().getValue() == MapType_LatLong )
				{
					// both latlong/cube respond the same way according to BMRT
					// It makes sense since both use environment() shader fct.
					pV.SetString( "environment" );
					Ret = 1.0f;

				}


			}
		}

		if ( _aq_dataname.compareTo( "channels" ) == 0 )
		{
			if ( pV.Type().getValue() == type_float )
			{
				pV.SetFloat( (float)( pMap.SamplesPerPixel() ) );
				Ret = 1.0f;
			}

		}

		if ( _aq_dataname.compareTo( "viewingmatrix" ) == 0 )
		{
			if ( ( ( pV.Type().getValue() == type_float ) && ( pV.ArrayLength() == 16 ) ) ||
			        ( pV.Type().getValue() == type_matrix ) )
			{
				if ( pSMap!=null )   // && pSMap.Type().getValue() == MapType_Shadow)
				{

					CqMatrix m = pSMap.GetMatrix( 0 );  /* WorldToCamera */
					if ( pV.ArrayLength() == 16 )
					{

						pV.ArrayEntry( 0 ) .SetFloat( (float)( m.m_aaElement[0][0] ) );
						pV.ArrayEntry( 1 ) .SetFloat( (float)( m.m_aaElement[0][1] ) );
						pV.ArrayEntry( 2 ) .SetFloat( (float)( m.m_aaElement[0][2] ) );
						pV.ArrayEntry( 3 ) .SetFloat( (float)( m.m_aaElement[0][3] ) );
						pV.ArrayEntry( 4 ) .SetFloat( (float)( m.m_aaElement[1][0] ) );
						pV.ArrayEntry( 5 ) .SetFloat( (float)( m.m_aaElement[1][1] ) );
						pV.ArrayEntry( 6 ) .SetFloat( (float)( m.m_aaElement[1][2] ) );
						pV.ArrayEntry( 7 ) .SetFloat( (float)( m.m_aaElement[1][3] ) );
						pV.ArrayEntry( 8 ) .SetFloat( (float)( m.m_aaElement[2][0] ) );
						pV.ArrayEntry( 9 ) .SetFloat( (float)( m.m_aaElement[2][1] ) );
						pV.ArrayEntry( 10 ) .SetFloat( (float)( m.m_aaElement[2][2] ) );
						pV.ArrayEntry( 11 ) .SetFloat( (float)( m.m_aaElement[2][3] ) );
						pV.ArrayEntry( 12 ) .SetFloat( (float)( m.m_aaElement[3][0] ) );
						pV.ArrayEntry( 13 ) .SetFloat( (float)( m.m_aaElement[3][1] ) );
						pV.ArrayEntry( 14 ) .SetFloat( (float)( m.m_aaElement[3][2] ) );
						pV.ArrayEntry( 15 ) .SetFloat( (float)( m.m_aaElement[3][3] ) );

					}
					else
					{
						pV.SetMatrix( m, 0 );
					}
					Ret = 1.0f;

				}

			}
		}

		if ( _aq_dataname.compareTo( "projectionmatrix" ) == 0 )
		{
			if ( ( ( pV.Type().getValue() == type_float ) && ( pV.ArrayLength() == 16 ) ) ||
			        ( pV.Type().getValue() == type_matrix ) )
			{
				if ( pSMap!=null )    // && pSMap.Type()== MapType_Shadow)
				{

					CqMatrix m = pSMap.GetMatrix( 1 ); /* WorldToScreen */

					if ( pV.ArrayLength() == 16 )
					{
						pV.ArrayEntry( 0 ) .SetFloat( (float)( m.m_aaElement[0][0] ) );
						pV.ArrayEntry( 1 ) .SetFloat( (float)( m.m_aaElement[0][1] ) );
						pV.ArrayEntry( 2 ) .SetFloat( (float)( m.m_aaElement[0][2] ) );
						pV.ArrayEntry( 3 ) .SetFloat( (float)( m.m_aaElement[0][3] ) );
						pV.ArrayEntry( 4 ) .SetFloat( (float)( m.m_aaElement[1][0] ) );
						pV.ArrayEntry( 5 ) .SetFloat( (float)( m.m_aaElement[1][1] ) );
						pV.ArrayEntry( 6 ) .SetFloat( (float)( m.m_aaElement[1][2] ) );
						pV.ArrayEntry( 7 ) .SetFloat( (float)( m.m_aaElement[1][3] ) );
						pV.ArrayEntry( 8 ) .SetFloat( (float)( m.m_aaElement[2][0] ) );
						pV.ArrayEntry( 9 ) .SetFloat( (float)( m.m_aaElement[2][1] ) );
						pV.ArrayEntry( 10 ) .SetFloat( (float)( m.m_aaElement[2][2] ) );
						pV.ArrayEntry( 11 ) .SetFloat( (float)( m.m_aaElement[2][3] ) );
						pV.ArrayEntry( 12 ) .SetFloat( (float)( m.m_aaElement[3][0] ) );
						pV.ArrayEntry( 13 ) .SetFloat( (float)( m.m_aaElement[3][1] ) );
						pV.ArrayEntry( 14 ) .SetFloat( (float)( m.m_aaElement[3][2] ) );
						pV.ArrayEntry( 15 ) .SetFloat( (float)( m.m_aaElement[3][3] ) );


					}
					else
					{
						pV.SetMatrix( m, 0 );

					}
					Ret = 1.0f;
				}

			}
		}

//		delete pMap;
		pMap = null;

		(Result).SetFloat(Ret,__iGrid);

	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bake_f(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bake_f(IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_bake );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(f).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		HashMap<String, BakingChannel> bd = bake_init(  /*STRING( name) */ );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s = new p_float();
				(s).GetFloat(_aq_s,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				p_float _aq_f = new p_float();
				(f).GetFloat(_aq_f,__iGrid);
				bake_f( bd, _aq_name.value, _aq_s.value, _aq_t.value, _aq_f.value );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		__iGrid = 0;
		bake_done( bd );

	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bake_3c(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bake_3c(IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_bake );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(f).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		float[] rgb = new float[ 3 ];

		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		HashMap<String, BakingChannel> bd = bake_init(  /*(char *) STRING( name )*/ );

		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s = new p_float();
				(s).GetFloat(_aq_s,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqColor _aq_f = new CqColor();
				(f).GetColor(_aq_f,__iGrid);
				
				p_float[] p_rgb = new p_float[3];
				for( int i=0; i<p_rgb.length; i++ ) p_rgb[i] = new p_float();
				_aq_f.GetColorRGB( p_rgb[ 0 ], p_rgb[ 1 ], p_rgb[ 2 ] );
				rgb[0] = p_rgb[0].value;
				rgb[1] = p_rgb[1].value;
				rgb[2] = p_rgb[2].value;
				
				bake_3( bd, _aq_name.value, _aq_s.value, _aq_t.value, rgb );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);
		__iGrid = 0;
		bake_done( bd );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bake_3p(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bake_3p(IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_bake );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(f).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		HashMap<String, BakingChannel> bd = bake_init(  /*(char *) STRING( name )  */ );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s = new p_float();
				(s).GetFloat(_aq_s,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_f = new CqVector3D();
				(f).GetPoint(_aq_f,__iGrid);
				float[] rgb = new float[ 3 ];
				rgb[ 0 ] = _aq_f.valueAt( 0 );
				rgb[ 1 ] = _aq_f.valueAt( 1 );
				rgb[ 2 ] = _aq_f.valueAt( 2 );
				
				bake_3( bd, _aq_name.value, _aq_s.value, _aq_t.value, rgb );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		__iGrid = 0;
		bake_done( bd );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bake_3n(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bake_3n(IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_bake );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(f).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		HashMap<String, BakingChannel> bd = bake_init(  /*(char *) STRING( name ) */ );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s = new p_float();
				(s).GetFloat(_aq_s,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_f = new CqVector3D();
				(f).GetNormal(_aq_f,__iGrid);
				float[] rgb = new float[ 3 ];
				rgb[ 0 ] = _aq_f.valueAt( 0 );
				rgb[ 1 ] = _aq_f.valueAt( 1 );
				rgb[ 2 ] = _aq_f.valueAt( 2 );
				bake_3( bd, _aq_name.value, _aq_s.value, _aq_t.value, rgb );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		__iGrid = 0;
		bake_done( bd );

	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_bake_3v(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_bake_3v(IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_bake );

		boolean __fVarying=false;
		int __iGrid;
		__fVarying=(f).Class().getValue()==class_varying||__fVarying;
		__fVarying=(s).Class().getValue()==class_varying||__fVarying;
		__fVarying=(t).Class().getValue()==class_varying||__fVarying;

		__iGrid = 0;
		p_String _aq_name = new p_String();
		(name).GetString(_aq_name,__iGrid);
		HashMap<String, BakingChannel> bd = bake_init(  /*(char *) STRING( name )  */ );


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{
				p_float _aq_s = new p_float();
				(s).GetFloat(_aq_s,__iGrid);
				p_float _aq_t = new p_float();
				(t).GetFloat(_aq_t,__iGrid);
				CqVector3D _aq_f = new CqVector3D();
				(f).GetVector(_aq_f,__iGrid);
				float[] rgb = new float[ 3 ];
				rgb[ 0 ] = _aq_f.valueAt( 0 );
				rgb[ 1 ] = _aq_f.valueAt( 1 );
				rgb[ 2 ] = _aq_f.valueAt( 2 );
				bake_3( bd, _aq_name.value, _aq_s.value, _aq_t.value, rgb );
			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

		__iGrid = 0;
		bake_done( bd );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_external(java.lang.reflect.Method, java.lang.Object, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_external(Method method, Object initData, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_external );

		boolean __fVarying=false;
		int __iGrid;

		__fVarying=(Result).Class().getValue()==class_varying||__fVarying;
		int p;
		for ( p = 0;p < cParams;p++ )
		{
			__fVarying=(apParams[ p ]).Class().getValue()==class_varying||__fVarying;
		};

		int dso_argc = cParams + 1; // dso_argv[0] is used for the return value
		Object[] dso_argv = new Object[ dso_argc ] ;

		// create storage for the returned value
		switch ( Result.Type().getValue() )
		{

				case type_float:
				dso_argv[ 0 ] = new p_float();
				break;
				case type_point:
				case type_color:
				case type_triple:
				case type_vector:
				case type_normal:
				case type_hpoint:
				dso_argv[ 0 ] = new float[ 3 ];
				break;
				case type_string:
				dso_argv[ 0 ] = new String();
				break;
				case type_matrix:
				case type_sixteentuple:
				dso_argv[ 0 ] = new float[ 16 ];
				break;
				default:
				// Unhandled TYpe
				break;
		};

		// Allocate space for the arguments
		for ( p = 1;p <= cParams;p++ )
		{

			switch ( apParams[ p - 1 ] .Type().getValue() )
			{
					case type_float:
					dso_argv[ p ] = new p_float();
					break;
					case type_hpoint:
					case type_point:
					case type_triple:  // This seems reasonable
					case type_vector:
					case type_normal:
					case type_color:
					dso_argv[ p ] = new float[ 3 ];
					break;
					case type_string:
					dso_argv[ p ] = new String();
					break;
					case type_matrix:
					case type_sixteentuple:
					dso_argv[ 0 ] = new float[ 16 ];
					break;
					default:
					// Unhandled TYpe
					break;
			};
		};


		__iGrid = 0;
		CqBitVector RS = RunningState();
		do
		{
			if(!__fVarying || RS.Value( __iGrid ) )
			{

				// Convert the arguments to the required format for the DSO
				for ( p = 1;p <= cParams;p++ )
				{

					switch ( apParams[ p - 1 ] .Type().getValue() )
					{
							case type_float:
							apParams[ p - 1 ] .GetFloat( (p_float)dso_argv[ p ], __iGrid );
							break;
							case type_hpoint:
							case type_point:
							{
								CqVector3D v = new CqVector3D();
								apParams[ p - 1 ] .GetPoint( v, __iGrid );
								( ( float[] ) dso_argv[ p ] ) [ 0 ] = v.valueAt( 0 );
								( ( float[] ) dso_argv[ p ] ) [ 1 ] = v.valueAt( 1 );
								( ( float[] ) dso_argv[ p ] ) [ 2 ] = v.valueAt( 2 );
							}
							;
							break;
							case type_triple:  // This seems reasonable
							case type_vector:
							{
								CqVector3D v = new CqVector3D();
								apParams[ p - 1 ] .GetVector( v, __iGrid );
								( ( float[] ) dso_argv[ p ] ) [ 0 ] = v.valueAt( 0 );
								( ( float[] ) dso_argv[ p ] ) [ 1 ] = v.valueAt( 1 );
								( ( float[] ) dso_argv[ p ] ) [ 2 ] = v.valueAt( 2 );
							}
							;
							break;
							case type_normal:
							{
								CqVector3D v = new CqVector3D();
								apParams[ p - 1 ] .GetNormal( v, __iGrid );
								( ( float[] ) dso_argv[ p ] ) [ 0 ] = v.valueAt( 0 );
								( ( float[] ) dso_argv[ p ] ) [ 1 ] = v.valueAt( 1 );
								( ( float[] ) dso_argv[ p ] ) [ 2 ] = v.valueAt( 2 );
							}
							;
							break;
							case type_color:
							{
								CqColor c = new CqColor();
								apParams[ p - 1 ] .GetColor( c, __iGrid );
								( ( float[] ) dso_argv[ p ] ) [ 0 ] = c.valueAt( 0 );
								( ( float[] ) dso_argv[ p ] ) [ 1 ] = c.valueAt( 1 );
								( ( float[] ) dso_argv[ p ] ) [ 2 ] = c.valueAt( 2 );
							}
							;
							break;
							case type_string:
							{
								p_String s = new p_String();
								apParams[ p - 1 ] .GetString( s, __iGrid );
//								char *ps = new char[ s.size() + 1 ];
//								strncpy ( ps, s, s.size() + 1 );
//								( ( STRING_DESC* ) dso_argv[ p ] ) .s = ps;
//								( ( STRING_DESC* ) dso_argv[ p ] ) .bufflen = s.size() + 1;
								dso_argv[ p ] = s.value ;
							}
							;
							break;
							case type_matrix:
							case type_sixteentuple:
							{
								CqMatrix m = new CqMatrix();
								int r, c;
								apParams[ p - 1 ] .GetMatrix( m, __iGrid );
								for ( r = 0; r < 4; r++ )
									for ( c = 0; c < 4; c++ )
										( ( float[] ) dso_argv[ p ] ) [ ( r * 4 ) + c ] = m.m_aaElement[ r ][ c ] ;
							}
							;
							break;
							default:
							// Unhandled TYpe
							break;
					};
				};

				// Atlast, we call the shadeop method, looks rather dull after all this effort.
				try {
					
//					method( initData, dso_argc, dso_argv );
					method.invoke( null, initData, dso_argc, dso_argv );
					
				} catch (Exception e) {
					HimawariLogger.outputException( e );
				}

				// Pass the returned value back to aqsis
				switch ( Result.Type().getValue() )
				{

						case type_float:
						{
							float val = ((p_float)dso_argv[ 0 ]).value;
							Result.SetFloat( val, __iGrid );
						}
						;
						break;
						case type_hpoint:
						case type_point:
						{
							CqVector3D v = new CqVector3D();
							v.x = ( ( float[] ) dso_argv[ 0 ] ) [ 0 ];
							v.y = ( ( float[] ) dso_argv[ 0 ] ) [ 1 ];
							v.z = ( ( float[] ) dso_argv[ 0 ] ) [ 2 ];
							Result.SetPoint( v, __iGrid );
						}
						;
						break;
						case type_triple:  // This seems reasonable
						case type_vector:
						{
							CqVector3D v = new CqVector3D();
							v.x = ( ( float[] ) dso_argv[ 0 ] ) [ 0 ];
							v.y = ( ( float[] ) dso_argv[ 0 ] ) [ 1 ];
							v.z = ( ( float[] ) dso_argv[ 0 ] ) [ 2 ];
							Result.SetVector( v, __iGrid );
						}
						;
						break;
						case type_normal:
						{
							CqVector3D v = new CqVector3D();
							v.x = ( ( float[] ) dso_argv[ 0 ] ) [ 0 ];
							v.y = ( ( float[] ) dso_argv[ 0 ] ) [ 1 ];
							v.z = ( ( float[] ) dso_argv[ 0 ] ) [ 2 ];
							Result.SetNormal( v, __iGrid );
						}
						;
						break;
						case type_color:
						{
							CqColor c = new CqColor();
							c.setValueAt( 0, ( ( float[] ) dso_argv[ 0 ] ) [ 0 ] );
							c.setValueAt( 1, ( ( float[] ) dso_argv[ 0 ] ) [ 1 ] );
							c.setValueAt( 2, ( ( float[] ) dso_argv[ 0 ] ) [ 2 ] );
							Result.SetColor( c, __iGrid );
						}
						;
						break;
						case type_string:
						{
							String s = new String( (String)dso_argv[ 0 ] );
							Result.SetString( s, __iGrid );
						}
						;
						break;
						case type_matrix:
						case type_sixteentuple:
						{
							CqMatrix m = new CqMatrix( ( float[] ) dso_argv[ 0 ] );
							Result.SetMatrix( m, __iGrid );
						}
						;
						break;
						default:
						// Unhandled TYpe
//						std::cout << "Unsupported type" << std::endl;
						System.out.println( "Unsupported type" );
						break;
				};


				// Set the values that were altered by the Shadeop
				for ( p = 1;p <= cParams;p++ )
				{
					switch ( apParams[ p - 1 ] .Type().getValue() )
					{
							case type_float:
							{
								float val = ( ( p_float ) dso_argv[ p ] ).value ;
								apParams[ p - 1 ] .SetFloat( val, __iGrid );
							}
							;
							break;
							case type_hpoint:
							case type_point:
							{
								CqVector3D v = new CqVector3D();
								v.x = ( ( float[] ) dso_argv[ p ] ) [ 0 ];
								v.y = ( ( float[] ) dso_argv[ p ] ) [ 1 ];
								v.z = ( ( float[] ) dso_argv[ p ] ) [ 2 ];
								apParams[ p - 1 ] .SetPoint( v, __iGrid );
							}
							;
							break;
							case type_triple:  // This seems reasonable
							case type_vector:
							{
								CqVector3D v = new CqVector3D();
								v.x = ( ( float[] ) dso_argv[ p ] ) [ 0 ];
								v.y = ( ( float[] ) dso_argv[ p ] ) [ 1 ];
								v.z = ( ( float[] ) dso_argv[ p ] ) [ 2 ];
								apParams[ p - 1 ] .SetVector( v, __iGrid );
							}
							;
							break;
							case type_normal:
							{
								CqVector3D v = new CqVector3D();
								v.x = ( ( float[] ) dso_argv[ p ] ) [ 0 ];
								v.y = ( ( float[] ) dso_argv[ p ] ) [ 1 ];
								v.z = ( ( float[] ) dso_argv[ p ] ) [ 2 ];
								apParams[ p - 1 ] .SetNormal( v, __iGrid );
							}
							;
							break;
							case type_color:
							{
								CqColor c = new CqColor();
								c.setValueAt( 0, ( ( float[] ) dso_argv[ p ] ) [ 0 ] );
								c.setValueAt( 1, ( ( float[] ) dso_argv[ p ] ) [ 1 ] );
								c.setValueAt( 2, ( ( float[] ) dso_argv[ p ] ) [ 2 ] );
								apParams[ p - 1 ] .SetColor( c, __iGrid );
							}
							;
							break;
							case type_string:
							{
								String s = new String( (String)dso_argv[ p ] );
								apParams[ p - 1 ] .SetString( s, __iGrid );
							}
							;
							break;
							case type_matrix:
							case type_sixteentuple:
							{
								CqMatrix m = new CqMatrix( ( float[] ) dso_argv[ p ] );
								apParams[ p - 1 ] .SetMatrix( m, __iGrid );
							}
							;
							break;
							default:
							// Unhandled TYpe
							break;
					};
				};

			}
		}
		while( ( ++__iGrid < GridSize() ) && __fVarying);

//		// Free up the storage allocated for the return type
//		switch ( Result.Type() )
//		{
//
//				case type_float:
//				delete ( float[] ) dso_argv[ 0 ];
//				break;
//				case type_point:
//				case type_triple:  // This seems reasonable
//				case type_vector:
//				case type_normal:
//				case type_color:
//				case type_hpoint:
//				delete[] ( float[] ) dso_argv[ 0 ];
//				break;
//				case type_string:  // Need to look into these
//				delete ( STRING_DESC* ) dso_argv[ 0 ];
//				break;
//				case type_matrix:
//				case type_sixteentuple:
//				delete[] ( float[] ) dso_argv[ 0 ];
//				break;
//				default:
//				// Unhandled TYpe
//				break;
//		};
//
//		// Free up the storage allocated for the args
//		for ( p = 1;p <= cParams;p++ )
//		{
//			switch ( apParams[ p - 1 ] .Type() )
//			{
//					case type_float:
//					delete ( float[] ) dso_argv[ p ];
//					break;
//					case type_point:
//					case type_triple:
//					case type_vector:
//					case type_normal:
//					case type_color:
//					case type_hpoint:
//					delete[] ( float[] ) dso_argv[ p ];
//					break;
//					case type_string:
//					delete ( STRING_DESC* ) dso_argv[ p ];
//					break;
//					case type_matrix:
//					case type_sixteentuple:
//					delete[] ( float[] ) dso_argv[ p ];
//					break;
//					default:
//					// Unhandled TYpe
//					break;
//			};
//		};
//
//		delete dso_argv;
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.IqShaderExecEnv#SO_occlusion(net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShaderData, net.cellcomputing.himawari.library.IqShader, int, net.cellcomputing.himawari.library.IqShaderData[])
	 */
	@Override
	public void SO_occlusion(IqShaderData occlmap, IqShaderData channel, IqShaderData P, IqShaderData N, IqShaderData samples, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams) {

		STATS_INC( SHD_so_occlusion );

		boolean __fVarying=false;
		int __iGrid;

		if ( null == QGetRenderContextI() )
			return ;

		HashMap<String, IqShaderData> paramMap = new HashMap<String, IqShaderData>();
		GetTexParams(cParams, apParams, paramMap);

		__iGrid = 0;
		p_String _aq_occlmap = new p_String();
		(occlmap).GetString(_aq_occlmap,__iGrid);
//		CqVector3D _aq_N = new CqVector3D();
//		(N).GetNormal(_aq_N,__iGrid);
		p_float _aq_samples = new p_float();
		(samples).GetFloat(_aq_samples,__iGrid);
		IqTextureMap pMap = QGetRenderContextI() .GetShadowMap( _aq_occlmap.value );


		CqVector3D L = new CqVector3D(0,0,-1);

		__fVarying = true;
		if ( pMap != null && pMap.IsValid() )
		{
			Valarray fv = new Valarray();
			pMap.PrepareSampleOptions( paramMap );

			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					// Storage for the final combined occlusion value.
					float occlsum = 0.0f;
					float dotsum = 0.0f;

					CqVector3D swidth = new CqVector3D(0.0f), twidth = new CqVector3D(0.0f);


					CqVector3D _aq_P = new CqVector3D();
					(P).GetPoint(_aq_P,__iGrid);
					CqVector3D _aq_N = new CqVector3D();
					(N).GetNormal(_aq_N,__iGrid);
					int i = pMap.NumPages() - 1;
					for( ; i >= 0; i-- )
					{
						// Check if the lightsource is behind the sample.
						CqVector3D Nl = pMap.GetMatrix(2, i) .multiply( _aq_N );
						float cosangle = Nl .mul( L );
						if( cosangle < 0.0f )
							continue;

					        pMap.SampleMap( _aq_P, swidth, twidth, fv, i );
						occlsum += cosangle * fv.valueAt(0);
						dotsum += cosangle;
					}
					occlsum /= dotsum;
					(Result).SetFloat(occlsum,__iGrid);
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
		else
		{
			__iGrid = 0;
			CqBitVector RS = RunningState();
			do
			{
				if(!__fVarying || RS.Value( __iGrid ) )
				{
					(Result).SetFloat(0.0f,__iGrid);	// Default, completely lit
				}
			}
			while( ( ++__iGrid < GridSize() ) && __fVarying);
		}
	}	
	

	
}