// Aqsis
// Copyright (c) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**
 * Copyright (C) 2006-2007  NTT DATA CORPORATION
 * 
 * Version: 1.0.0 2007/04/01
 *  
 */
package net.cellcomputing.himawari.library;

import static net.cellcomputing.himawari.library.CqStats.STATS_INC;
import static net.cellcomputing.himawari.library.EqIntIndex.GPR_crv;
import static net.cellcomputing.himawari.library.RiGlobal.QGetRenderContext;
import static net.cellcomputing.himawari.library.Float_h.FLT_MAX;
import static net.cellcomputing.himawari.library.EqVariableType.type_float;
import static net.cellcomputing.himawari.library.CqCurve.EssSplitDecision.*;
import static java.lang.Math.sqrt;
import net.cellcomputing.himawari.accessory.primitive.p_float;
import net.cellcomputing.himawari.library.types.CqVector3D;
import net.cellcomputing.himawari.library.types.CqVector4D;

/**
 * 
 * Abstract base class for all curve objects.  This class provides facilities
 * for accessing information common to all curves, such as width information.
 * 
 * @author NTT DATA Corporation
 */
public abstract strictfp class CqCurve extends CqSurface {
	
	/** Index of the width parameter within the m_aUserParams array of
	 * user parameters. */
	protected int m_widthParamIndex;
	
	/** Index of the constantwidth parameter within the m_aUserParams array
	 * of user parameters. */
	protected int m_constantwidthParamIndex;
	
//	enum EssSplitDecision
//	{
//	Split_Undecided = 0,
//	Split_Curve,
//	Split_Patch,
//	};
	protected class EssSplitDecision {
		
		public static final int Split_Undecided	= 0x0000;   ///< Invalid.
		public static final int Split_Curve		= 0x0001; 	///< Red Green and Blue channels.
		public static final int Split_Patch		= 0x0002;  	///< Alpha channel.
		
//		private int value;
//		
//		public int getValue()
//		{
//			return value;
//		}
//		public void setValue(int val)
//		{
//			value = val;
//		}
	}
	
	/** Stored decision about split to curves or patches.
	 */
	protected int m_splitDecision;
	
	protected static int hwidth = "width".hashCode();
	protected static int hcwidth = "constantwidth".hashCode();
	protected static int hp = "P".hashCode();
	protected static int hu = "u".hashCode();
	protected static int hn = "N".hashCode();
	protected static int hv = "v".hashCode();
	
	
	public CqCurve()
	{
		super();
		m_widthParamIndex = -1;
		m_constantwidthParamIndex = -1;
		m_splitDecision = Split_Undecided;
		
		STATS_INC( GPR_crv );
	}
	
	public CqCurve( final CqCurve from )
	{
		this.assignment( from );
		
		STATS_INC( GPR_crv );
	}
	
//	virtual ~CqCurve();
	
	/**
	 * Adds a primitive variable to the list of user parameters.  This method
	 * caches the indexes of the "width" and "constantwidth" parameters within
	 * the array of user parameters for later access.
	 *
	 * @param pParam        Pointer to the parameter to add.
	 */
	@Override
	public void AddPrimitiveVariable( CqParameter pParam )
	{
		// add the primitive variable using the superclass method
		super.AddPrimitiveVariable( pParam );
		
		// trap the indexes of "width" and "constantwidth" parameters
		if ( pParam.hash() == hwidth )
		{
			assert( m_widthParamIndex == -1 );
			m_widthParamIndex = m_aUserParams.size() - 1;
		}
		else if ( pParam.hash() == hcwidth )
		{
			assert( m_constantwidthParamIndex == -1 );
			m_constantwidthParamIndex = m_aUserParams.size() - 1;
		}
	}
	
	/**
	 * Calculates bounds for a CqCurve.
	 *
	 * NOTE: This method makes the same assumptions as 
	 * CqSurfacePatchBicubic::Bound() does about the convex-hull property of the
	 * curve.  This is fine most of the time, but the user can specify basis
	 * matrices like Catmull-Rom, which are non-convex.
	 *
	 * FIXME: Make sure that all hulls which reach this method are convex!
	 *
	 * @return CqBound object containing the bounds.
	 */
	public 	CqBound	Bound()
	{
		// Get the boundary in camera space.
		CqVector3D vecA = new CqVector3D( FLT_MAX, FLT_MAX, FLT_MAX );
		CqVector3D vecB = new CqVector3D( -FLT_MAX, -FLT_MAX, -FLT_MAX );
		float maxCameraSpaceWidth = 0;
		int nWidthParams = cVarying();
		for ( int i = 0; i < ( P() ).Size(); i++ )
		{
			// expand the boundary if necessary to accomodate the
			//  current vertex
			CqVector3D vecV = new CqVector3D( (CqVector4D)P().pValue_get( i , 0) );
			if ( vecV.x < vecA.x ) vecA.x( vecV.x );
			if ( vecV.y < vecA.y ) vecA.y( vecV.y );
			if ( vecV.x > vecB.x ) vecB.x( vecV.x );
			if ( vecV.y > vecB.y ) vecB.y( vecV.y );
			if ( vecV.z < vecA.z ) vecA.z( vecV.z );
			if ( vecV.z > vecB.z ) vecB.z( vecV.z );
			
			// increase the maximum camera space width of the curve if
			//  necessary
			if ( i < nWidthParams )
			{
				float camSpaceWidth = ((p_float)width().pValue_get( i , 0)).value;
				if ( camSpaceWidth > maxCameraSpaceWidth )
				{
					maxCameraSpaceWidth = camSpaceWidth;
				}
			}
			
		}
		
		// increase the size of the boundary by half the width of the
		//  curve in camera space
		vecA.assignSub( ( maxCameraSpaceWidth / 2.0f ) );
		vecB.assignAdd( ( maxCameraSpaceWidth / 2.0f ) );
		
		// return the boundary
		CqBound	B = new CqBound();
		B.vecMin().assignment( vecA );
		B.vecMax().assignment( vecB );
		return ( AdjustBoundForTransformationMotion( B ) );
		
	}
	
	/**
	 * CqCurve assignment operator.
	 *
	 * @param from  CqCurve to make this one equal to.
	 *
	 * @return Reference to (*this).
	 */
	public CqCurve assignment( final CqCurve from )
	{
	    super.assignment( (CqSurface)from );
	    return ( this );
	}
	
	/**
	 * Sets the default primitive variables.
	 *
	 * @param bUseDef_st
	 */
	@Override
	public void SetDefaultPrimitiveVariables( boolean bUseDef_st )
	{
	    // we don't want any default primitive variables.

	    // s,t are set to u,v for curves
	}
	
	/**
	 * Returns the approximate "length" of an edge of a grid in raster space.
	 *
	 * @return Approximate grid length.
	 */
	protected float GetGridLength()
	{
		// we want to find the number of micropolygons per grid - the default
	    //  is 256 (16x16 micropolygon grid).
	    float micropolysPerGrid = 256;
	    final int[] poptGridSize =
	        QGetRenderContext() .optCurrent().GetIntegerOption(
	            "limits", "gridsize"
	        );
	    if ( poptGridSize != null )
	    {
	        micropolysPerGrid =
	            poptGridSize[ 0 ] * poptGridSize[ 1 ];
	    }

	    // find the shading rate
	    float ShadingRate = pAttributes() .GetFloatAttribute(
	                              "System", "ShadingRate"
	                          ) [ 0 ];

	    // we assume that the grids are square and take the square root to find
	    //  the number of micropolygons along one side
	    float mpgsAlongSide = (float)sqrt( micropolysPerGrid );

	    // now, the number of pixels (raster space length) taken up by one
	    //  micropolygon is given by 1 / shading rate.  So, to find the length
	    //  in raster space of the edge of the micropolygon grid, we divide its
	    //  length (in micropolygons) by the shading rate
	    return mpgsAlongSide / ShadingRate;

	}
	
	/**
	 * Populates the "width" parameter if it is not already present (ie supplied
	 * by the user).  The "width" is populated either by the value of
	 * "constantwidth", or by the default object-space width 1.0.
	 */
	protected void PopulateWidth()
	{
		// if the width parameter has been supplied by the user then bail
	    //  immediately
	    if ( width() != null )
	        return ;

	    // otherwise, find the value to fill the width array with; default
	    //  value is 1.0 which can be overridden by the "constantwidth"
	    //  parameter
	    float widthvalue = 1.0f;
	    if ( constantwidth() != null )
	    {
	        widthvalue = ( ((p_float)constantwidth() .pValue_get( 0 , 0)).value );
	    }

	    // create and fill in the width array
	    CqParameterTypedVarying<p_float, p_float> widthP =
	        new CqParameterTypedVarying<p_float, p_float>(
	            "width", new EqVariableType(type_float), p_float.class, p_float.class
	        );
	    widthP.SetSize( cVarying() );
	    for ( int i = 0; i < cVarying(); i++ )
	    {
	        ((p_float)widthP.pValue_get( i , 0 )).value = widthvalue;
	    }

	    // add the width array to the curve as a primitive variable
	    AddPrimitiveVariable( widthP );
	}
	
	
	
    /** Returns a const reference to the "constantwidth" parameter, or
     * null if the parameter is not present. */
    public CqParameterTypedConstant<p_float, p_float> constantwidth()
    {
        if ( m_constantwidthParamIndex >= 0 )
        {
            return (CqParameterTypedConstant<p_float, p_float>)( m_aUserParams.get( m_constantwidthParamIndex ) );
        }
        else
        {
            return ( null );
        }
    }
    
    /** Returns whether the curve is diceable - at the moment, no Curves
     * are directly diceable since they're converted to patches just prior
     * to rendering. */
    @Override
    public boolean Diceable()
    {
        // OK, here the CqCubicCurveSegment line has two options:
        //  1. split into two more lines
        //  2. turn into a bilinear patch for rendering
        // We don't want to go turning into a patch unless absolutely
        // necessary, since patches cost more.  We only want to become a patch
        // if the current curve is "best handled" as a patch.  For now, I'm
        // choosing to define that the curve is best handled as a patch under
        // one or more of the following two conditions:
        //  1. If the maximum width is a significant fraction of the length of
        //      the line (width greater than 0.75 x length; ignoring normals).
        //  2. If the length of the line (ignoring the width; cos' it's
        //      covered by point 1) is such that it's likely a bilinear
        //      patch would be diced immediately if we created one (so that
        //      patches don't have to get split!).
        //  3. If the curve crosses the eye plane (m_fDiceable == false).

        // find the length of the CqLinearCurveSegment line in raster space
        if( m_splitDecision == Split_Undecided )
        {
			// AGG - 31/07/04
			// well, if we follow the above statagy we end up splitting into
			// far too many grids (with roughly 1 mpg per grid). so after
			// profiling a few scenes, the fastest method seems to be just
			// to convert to a patch immediatly.
			// we really need a native dice for curves but until that time
			// i reckon this is best.
			m_splitDecision = Split_Patch;


       /*     const CqMatrix & matCtoR = QGetRenderContext() .matSpaceToSpace(
                                           "camera", "raster",
											CqMatrix(), CqMatrix(),
											QGetRenderContextI().Time()
                                       );
            CqVector2D hull[ 2 ];     // control hull
            hull[ 0 ] = matCtoR * P().pValue( 0 )[0];
            hull[ 1 ] = matCtoR * P().pValue( 1 )[0];
            CqVector2D lengthVector = hull[ 1 ] - hull[ 0 ];
            float lengthraster = lengthVector.Magnitude();

            // find the maximum width of the line in raster space
            CqVector3D pp0 = hull[ 0 ] -
                             matCtoR * ( P().pValue( 0 )[0] + CqVector4D( width().pValue( 0 )[0], 0, 0, 1 ) );
            CqVector3D pp1 = hull[ 1 ] -
                             matCtoR * ( P().pValue( 1 )[0] + CqVector4D( width().pValue( 1 )[0], 0, 0, 1 ) );
            float width0 = pp0.Magnitude();
            float width1 = pp1.Magnitude();
            float maxwidthraster = ( width0 > width1 ) ? width0 : width1;

            // find the approximate "length" of a diced patch in raster space
            float gridlength = GetGridLength();

            // decide whether to split into more curve segments or a patch
            if (
                ( maxwidthraster > ( 0.75 * lengthraster ) ) ||
                ( lengthraster <= gridlength ) ||
                ( !m_fDiceable )
            )
            {
                // split into a patch
                m_splitDecision = Split_Patch;
            }
            else
            {
                // split into smaller curves
                m_splitDecision = Split_Curve;
            }
	*/
        }

        return false;
    }

    /** Determine whether the passed surface is valid to be used as a
     *  frame in motion blur for this surface.
     */
    @Override
    public boolean IsMotionBlurMatch( CqBasicSurface pSurf )
    {
        return( false );
    }
    
    /** Copy the information about splitting and dicing from the specified GPrim.
     * @param From A CqBasicSurface reference to copy the information from.
     */
    @Override
    public void CopySplitInfo( final CqBasicSurface From )
    {
        super.CopySplitInfo( From );
//        const CqCurve* pCurve = dynamic_cast<const CqCurve*>(From);
//        if( NULL != pCurve )
//            m_splitDecision = pCurve->m_splitDecision;
        
        //dynamic_cast JavaŁB
        try{
        	final CqCurve pCurve = (CqCurve)(From);
        	m_splitDecision = pCurve.m_splitDecision;
        }
        catch( ClassCastException e ){}
    }

    /** Returns a normal to the curve. */
    public boolean GetNormal( int index, CqVector3D normal )
    {
        if ( N() != null )
        {
            normal.assignment( (CqVector3D)N().pValue_get( index , 0) );
            return true;
        }
        else
        {
            normal.assignment( new CqVector3D( 0, 0, -1 ) );  // default camera normal
            return true;
        }
    }
    
    /** Returns a reference to the "width" parameter, or null if
     * the parameter is not present. */
    public CqParameterTypedVarying <p_float, p_float> width()
    {
        if ( m_widthParamIndex >= 0 )
        {
            return (CqParameterTypedVarying <p_float, p_float>)( m_aUserParams.get( m_widthParamIndex ) );
        }
        else
        {
            return ( null );
        }

    }
  
	
}
