<?php
/**
 * LDAP.php
 *
 * PHP 5.4
 *
 * @package Auth
 *
 * @author Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 * @copyright 2020 SECIOSS, INC.
 */
namespace Secioss\Simple;

use PEAR;

/**
 * Secioss\Simple\LDAP
 */
class LDAP implements Storage
{
    /**
     * LDAPクラスのコンストラクタ
     *
     * @access public
     *
     * @param mixed $options LDAPの設定
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function __construct($options)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);

        $rc = 0;
        if (strtoupper(substr(PHP_OS, 0, 3)) != 'WIN') {
            $rc = $this->connect();
            register_shutdown_function([&$this, 'disconnect']);
        }

        return $rc;
    }

    /**
     * LDAPサーバに接続する
     *
     * @access public
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function connect()
    {
        $this->conn = @ldap_connect($this->options['uri']);
        ldap_set_option($this->conn, LDAP_OPT_PROTOCOL_VERSION, 3);
        if (!@ldap_bind($this->conn, $this->options['binddn'], $this->options['bindpw'])) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        return 0;
    }

    /**
     * LDAPサーバへの接続を切断する
     *
     * @access public
     */
    public function disconnect()
    {
        @ldap_unbind($this->conn);
    }

    /**
     * LDAPサーバからユーザ情報を取得する
     *
     * @param string ユーザ名
     * @param mixed $username
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function fetchData($username)
    {
        if (!$username || strlen($username) > 255) {
            return PEAR::raiseError('Invalid user id', AUTO_LOGIN_INVALID_VALUE);
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        $filter = '('.$this->options['userattr'].'='.preg_replace('/([*()])/', '\\$1', $username).')';
        if ($this->options['userfilter']) {
            $filter = '(&'.$filter.$this->options['userfilter'].')';
        }
        $res = @ldap_search($this->conn, $this->options['basedn'], $filter);
        if ($res == false) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        $num = ldap_count_entries($this->conn, $res);

        if ($num == 0) {
            return PEAR::raiseError("User doesn't exist", AUTO_LOGIN_NO_USER);
        } elseif ($num != 1) {
            return PEAR::raiseError("User isn't unique", AUTO_LOGIN_ERROR);
        }

        $entries = ldap_get_entries($this->conn, $res);
        $entry = $entries[0];

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        $this->id = $entry['dn'];

        $this->prop = [];
        for ($i = 0; $i < $entry['count']; $i++) {
            $key = $entry[$i];

            if ($entry[$key]['count'] == 1) {
                $this->prop[$key] = $entry[$key][0];
            } else {
                $v = [];
                for ($j = 0; $j < $entry[$key]['count']; $j++) {
                    $v[] = $entry[$key][$j];
                }
                $this->prop[$key] = $v;
            }
        }

        if (isset($this->prop[$this->options['statusattr']])) {
            $this->status = $this->prop[$this->options['statusattr']];
        }

        return true;
    }

    /**
     * ユーザの認証を行う
     *
     * @access public
     *
     * @param string $password パスワード
     *
     * @return string 0:成功 1:失敗
     */
    public function auth($password)
    {
        if (!$password) {
            return 1;
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        $rc = @ldap_bind($this->conn, $this->id, $password) ? 0 : 1;
        @ldap_bind($this->conn, $this->options['binddn'], $this->options['bindpw']);

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        return $rc;
    }

    /**
     * ステータスを取得する。
     *
     * @access public
     *
     * @return string ステータス
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * テナント情報取得
     *
     * @access public
     *
     * @param null|string $tenant テナントID
     *
     * @return mixed テナント情報:正常終了 PEAR_Error:エラー
     */
    public function getTenants($tenant = null)
    {
        if ($tenant) {
            $tenant = addcslashes($tenant, '*()`\\');
            $filter = '('.$this->options['tenantattr']."=$tenant)";
        } else {
            $filter = '('.$this->options['tenantattr'].'=*)';
        }
        if ($this->options['tenantfilter']) {
            $filter = '(&'.$filter.$this->options['tenantfilter'].')';
        }
        $res = @ldap_list($this->conn, $this->options['basedn'], $filter, [$this->options['tenantattr'], 'seciossallowedfunction', 'seciossconfigserializeddata', 'seciossconfigserializeddata;x-type-html', 'mail', 'seciosstenantmaxusers;x-type-device']);
        if ($res == false) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        $entries = ldap_get_entries($this->conn, $res);
        $tenants = [];
        for ($i = 0; $i < $entries['count']; $i++) {
            $tenants[$entries[$i][$this->options['tenantattr']][0]] = ['func' => isset($entries[$i]['seciossallowedfunction']) ? $entries[$i]['seciossallowedfunction'] : []];
            if (isset($entries[$i]['seciossconfigserializeddata'])) {
                $tenantconf = unserialize($entries[$i]['seciossconfigserializeddata'][0]);
                if ($tenantconf) {
                    foreach (array_keys($tenantconf) as $key) {
                        if (!is_array($tenantconf[$key])) {
                            continue;
                        }
                        $tenants[$entries[$i][$this->options['tenantattr']][0]][$key] = $tenantconf[$key];
                    }
                }
            }
            if (isset($entries[$i]['seciossconfigserializeddata;x-type-html'])) {
                $tpl_html = unserialize($entries[$i]['seciossconfigserializeddata;x-type-html'][0]);
                if ($tpl_html) {
                    foreach (array_keys($tpl_html) as $key) {
                        $tenants[$entries[$i][$this->options['tenantattr']][0]]['template']['v2'][$key] = $tpl_html[$key];
                    }
                }
            }

            if (isset($entries[$i]['mail'])) {
                $tenants[$entries[$i][$this->options['tenantattr']][0]]['mail'] = $entries[$i]['mail'];
            }
            if (isset($entries[$i]['mail;x-type-admin'])) {
                $tenants[$entries[$i][$this->options['tenantattr']][0]]['mail;x-type-admin'] = $entries[$i]['mail;x-type-admin'];
            }
            if (isset($entries[$i]['seciosstenantmaxusers;x-type-device'])) {
                $tenants[$entries[$i][$this->options['tenantattr']][0]]['seciosstenantmaxusers;x-type-device'] = $entries[$i]['seciosstenantmaxusers;x-type-device'][0];
            }
        }

        return $tenants;
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * @access public
     *
     * @param array $prop プロパティ
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function setProp($prop)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        if (!@ldap_mod_replace($this->conn, $this->id, $prop)) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        return true;
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * @access public
     *
     * @param array $prop プロパティ
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function delProp($prop)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        if (!@ldap_mod_del($this->conn, $this->id, $prop)) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        return true;
    }

    /**
     * 暗号化されているシークレットを復号化して取得する
     *
     * @access public
     *
     * @return string シークレット
     */
    public function getSecret()
    {
        $secret = '';
        $sslkey = [];
        if (isset($this->options['privatekey'])) {
            array_push($sslkey, $this->options['privatekey']);
        }
        if (isset($this->options['oldprivatekey'])) {
            array_push($sslkey, $this->options['oldprivatekey']);
        }
        $aeskey = isset($this->options['keyfile']) ? Crypt::getSecretKey($this->options['keyfile']) : null;

        if (isset($this->prop[$this->options['secretattr']])) {
            $encrypt = $this->prop[$this->options['secretattr']];
            $secret = Util::decrypt($encrypt, $sslkey, $aeskey);
        }

        return $secret;
    }

    /**
     * シークレットを暗号化してLDAPに格納する
     *
     * @access public
     *
     * @param string      $secret
     * @param null|string $pin
     * @param null|string $deviceid
     * @param null|string $device
     * @param null|int    $otplen
     * @param null|int    $timewindow
     * @param null|string $os
     * @param null|string $ip
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function setSecret($secret, $pin = null, $deviceid = null, $device = null, $otplen = null, $timewindow = null, $os = null, $ip = null)
    {
        $secret_exists = false;
        $agent = '';

        if (!$secret || strlen($secret) > 255) {
            return PEAR::raiseError('Invalid secret', AUTO_LOGIN_INVALID_VALUE);
        }

        if ($device && !$deviceid) {
            return PEAR::raiseError('No device id', AUTO_LOGIN_INVALID_VALUE);
        }

        if ($deviceid) {
            $agent = 'computer';
            if (preg_match('/iPad/', $_SERVER['HTTP_USER_AGENT'])) {
                $agent = 'ipad';
            } elseif (preg_match('/iPhone/', $_SERVER['HTTP_USER_AGENT'])) {
                $agent = 'iphone';
            } elseif (preg_match('/Android/', $_SERVER['HTTP_USER_AGENT'])) {
                $agent = 'android';
            }
            if ($os) {
                if (preg_match('/Windows/', $os)) {
                    $agent = 'windows';
                } elseif (preg_match('/Mac/', $os)) {
                    $agent = 'mac';
                } elseif (preg_match('/Linux/', $os)) {
                    $agent = 'linux';
                }
            }
        }

        $prop = [];
        if (array_search('seciossOtpUser', $this->prop['objectclass']) === false) {
            $prop['objectclass'] = $this->prop['objectclass'];
            array_push($prop['objectclass'], 'seciossOtpUser');
        }

        $sslkey = isset($this->options['publickey']) ? $this->options['publickey'] : null;
        $aeskey = isset($this->options['keyfile']) ? Crypt::getSecretKey($this->options['keyfile']) : null;

        $encrypt = Util::encrypt($secret, $sslkey, $aeskey);
        if (PEAR::isError($encrypt)) {
            return $encrypt;
        }
        if ($device) {
            $attr = $this->options['secretattr'].";x-dev-$device";
            $hashid = md5($deviceid);
            $secrets = isset($this->prop[$attr]) ? Util::to_array($this->prop[$attr]) : [];
            $match = false;
            for ($i = 0; $i < count($secrets); $i++) {
                if (preg_match("/^$hashid#/", $secrets[$i])) {
                    $secrets[$i] = $hashid.'#'.$encrypt;
                    $match = true;
                    break;
                }
            }
            if ($match) {
                $secret_exists;
            } else {
                array_push($secrets, $hashid.'#'.$encrypt);
            }
            $prop[$attr] = $secrets;
        } else {
            $prop[$this->options['secretattr']] = $encrypt;
        }

        if ($pin) {
            $encrypt = Util::encrypt($pin, $sslkey, $aeskey);
            if (PEAR::isError($encrypt)) {
                return $encrypt;
            }
            if ($device) {
                $attr = "seciossotpsecretpin;x-dev-$device";
                $hashid = md5($deviceid);
                $pins = isset($this->prop[$attr]) ? Util::to_array($this->prop[$attr]) : [];
                $match = false;
                for ($i = 0; $i < count($pins); $i++) {
                    if (preg_match("/^$hashid#/", $pins[$i])) {
                        $pins[$i] = $hashid.'#'.$encrypt;
                        $match = true;
                        break;
                    }
                }
                if (!$match) {
                    array_push($pins, $hashid.'#'.$encrypt);
                }
                $prop[$attr] = $pins;
            } else {
                $prop['seciossotpsecretpin'] = $encrypt;
            }
        }
        if ($deviceid) {
            $attr = "seciossdeviceid;x-dev-$device";
            $devids = Util::to_array(isset($this->prop[$attr]) ? $this->prop[$attr] : []);
            $match = false;
            for ($i = 0; $i < count($devids); $i++) {
                if (preg_match('/^'.$deviceid.'#/', $devids[$i])) {
                    $devids[$i] = preg_replace("/^$deviceid#(active|inactive)=.*$/i", "$deviceid#\\1=$agent+".date('Y/m/d H:i:s')."+$ip", $devids[$i]);
                    $match = true;
                    break;
                }
            }
            if (!$match) {
                array_push($devids, $deviceid."#inactive=$agent+".date('Y/m/d H:i:s')."+$ip");
            }
            $prop[$attr] = $devids;
        }
        if ($otplen) {
            $prop['seciossotpoption'] = "otplen=$otplen";
        }
        if ($timewindow) {
            if (isset($prop['seciossotpoption'])) {
                $prop['seciossotpoption'] .= "#timewindow=$timewindow";
            } else {
                $prop['seciossotpoption'] = "timewindow=$timewindow";
            }
            $file = null;
            if (isset($_GET['app'])) {
                $file = $_GET['app'];
            } elseif (preg_match('/\/(secret|qrsecret|websecret)\.php$/', $_SERVER['SCRIPT_NAME'], $matches)) {
                $file = $matches[1];
            }
            $type = null;
            switch ($file) {
                case 'secret':
                    $type = 'hard';
                    break;
                case 'qrsecret':
                    if (isset($_GET['st'])) {
                        $type = 'soft';
                    } else {
                        $type = 'slinkpass';
                    }
                    break;
                case 'websecret':
                    $type = 'web';
                    break;
            }
            if ($type) {
                $prop['seciossotpoption'] .= "#type=$type";
            }
        }

        $rc = $this->setProp($prop);
        if (PEAR::isError($rc)) {
            return $rc;
        } elseif ($secret_exists) {
            return PEAR::raiseError('Secret has been changed', AUTO_LOGIN_IN_HISTORY);
        }
        return $rc;
    }

    /**
     * optionsにデフォルト値を設定する
     *
     * @access protected
     */
    protected function _setDefaults()
    {
        $this->options['uri'] = 'ldap://localhost';
        $this->options['binddn'] = '';
        $this->options['bindpw'] = '';
        $this->options['basedn'] = '';
        $this->options['tenantattr'] = 'o';
        $this->options['tenantfilter'] = '(&(objectClass=organization)(!(seciossTenantStatus=inactive)))';
        $this->options['userattr'] = 'uid';
        $this->options['userfilter'] = '';
        $this->options['statusattr'] = 'seciossaccountstatus';
        $this->options['secretattr'] = 'seciossotpinitsecret';
    }

    /**
     * optionsに値を設定する
     *
     * @access protected
     *
     * @param  array
     * @param mixed $array
     */
    protected function _parseOptions($array)
    {
        if (is_array($array)) {
            foreach ($array as $key => $value) {
                $this->options[$key] = $value;
            }
        }
    }
}
