<?php
/**
 * LDAP.php
 *
 * PHP 5.4
 *
 * @package Auth
 *
 * @author Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 * @copyright 2020 SECIOSS, INC.
 */
namespace Secioss\Enterprise;

use PEAR;

define('PASSWORD_SYNTAX', '2.5.4.35');

/**
 * Secioss\Enterprise\LDAP
 */
class LDAP extends \Secioss\Simple\LDAP implements \Secioss\Enterprise\Storage
{
    /**
     * LDAPクラスのコンストラクタ
     *
     * @access public
     *
     * @param mixed $options LDAPの設定
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function __construct($options)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);

        if ($this->options['pwhash'] == 'AD') {
            $this->options['basedn'] = mb_convert_encoding($this->options['basedn'], 'SJIS');
        }

        $rc = parent::__construct($options);

        return $rc;
    }

    /**
     * LDAPサーバからユーザ情報を取得する
     *
     * @param string $username ユーザ名
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function fetchData($username)
    {
        $rc = parent::fetchData($username);
        if (PEAR::isError($rc)) {
            return $rc;
        }

        if (!isset($this->prop['seciossloginid'])) {
            $this->prop['seciossloginid'] = $username;
        }
        $this->prop['seciosssystemid'] = preg_replace('/@[^@]+$/', '', $username);

        if (isset($this->prop[$this->options['loginattr']])) {
            $loginid = $this->prop[$this->options['loginattr']];
            if (is_array($loginid)) {
                $this->loginid = $loginid[0];
            } else {
                $this->loginid = $loginid;
            }
        }

        if (isset($this->prop[$this->options['pwdlockoutattr']])) {
            if ($this->prop[$this->options['pwdlockoutattr']] == 'TRUE') {
                $this->pwdlockout = true;
            } else {
                $this->pwdlockout = false;
            }
        }

        $tenant = '';
        if (preg_match('/^[^@]+@(.*)$/', $username, $matches)) {
            $tenant = $matches[1];
            $tenant = preg_replace('/([*()])/', '\\$1', $tenant);
        }

        if (!$tenant && preg_match('/[^,]+,[^,]+,'.$this->options['tenantattr'].'=([^,]+),'.$this->options['basedn'].'/i', $this->id, $matches)) {
            $tenant = $matches[1];
        }

        if ($this->options['pwdpolicydb']) {
            $rc = $this->getPwdPolicy($tenant);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        }

        return true;
    }

    /**
     * ユーザの認証を行う
     *
     * @access public
     *
     * @param string パスワード
     * @param mixed $password
     *
     * @return string 0:成功 1:失敗
     */
    public function auth($password)
    {
        $rc = parent::auth($password);
        if ($rc) {
            return $rc;
        }

        if ($this->options['oldpwdauth'] && isset($this->prop[$this->options['pwdmustchangeattr']]) && $this->prop[$this->options['pwdmustchangeattr']] == 'TRUE') {
            if (isset($this->prop[$this->options['pwdhistoryattr']])) {
                $pwdhistory = Util::to_array($this->prop[$this->options['pwdhistoryattr']]);
                if (!preg_match('/#([^#]+)$/', $pwdhistory[0], $matches)) {
                    return 1;
                }
                $oldpasswd = $matches[1];

                $hash = '';
                if (preg_match('/^{([^}]+)}/', $oldpasswd, $matches)) {
                    $hash = $matches[1];
                }

                if ($this->cmpPasswd($password, $oldpasswd, $hash)) {
                    return 0;
                }
            }
        }

        return $rc;
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * @access public
     *
     * @param array $prop プロパティ
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function setProp($prop)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if ($this->options['api_url']) {
            $rc = $this->_rest_modify($this->id, $prop);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        } else {
            $rc = parent::setProp($prop);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        }

        return true;
    }

    /**
     * プロパティへのアクセサ(W)
     *
     * @access public
     *
     * @param array $prop プロパティ
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function delProp($prop)
    {
        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if ($this->options['api_url']) {
            $rc = $this->_rest_modify($this->id, $prop, 'delete');
            if (PEAR::isError($rc)) {
                return $rc;
            }
        } else {
            $rc = parent::delProp($prop);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        }

        return true;
    }

    /**
     * ログインIDを取得する。
     *
     * @access public
     *
     * @return string ログインID
     */
    public function getLoginId()
    {
        return $this->loginid;
    }

    /**
     * 暗号化されているパスワードを復号化して取得する
     *
     * @access public
     *
     * @param null|mixed $app
     *
     * @return string パスワード
     */
    public function getPassword($app = null)
    {
        $password = '';
        $regex_app = addcslashes($app, '.*+?[]()|^$\\/');

        if (isset($this->prop[$this->options['encryptpwdattr']])) {
            $encryptpwds = Util::to_array($this->prop[$this->options['encryptpwdattr']]);

            $sslkey = [];
            if (isset($this->options['privatekey'])) {
                array_push($sslkey, $this->options['privatekey']);
            }
            if (isset($this->options['oldprivatekey'])) {
                array_push($sslkey, $this->options['oldprivatekey']);
            }
            $aeskey = isset($this->options['keyfile']) ? Crypt::getSecretKey($this->options['keyfile']) : null;

            for ($i = 0; $i < count($encryptpwds); $i++) {
                if ($app) {
                    if (preg_match('/^{'.$regex_app.'}(.+)/i', $encryptpwds[$i], $matches)) {
                        $password = Util::decrypt($matches[1], $sslkey, $aeskey);
                        break;
                    }
                }
                if (!preg_match('/^{[^}]+}/', $encryptpwds[$i])) {
                    $password = Util::decrypt($encryptpwds[$i], $sslkey, $aeskey);
                }
            }
        }

        return $password;
    }

    /**
     * パスワードを暗号化してLDAPに格納する
     *
     * @access public
     *
     * @param string パスワード
     * @param null|mixed $password
     * @param null|mixed $app
     * @param mixed      $init
     * @param null|mixed $random
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function setPassword($password = null, $app = null, $init = false, $random = null)
    {
        $regex_app = addcslashes($app, '.*+?[]()|^$\\/');

        if (!$this->id) {
            return PEAR::raiseError('Must fetch data', AUTO_LOGIN_ERROR);
        }

        if (!$password || strlen($password) > 255) {
            return PEAR::raiseError('Invalid password', AUTO_LOGIN_INVALID_VALUE);
        }

        $sslkey = isset($this->options['publickey']) ? $this->options['publickey'] : null;
        $aeskey = isset($this->options['keyfile']) ? Crypt::getSecretKey($this->options['keyfile']) : null;

        if ($this->options['encryptpwdattr']) {
            $encrypt = Util::encrypt($password, $sslkey, $aeskey);
            if (PEAR::isError($encrypt)) {
                return $encrypt;
            }

            $encryptpwds = [];
            if (isset($this->prop[$this->options['encryptpwdattr']])) {
                $encryptpwds = Util::to_array($this->prop[$this->options['encryptpwdattr']]);
            }
        }

        if ($this->options['randompwdattr']) {
            if (!$random) {
                $pwlen = strlen($password);
                $random = Util::random($pwlen);
            }
            $rndencrypt = Util::encrypt($random, $sslkey, $aeskey);
            if (PEAR::isError($rndencrypt)) {
                return $rndencrypt;
            }

            $info[$this->options['randompwdattr']] = $rndencrypt;
        }

        if (!$app) {
            if ($this->options['pwdattr']) {
                $hash = $this->options['pwhash'];
                $tab = '';
                if ($hash == 'CRYPT' || $hash == 'MD5' || $hash == 'SHA' || $hash == 'SSHA') {
                    $tab = '{'.$hash.'}';
                }
                $info[$this->options['pwdattr']] = $tab.(Util::hashPasswd($password));
            }

            if ($this->options['encryptpwdattr']) {
                $match = false;
                for ($i = 0; $i < count($encryptpwds); $i++) {
                    if (!preg_match('/^{[^}]+}/', $encryptpwds[$i])) {
                        $encryptpwds[$i] = $encrypt;
                        $match = true;
                    }
                }
                if (!$match) {
                    array_push($encryptpwds, $encrypt);
                }
                $info[$this->options['encryptpwdattr']] = $encryptpwds;
            }

            $oldpwdtime = '';
            if ($this->options['pwdtimeattr']) {
                $info[$this->options['pwdtimeattr']] = date('YmdHis').'Z';
                if (isset($this->prop[$this->options['pwdtimeattr']])) {
                    $oldpwdtime = $this->prop[$this->options['pwdtimeattr']];
                }
            }

            if ($this->options['pwdhistoryattr']) {
                $oldpasswd = isset($this->prop[$this->options['pwdattr']]) ? $this->prop[$this->options['pwdattr']] : '';
                $oldpwdtime = $oldpwdtime ? $oldpwdtime : '19700101000000Z';

                if ($oldpasswd) {
                    $pwdhistory = [];
                    $oldpasswd = $oldpwdtime.'#'.PASSWORD_SYNTAX.'#'.strlen($oldpasswd).'#'.$oldpasswd;
                    $pwinhistory = $this->options['pwinhistory'];
                    if ($pwinhistory) {
                        if (isset($this->prop[$this->options['pwdhistoryattr']])) {
                            $pwdhistory = Util::to_array($this->prop[$this->options['pwdhistoryattr']]);
                        }

                        if (count($pwdhistory) >= $pwinhistory - 1) {
                            for ($i = 0; $i < count($pwdhistory) - $pwinhistory + 1; $i++) {
                                array_pop($pwdhistory);
                            }
                        }
                    }
                    array_unshift($pwdhistory, $oldpasswd);

                    $info[$this->options['pwdhistoryattr']] = $pwdhistory;
                }
            }

            if ($this->options['pwdmustchangeattr']) {
                $info[$this->options['pwdmustchangeattr']] = $init ? 'TRUE' : 'FALSE';
                $info[$this->options['pwdexpwarnedattr']] = 'FALSE';
                $info[$this->options['pwdexpiredattr']] = 'FALSE';
            }
        } else {
            if ($this->options['encryptpwdattr']) {
                $match = false;
                for ($i = 0; $i < count($encryptpwds); $i++) {
                    if (preg_match('/^{'.$regex_app.'}/i', $encryptpwds[$i])) {
                        $encryptpwds[$i] = '{'.$app."}$encrypt";
                        $match = true;
                    }
                }
                if (!$match) {
                    array_push($encryptpwds, '{'.$app."}$encrypt");
                }
                $info[$this->options['encryptpwdattr']] = $encryptpwds;
            }
        }

        if ($this->options['api_url']) {
            $rc = $this->_rest_modify($this->id, $info);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        } else {
            if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
                $this->connect();
            }

            if (!@ldap_mod_replace($this->conn, $this->id, $info)) {
                if (ldap_errno($this->conn) == 19) {
                    return PEAR::raiseError(ldap_error($this->conn), AUTO_LOGIN_INVALID_VALUE);
                } else {
                    return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
                }
            }

            if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
                $this->disconnect();
            }
        }

        return true;
    }

    public function deletePassword($app)
    {
        if (!$app) {
            return true;
        }
        $regex_app = addcslashes($app, '.*+?[]()|^$\\/');

        if (isset($this->prop[$this->options['encryptpwdattr']])) {
            $encryptpwds = Util::to_array($this->prop[$this->options['encryptpwdattr']]);

            for ($i = 0; $i < count($encryptpwds); $i++) {
                if (preg_match('/^{'.$regex_app.'}/i', $encryptpwds[$i])) {
                    if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
                        $this->connect();
                    }

                    if (!@ldap_mod_del($this->conn, $this->id, [$this->options['encryptpwdattr'] => $encryptpwds[$i]])) {
                        return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
                    }

                    if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
                        $this->disconnect();
                    }

                    return true;
                }
            }
        }

        return true;
    }

    /**
     * パスワード変更時間を取得する
     *
     * @access public
     *
     * @return string パスワード変更時間（秒）
     */
    public function getPwdChangedTime()
    {
        if (!$this->options['pwdtimeattr']) {
            return 0;
        }

        $pwdtime = $this->prop[$this->options['pwdtimeattr']];

        if (!$pwdtime) {
            return 0;
        }

        if (!preg_match('/^([0-9]{4})([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{2})Z$/', $pwdtime, $matches)) {
            return 0;
        }

        return mktime($matches[4], $matches[5], $matches[6], $matches[2], $matches[3], $matches[1]);
    }

    /**
     * パスワード履歴を取得する
     *
     * @access public
     *
     * @return mixed パスワード履歴
     */
    public function getPwdHistory()
    {
        if (!$this->options['pwdhistoryattr'] || !isset($this->prop[$this->options['pwdhistoryattr']])) {
            return [];
        }

        $pwdhistory = Util::to_array($this->prop[$this->options['pwdhistoryattr']]);

        for ($i = 0; $i < count($pwdhistory); $i++) {
            $pwdhistory[$i] = preg_replace('/^[^#]+#[^#]+#[^#]+#/', '', $pwdhistory[$i]);
        }
        array_unshift($pwdhistory, $this->prop[$this->options['pwdattr']]);

        return $pwdhistory;
    }

    /**
     * ロックアウトを取得する。
     *
     * @access public
     *
     * @return string ロックアウト
     */
    public function getPwdLockout()
    {
        return $this->pwdlockout;
    }

    public function isPwdMustChange()
    {
        if (isset($this->prop[$this->options['pwdmustchangeattr']]) &&
            $this->prop[$this->options['pwdmustchangeattr']] == 'TRUE') {
            return true;
        }
        return false;
    }

    /**
     * プロファイル一覧を取得する
     *
     * @access public
     *
     * @return array プロファイルDN一覧
     */
    public function getProfiles()
    {
        $profiles = [];

        $profiles = array_merge($profiles, (array) $this->get('seciossbusinessrole'));
        $profiles = array_merge($profiles, (array) $this->get('seciossbusinessrole;x-perm-group'));

        return $profiles;
    }

    /**
     * プロファイルのカスタムコンフィグを取得する
     *
     * @access public
     *
     * @param null|mixed $tenant
     *
     * @return array プロファイルのカスタムコンフィグ
     */
    public function getProfileConf($tenant = null)
    {
        $profileconf = [];

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        $filter = null;
        $profiles = $this->getProfiles();
        foreach ($profiles as $profile) {
            if (preg_match('/^([^=]+=[^,]+),/', $profile, $matches)) {
                if ($filter) {
                    $filter = '(|('.$matches[1].')'.$filter.')';
                } else {
                    $filter = '('.$matches[1].')';
                }
            }
        }
        if (!$filter) {
            return $profileconf;
        }

        $filter = "(&(&(objectClass=seciossRole)(seciossConfigSerializedData=*))$filter)";
        $basedn = $tenant ? $this->options['tenantattr']."=$tenant,".$this->options['basedn'] : $this->options['basedn'];
        $res = @ldap_search($this->conn, $basedn, $filter);
        if ($res == false) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }

        $entries = ldap_get_entries($this->conn, $res);
        for ($i = 0; $i < $entries['count']; $i++) {
            $seciossconfigserializeddata = $entries[$i]['seciossconfigserializeddata'][0];
            $tmpcpnf = unserialize($seciossconfigserializeddata);
            $profileconf = hashMerge($profileconf, $tmpcpnf);
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        return $profileconf;
    }

    /**
     * Secioss_User.php
     *
     * @param null|mixed $tenant
     */
    public function getPwdPolicy($tenant = null)
    {
        $pwdpolicy = null;

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->connect();
        }

        $filter = null;
        $profiles = $this->getProfiles();
        foreach ($profiles as $profile) {
            if (preg_match('/^([^=]+=[^,]+),/', $profile, $matches)) {
                if ($filter) {
                    $filter = '(|('.$matches[1].')'.$filter.')';
                } else {
                    $filter = '('.$matches[1].')';
                }
            }
        }
        if ($filter) {
            $filter = "(&(&(objectClass=seciossRole)(seciossPwdPolicyEnabled=TRUE))$filter)";
            $res = @ldap_search($this->conn, $tenant ? $this->options['tenantattr']."=$tenant,".$this->options['basedn'] : $this->options['basedn'], $filter);
            if ($res == false) {
                return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
            }

            $priority = 0;
            $entries = ldap_get_entries($this->conn, $res);
            for ($i = 0; $i < $entries['count']; $i++) {
                if ($entries[$i]['seciossrolespecification'][0] > $priority) {
                    $pwdpolicy = $entries[$i];
                    $priority = $entries[$i]['seciossrolespecification'][0];
                }
            }
        }

        if (!$pwdpolicy && $tenant) {
            $res = @ldap_search($this->conn, $this->options['basedn'], '(&(objectClass=organization)(&(seciossPwdPolicyEnabled=TRUE)('.$this->options['tenantattr']."=$tenant)))");
            if ($res == false) {
                return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
            }

            $num = ldap_count_entries($this->conn, $res);
            if ($num > 0) {
                $entries = ldap_get_entries($this->conn, $res);
                $pwdpolicy = $entries[0];
            }
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) == 'WIN') {
            $this->disconnect();
        }

        if (!$pwdpolicy) {
            return true;
        }

        $this->options['pwminlen'] = 0;
        $this->options['pwmaxlen'] = 255;
        $this->options['pwallow'] = null;
        $this->options['pwallowlimit'] = 0;
        $this->options['pwdeny'] = null;
        $this->options['pwminage'] = 0;
        $this->options['pwinhistory'] = 0;
        $this->options['pwstrength'] = 0;
        $this->options['pwprohibitattr'] = null;

        for ($i = 0; $i < $pwdpolicy['count']; $i++) {
            $key = $pwdpolicy[$i];
            if ($pwdpolicy[$key]['count'] == 1) {
                $v = $pwdpolicy[$key][0];
            } else {
                $v = [];
                for ($j = 0; $j < $pwdpolicy[$key]['count']; $j++) {
                    $v[] = $pwdpolicy[$key][$j];
                }
            }
            if (!is_array($v) && preg_match('/^ *$/', $v)) {
                continue;
            }

            switch ($key) {
                case 'pwdminlength':
                    $this->options['pwminlen'] = $v;
                    break;
                case 'seciosspwdmaxlength':
                    $this->options['pwmaxlen'] = $v;
                    break;
                case 'seciosspwdallowedchars':
                    $v = is_array($v) ? $v : [$v];
                    $this->options['pwallow'] = $v;
                    break;
                case 'seciosspwddeniedchars':
                    $v = is_array($v) ? $v : [$v];
                    $this->options['pwdeny'] = $v;
                    break;
                case 'pwdminage':
                    $this->options['pwminage'] = $v;
                    break;
                case 'pwdinhistory':
                    $this->options['pwinhistory'] = $v;
                    break;
                case 'pwdcheckquality':
                    $this->options['pwstrength'] = $v;
                    break;
                case 'seciosspwdserializeddata':
                    $v = unserialize($v);
                    $this->options['pwprohibitattr'] = isset($v['pwprohibitattr']) ? implode(',', $v['pwprohibitattr']) : null;
                    $this->options['pwallowlimit'] = isset($v['pwallowlimit']) ? $v['pwallowlimit'] : 0;
                    break;
            }
        }

        return true;
    }

    /**
     * seciossDeviceId;x-dev-$device: $deviceid#$status=$os+DATE+$ip#user#note#appver
     *
     * @param mixed $device
     * @param mixed $deviceid
     * @param mixed $status
     * @param mixed $os
     * @param mixed $ip
     * @param mixed $note
     * @param mixed $appver
     */
    public function setDevice($device, $deviceid, $status, $os, $ip, $note, $appver)
    {
        $prop = [];
        $attr = "seciossdeviceid;x-dev-$device";
        $devids = Util::to_array(isset($this->prop[$attr]) ? $this->prop[$attr] : []);
        $match = false;
        for ($i = 0; $i < count($devids); $i++) {
            if (preg_match('/^'.$deviceid.'#([^=]+)=/', $devids[$i], $matches)) {
                $current_status = $matches[1];
                if (strcmp('active', $current_status) === 0) {
                    $status = 'active';
                } elseif (strcmp('inactive', $current_status) === 0) {
                    return PEAR::raiseError('Device is inactive', AUTO_LOGIN_DEVICE_INACTIVE);
                }
                $devids[$i] = preg_replace("/^$deviceid#(active|pending)=.*$/i", "$deviceid#$status=$os+".date('Y/m/d H:i:s')."+$ip###$appver", $devids[$i]);
                $match = true;
                break;
            }
        }
        if (!$match) {
            array_push($devids, $deviceid."#$status=$os+".date('Y/m/d H:i:s')."+$ip###$appver");
        }
        $prop[$attr] = $devids;
        $rc = $this->setProp($prop);
        if (PEAR::isError($rc)) {
            return $rc;
        } elseif ($match) {
            return PEAR::raiseError('Device has been updated', AUTO_LOGIN_IN_HISTORY);
        }
        return $rc;
    }

    public function deleteDevice($device, $deviceid)
    {
        $prop = [];
        $attr = "seciossdeviceid;x-dev-$device";
        $devids = Util::to_array(isset($this->prop[$attr]) ? $this->prop[$attr] : []);
        $match = false;
        for ($i = 0; $i < count($devids); $i++) {
            if (preg_match('/^'.$deviceid.'#([^=]+)=/', $devids[$i], $matches)) {
                $match = true;
                unset($devids[$i]);
                continue;
            }
        }
        if (!$match) {
            return PEAR::raiseError('Device id is not exist', AUTO_LOGIN_DEVICE_NOEXIST);
        }
        $prop[$attr] = $devids;
        $rc = $this->setProp($prop);
        if (PEAR::isError($rc)) {
            return $rc;
        }
    }

    public function getDeviceCurrentNum($ltype, $dtype, $newid, $tenant)
    {
        $num = 0;
        $ltype_list = ['device', 'user', 'deviceperuser'];
        if (!$ltype || !in_array($ltype, $ltype_list)) {
            return PEAR::raiseError('Invalid license type', AUTO_LOGIN_INVALID_VALUE);
        }
        if (!$dtype) {
            return PEAR::raiseError('Invalid device type', AUTO_LOGIN_INVALID_VALUE);
        }
        if (!$newid) {
            return PEAR::raiseError('Invalid device id', AUTO_LOGIN_INVALID_VALUE);
        }

        $basedn = $this->options['basedn'];
        if ($tenant) {
            $basedn = $this->options['tenantattr']."=$tenant,$basedn";
        }
        $basedn = "ou=people,$basedn";
        $filter = '(seciossdeviceid=*)';
        if ($this->options['userfilter']) {
            $filter = '(&'.$filter.$this->options['userfilter'].')';
        }
        if ($this->options['statusattr']) {
            $filter = '(&'.$filter.'(!('.$this->options['statusattr'].'=deleted)))';
        }
        $attrs = [];
        if ((strcmp('smartphone', $dtype) === 0) || (strcmp('computer', $dtype) === 0)) {
            array_push($attrs, 'seciossdeviceid;x-dev-smartphone');
            array_push($attrs, 'seciossdeviceid;x-dev-computer');
        } else {
            array_push($attrs, 'seciossdeviceid;x-dev-'.$dtype);
        }
        $keys = array_merge(['dn'], $attrs);
        $res = @ldap_list($this->conn, $basedn, $filter, $keys);
        if ($res == false) {
            return PEAR::raiseError(ldap_error($this->conn), ldap_errno($this->conn));
        }
        $users = ldap_get_entries($this->conn, $res);
        if (strcmp('user', $ltype) === 0) {
            for ($i = 0; $i < $users['count'] && isset($users[$i]); $i++) {
                foreach ($attrs as $attr) {
                    if (isset($users[$i][$attr])) {
                        if (strcasecmp($this->id, $users[$i]['dn']) === 0) {
                            break;
                        }
                        $ids = $users[$i][$attr];
                        $exist = false;
                        for ($j = 0; $j < $ids['count'] && isset($ids[$j]); $j++) {
                            if (!preg_match('/^[^#]+#active/', $ids[$j])) {
                                $exist = true;
                                break;
                            }
                        }
                        if ($exist) {
                            $num++;
                            break;
                        }
                    }
                }
            }
        } elseif (strcmp('deviceperuser', $ltype) === 0) {
            for ($i = 0; $i < $users['count'] && isset($users[$i]); $i++) {
                if (strcasecmp($this->id, $users[$i]['dn']) === 0) {
                    foreach ($attrs as $attr) {
                        if (isset($users[$i][$attr])) {
                            $ids = $users[$i][$attr];
                            for ($j = 0; $j < $ids['count'] && isset($ids[$j]); $j++) {
                                if (!preg_match('/^[^#]+#active/', $ids[$j])) {
                                    continue;
                                }
                                if (preg_match('/^'.$newid.'#[^#]+#?([^#]*)/', $ids[$j])) {
                                    continue;
                                }
                                $num++;
                            }
                        }
                    }
                    break;
                }
            }
        } elseif (strcmp('device', $ltype) === 0) {
            for ($i = 0; $i < $users['count'] && isset($users[$i]); $i++) {
                foreach ($attrs as $attr) {
                    if (isset($users[$i][$attr])) {
                        $ids = $users[$i][$attr];
                        $dn = $users[$i]['dn'];
                        for ($j = 0; $j < $ids['count'] && isset($ids[$j]); $j++) {
                            if (!preg_match('/^[^#]+#active/', $ids[$j])) {
                                continue;
                            }
                            if ((strcasecmp($this->id, $dn) === 0) && (preg_match('/^'.$newid.'#[^#]+#?([^#]*)/', $ids[$j]))) {
                                continue;
                            }
                            $num++;
                        }
                    }
                }
            }
        }
        return $num;
    }

    public function _rest_modify($dn, $info, $op = 'replace')
    {
        if ($this->options['api_basedn']) {
            $dn = preg_replace('/,'.$this->options['basedn'].'$/i', ','.$this->options['api_basedn'], $dn);
        }
        $info = [$op => $info];
        $url = $this->options['api_url'].(strpos($this->options['api_url'], '?') === false ? '?' : '&').'action=modify&user='.($this->euser ? $this->euser : $dn).'&dn='.urlencode($dn).(isset($_POST['requestid']) ? '&requestid='.$_POST['requestid'] : '');
        $req = new HTTP_Request($url);
        $req->setMethod(HTTP_REQUEST_METHOD_POST);
        $req->addHeader('Content-type', 'application/json');
        if ($info) {
            $req->setBody(json_encode($info));
        }

        $error = null;
        for ($i = 0; $i < 5; $i++) {
            $res = $req->sendRequest();
            if (PEAR::isError($res)) {
                $error = 'Send request failure('.$res->getMessage().')';
                continue;
            }

            $response = $req->getResponseCode();
            if ($response >= 300) {
                $error = "Bad response code($response)";
            } elseif ($response) {
                $error = null;
                break;
            }
        }
        if ($error) {
            return PEAR::raiseError($error, 52);
        }

        $r = json_decode($req->getResponseBody());
        if ($r->code) {
            return PEAR::raiseError($r->message, $r->code);
        }

        return 0;
    }

    /**
     * optionsにデフォルト値を設定する
     *
     * @access protected
     */
    protected function _setDefaults()
    {
        parent::_setDefaults();

        $this->options['pwdattr'] = 'userpassword';
        $this->options['loginattr'] = 'uid';
        $this->options['encryptpwdattr'] = 'seciossencryptedpassword';
        $this->options['randompwdattr'] = 'seciossrandompassword';
        $this->options['pwdtimeattr'] = '';
        $this->options['pwdhistoryattr'] = '';
        $this->options['pwdmustchangeattr'] = 'seciosspwdmustchange';
        $this->options['pwdexpwarnedattr'] = 'seciosspwdexpwarned';
        $this->options['pwdexpiredattr'] = 'seciosspwdexpired';
        $this->options['pwdlockoutattr'] = 'seciosspwdlockout';
        $this->options['pwdpolicydb'] = '';
        $this->options['oldpwdauth'] = false;
        $this->options['api_url'] = '';
        $this->options['api_basedn'] = '';
    }
}
